/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.util.benutzer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.swing.event.EventListenerList;

import com.bitctrl.util.CollectionUtilities;
import com.bitctrl.util.Timestamp;

import de.bsvrz.dav.daf.accessControl.AccessControlMode;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.ConfigurationTaskException;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.DynamicObjectType;
import de.bsvrz.dav.daf.main.config.DynamicObjectType.DynamicObjectCreatedListener;
import de.bsvrz.dav.daf.main.config.InvalidationListener;
import de.bsvrz.dav.daf.main.config.management.UserAdministration;
import de.bsvrz.sys.funclib.bitctrl.daf.DavTools;
import de.bsvrz.sys.funclib.bitctrl.modell.AnmeldeException;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensendeException;
import de.bsvrz.sys.funclib.bitctrl.modell.DefaultObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlAngemeldeteApplikation;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlRollenRegionenPaar;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.attribute.AtlRollenRegionenPaarNeu;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.konfigurationsdaten.KdBenutzerEigenschaften;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.Applikation;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.Benutzer;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.Berechtigungsklasse;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.BerechtigungsklasseNeu;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.Datenverteiler;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.ZugriffsRegion;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.ZugriffsRegionNeu;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.ZugriffsRolle;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.objekte.ZugriffsRolleNeu;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.onlinedaten.OdAngemeldeteApplikationen;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.parameter.PdBenutzerParameter;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.parameter.PdBerechtigungsklassen;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.parameter.PdBerechtigungsklassen.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.systemmodellglobal.parameter.PdRollenRegionenPaareParameter;
import de.bsvrz.sys.funclib.bitctrl.modell.util.KappichModellUtil;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.dynobj.DynObjektException;
import de.bsvrz.sys.funclib.dynobj.DynamischeObjekte;

/**
 * Verwaltet die Benutzer des Datenverteilers.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public final class Benutzerverwaltung {

	/** Der Standardpräfix für die PID eines neuen Benutzer. */
	public static final String PRAEFIX_PID = "benutzer.";

	/**
	 * Map, in der pro {@link ObjektFactory} eine {@link Benutzerverwaltung}
	 * gespeichert wird, die dann via getInstanz angerufen werden kann.
	 */
	private static final Map<ObjektFactory, Benutzerverwaltung> benutzerverwaltungen = new HashMap<>();

	/** PID der Berechtigungsklasse ohne jede Zugriffsrechte. */
	public static final String PID_KLASSE_KEIN_ZUGRIFF = "berechtigungsklasse.keinZugriff";

	/** PID der Berechtigungsklasse mit allen Zugriffsrechten. */
	public static final String PID_KLASSE_ADMINISTRATOR = "berechtigungsklasse.administrator";

	/** PID der Berechtigungsklasse mit allen Zugriffsrechten. */
	public static final String PID_KLASSE_BEOBACHTER = "berechtigungsklasse.beobachter";

	/** PID der Zugriffsregion für alle Objekte. */
	public static final String PID_REGION_ALLES = "region.alles";

	/** PID der Zugriffsroll mit allen Zugriffsrechten. */
	public static final String PID_ROLLE_ADMINISTRATOR = "rolle.administrator";

	/** PID der Zugriffsrolle die das Parametrieren erlaubt. */
	public static final String PID_ROLLE_PARAMETRIEREN = "rolle.parametrieren";

	/** PID der Zugriffsrolle die nur beobachten darf. */
	public static final String PID_ROLLE_BEOBACHTER = "rolle.beobachter";

	private final ObjektFactory factory;

	/**
	 * Kapselt den Empfang von Login/Logout-Meldungen.
	 */
	private class PrivateListener
			implements DatensatzUpdateListener, DynamicObjectCreatedListener, InvalidationListener {

		/** Das zuletzt empfangene Datum. */
		private OdAngemeldeteApplikationen.Daten letztesDatum = new OdAngemeldeteApplikationen.Daten(null, null);

		@Override
		public void datensatzAktualisiert(final DatensatzUpdateEvent event) {
			if (event.getDatum() instanceof OdAngemeldeteApplikationen.Daten) {
				final OdAngemeldeteApplikationen.Daten aktuellesDatum = (OdAngemeldeteApplikationen.Daten) event
						.getDatum();

				for (final AtlAngemeldeteApplikation app : CollectionUtilities.difference(
						aktuellesDatum.getAngemeldeteApplikation(), letztesDatum.getAngemeldeteApplikation())) {
					fireAngemeldet(app.getBenutzer(), app.getApplikation(), new Timestamp(app.getSeit().getTime()));
				}

				for (final AtlAngemeldeteApplikation app : CollectionUtilities.difference(
						letztesDatum.getAngemeldeteApplikation(), aktuellesDatum.getAngemeldeteApplikation())) {
					fireAbgemeldet(app.getBenutzer(), app.getApplikation(), new Timestamp(app.getSeit().getTime()));
				}

				letztesDatum = aktuellesDatum;
			} else if (event.getDatum() instanceof PdBenutzerParameter.Daten) {
				final Benutzer benutzer = (Benutzer) event.getObjekt();
				final Berechtigungsklasse berechtigungsklasse = ((PdBenutzerParameter.Daten) event.getDatum())
						.getBerechtigungsklasse();
				fireBenutzerChanged(benutzer, berechtigungsklasse);
			}
		}

		@Override
		public void objectCreated(final DynamicObject createdObject) {
			final Benutzer benutzer = (Benutzer) factory.getModellobjekt(createdObject);
			final PdBenutzerParameter param = benutzer.getPdBenutzerParameter();

			param.addUpdateListener(this);
			fireBenutzerAdded(benutzer);
		}

		@Override
		public void invalidObject(final DynamicObject dynamicObject) {
			final Benutzer benutzer = (Benutzer) factory.getModellobjekt(dynamicObject);
			final PdBenutzerParameter param = benutzer.getPdBenutzerParameter();

			param.removeUpdateListener(this);
			fireBenutzerRemoved(benutzer);
		}

	}

	/** Der Logger der Klasse. */
	private final Debug log = Debug.getLogger();

	/** Die angemeldeten Listener der Klasse. */
	private final EventListenerList listeners = new EventListenerList();

	/**
	 * Lokale Kopie der Passworte, um nicht jedesmal eine TCP-Verbindung zum DaV
	 * aufbauen zu müssen.
	 */
	private final Map<Benutzer, String> cachedPasswords = new LinkedHashMap<>();

	/**
	 * Liefert immer wieder dieselbe {@link Benutzerverwaltung} für dieselbe
	 * {@link ObjektFactory}.
	 *
	 * @param factory TODO
	 *
	 * @return Instanz dieser Klasse
	 */
	public static Benutzerverwaltung getInstanz(final ObjektFactory factory) {
		if (benutzerverwaltungen.get(factory) == null) {
			benutzerverwaltungen.put(factory, new Benutzerverwaltung(factory));
		}
		return benutzerverwaltungen.get(factory);
	}

	/**
	 * Initialisierung.
	 *
	 * @param factory die zum erzeugen von Systemobjekten verwendete ObjektFactory
	 */
	public Benutzerverwaltung(final ObjektFactory factory) {
		this.factory = factory;
		final PrivateListener privateListener = new PrivateListener();

		final Datenverteiler dav = (Datenverteiler) factory.getModellobjekt(factory.getDav().getLocalDav());
		final OdAngemeldeteApplikationen datensatz = dav.getOdAngemeldeteApplikationen();

		// Listener auf An- und Abmelden
		datensatz.addUpdateListener(OdAngemeldeteApplikationen.Aspekte.Standard, privateListener);

		// Listener auf Änderungen der Menge der Benutzer
		final DynamicObjectType typ = (DynamicObjectType) factory.getModellobjekt(Benutzer.PID).getSystemObject();
		typ.addObjectCreationListener(privateListener);
		typ.addInvalidationListener(privateListener);

		// Listener auf Änderungen der Berechtigungsklasse der Nutzer
		for (final SystemObjekt so : factory.bestimmeModellobjekte(Benutzer.PID)) {
			if (so instanceof Benutzer) {
				final Benutzer benutzer = (Benutzer) so;
				benutzer.getPdBenutzerParameter().addUpdateListener(privateListener);
			}
		}

		log.info("Schnittstelle zur Nutzerverwaltung initialisiert.");
	}

	/**
	 * Registriert einen Listener an.
	 *
	 * @param l ein Listener.
	 */
	public void addLoginListener(final LoginListener l) {
		listeners.add(LoginListener.class, l);
	}

	/**
	 * Meldet einen Listener wieder ab.
	 *
	 * @param l ein Listener.
	 */
	public void removeLoginListener(final LoginListener l) {
		listeners.remove(LoginListener.class, l);
	}

	/**
	 * Benachrichtigt die angmeldeten Listener über die Anmeldung eines Benutzers.
	 *
	 * @param benutzer    der betroffene Benutzer.
	 * @param applikation die betroffene Applikation.
	 * @param anmeldezeit der Anmeldezeitpunkt.
	 */
	protected synchronized void fireAngemeldet(final Benutzer benutzer, final Applikation applikation,
			final Timestamp anmeldezeit) {
		final LoginEvent e = new LoginEvent(this, benutzer, applikation, anmeldezeit);

		for (final LoginListener l : listeners.getListeners(LoginListener.class)) {
			l.angemeldet(e);
		}
	}

	/**
	 * Benachrichtigt die angmeldeten Listener darüber, dass sich ein Benutzer
	 * abgemeldet hat.
	 *
	 * @param benutzer    der betroffene Benutzer.
	 * @param applikation die betroffene Applikation.
	 * @param anmeldezeit der Anmeldezeitpunkt.
	 */
	protected synchronized void fireAbgemeldet(final Benutzer benutzer, final Applikation applikation,
			final Timestamp anmeldezeit) {
		final LoginEvent e = new LoginEvent(this, benutzer, applikation, anmeldezeit);

		for (final LoginListener l : listeners.getListeners(LoginListener.class)) {
			l.abgemeldet(e);
		}
	}

	/**
	 * Registriert einen Listener an.
	 *
	 * @param l ein Listener.
	 */
	public void addBenutzerListener(final BenutzerListener l) {
		listeners.add(BenutzerListener.class, l);
	}

	/**
	 * Meldet einen Listener wieder ab.
	 *
	 * @param l ein Listener.
	 */
	public void removeBenutzerListener(final BenutzerListener l) {
		listeners.remove(BenutzerListener.class, l);
	}

	/**
	 * Benachrichtigt die angmeldeten Listener darüber, dass ein neuer Benutzer
	 * angelegt wurde.
	 *
	 * @param benutzer der betroffene Benutzer.
	 */
	protected synchronized void fireBenutzerAdded(final Benutzer benutzer) {
		final BenutzerEvent e = new BenutzerEvent(this, benutzer, null);

		for (final BenutzerListener l : listeners.getListeners(BenutzerListener.class)) {
			l.benutzerAdded(e);
		}
	}

	/**
	 * Benachrichtigt die angmeldeten Listener darüber, dass ein Benutzer gelöscht
	 * wurde.
	 *
	 * @param benutzer der betroffene Benutzer.
	 */
	protected synchronized void fireBenutzerRemoved(final Benutzer benutzer) {
		final BenutzerEvent e = new BenutzerEvent(this, benutzer, null);

		for (final BenutzerListener l : listeners.getListeners(BenutzerListener.class)) {
			l.benutzerRemoved(e);
		}
	}

	/**
	 * Benachrichtigt die angmeldeten Listener darüber, dass die Berechtigungsklasse
	 * eines Benutzer geändert wurde.
	 *
	 * @param benutzer            der betroffene Benutzer.
	 * @param berechtigungsklasse die Berechtigungsklasse des Benutzers.
	 */
	protected synchronized void fireBenutzerChanged(final Benutzer benutzer,
			final Berechtigungsklasse berechtigungsklasse) {
		final BenutzerEvent e = new BenutzerEvent(this, benutzer, berechtigungsklasse);

		for (final BenutzerListener l : listeners.getListeners(BenutzerListener.class)) {
			l.benutzerChanged(e);
		}
	}

	/**
	 * Gibt die lokale Klientapplikation zurück.
	 *
	 * @return die lokale Applikation.
	 */
	public Applikation getApplikation() {
		return KappichModellUtil.getApplikation(factory);
	}

	/**
	 * Gibt die Liste aller aktuell gültigen Anmeldungen eines Benutzers zurück.
	 *
	 * @param benutzer ein Benutzer.
	 * @return die Liste der Anmeldungen des Benutzers.
	 */
	public List<AtlAngemeldeteApplikation> getAnmeldungen(final Benutzer benutzer) {
		final Datenverteiler dav = KappichModellUtil.getDatenverteiler(factory);
		final OdAngemeldeteApplikationen.Daten datum = dav.getOdAngemeldeteApplikationen()
				.getDatum(OdAngemeldeteApplikationen.Aspekte.Standard);
		final List<AtlAngemeldeteApplikation> apps = new ArrayList<>();

		for (final AtlAngemeldeteApplikation app : datum.getAngemeldeteApplikation()) {
			if (benutzer.equals(app.getBenutzer())) {
				apps.add(app);
			}
		}

		return apps;
	}

	/**
	 * Gibt die Anmeldung zu einer Applikation zurück. Der Rückgabewert kann
	 * {@code null} sein, wenn es keine Anmeldung mehr zu dieser Applikation gibt,
	 * dass heißt sie wurde beendet.
	 *
	 * @param applikation eine Applikation.
	 * @return die aktuelle Anmeldung der Applikation.
	 */
	public AtlAngemeldeteApplikation getAnmeldungen(final Applikation applikation) {
		final Datenverteiler dav = KappichModellUtil.getDatenverteiler(factory);
		final OdAngemeldeteApplikationen.Daten datum = dav.getOdAngemeldeteApplikationen()
				.getDatum(OdAngemeldeteApplikationen.Aspekte.Standard);

		for (final AtlAngemeldeteApplikation app : datum.getAngemeldeteApplikation()) {
			if (applikation.equals(app.getApplikation())) {
				return app;
			}
		}

		return null;
	}

	/**
	 * Gibt den lokal angemeldeten Benutzer zurück.
	 *
	 * @return der angemeldete Benutzer.
	 */
	public Benutzer getAngemeldetenBenutzer() {
		return KappichModellUtil.getBenutzer(factory);
	}

	/**
	 * Gibt eine Liste aller Benutzer im System zurück.
	 *
	 * @return die Benutzerliste.
	 */
	public List<Benutzer> getBenutzer() {
		final List<Benutzer> benutzer = new ArrayList<>();

		for (final SystemObjekt so : factory.bestimmeModellobjekte(Benutzer.PID)) {
			benutzer.add((Benutzer) so);
		}

		return benutzer;
	}

	/**
	 * Gibt eine Liste aller Benutzer zurück, die einer bestimmten
	 * Berechtigungsklasse angehören.
	 *
	 * @param klasse eine Berechtigungsklase.
	 * @return die Benutzerliste.
	 */
	public List<Benutzer> getBenutzer(final Berechtigungsklasse klasse) {
		final List<Benutzer> benutzerListe = new ArrayList<>();

		for (final SystemObjekt so : factory.bestimmeModellobjekte(Benutzer.PID)) {
			final Benutzer benutzer = (Benutzer) so;
			if (isBerechtigungsklasse(benutzer, klasse)) {
				benutzerListe.add(benutzer);
			}
		}

		return benutzerListe;
	}

	/**
	 * Gibt eine Liste aller Berechtigungsklassen im System zurück.
	 *
	 * @return die Liste der Berechtigungsklassen.
	 * @deprecated Use {@link Benutzerverwaltung#getBerechtigungsklasseNeu()}
	 */
	public List<Berechtigungsklasse> getBerechtigungsklasse() {
		final List<Berechtigungsklasse> klassen = new ArrayList<>();

		for (final SystemObjekt so : factory.bestimmeModellobjekte(Berechtigungsklasse.PID)) {
			klassen.add((Berechtigungsklasse) so);
		}

		return klassen;
	}

	@SuppressWarnings("unchecked")
	public List<BerechtigungsklasseNeu> getBerechtigungsklasseNeu() {
		return (List<BerechtigungsklasseNeu>) factory.bestimmeModellobjekte(BerechtigungsklasseNeu.PID).stream()
				.collect(Collectors.toList());
	}

	/**
	 * Gibt eine Zusammenstellung aller Benutzer im System und deren Benutzerklasse
	 * zurück.
	 *
	 * @return die aktuelle Rechteverteilung.
	 * @deprecated use {@link Benutzerverwaltung#getBenutzerUndKlassenNeu()}
	 */
	public Map<Benutzer, Berechtigungsklasse> getBenutzerUndKlassen() {
		final Map<Benutzer, Berechtigungsklasse> rechte = new HashMap<>();

		for (final Benutzer benutzer : getBenutzer()) {
			rechte.put(benutzer, getBerechtigungsklasse(benutzer));
		}

		return rechte;
	}

	public Map<Benutzer, Collection<BerechtigungsklasseNeu>> getBenutzerUndKlassenNeu() {
		final Map<Benutzer, Collection<BerechtigungsklasseNeu>> rechte = new HashMap<>();

		for (final Benutzer benutzer : getBenutzer()) {
			rechte.put(benutzer, getBerechtigungsklasseNeu(benutzer));
		}

		return rechte;
	}

	/**
	 * Pr&uuml;ft, ob ein Bennutzer Administratorrechte am DAV besitzt.
	 * Administratoren d&uuml;rfen u. a. Benutzer l&ouml;schen, Passworte anderer
	 * Benuzter &auml;ndern und Autorisierungspassworte f&uuml;r andere Benuzter
	 * anlegen.<br>
	 * Da sich der Zustand des Admin-Flags derzeit nicht direkt &uuml;ber das
	 * DAV-API ermitteln l&auml;&szlig;t, hilft folgender Trick: Man versucht den
	 * Benuzter mit sich selbst zum Admin zu machen, klappt dies, so ist er bereits
	 * Admin gewesen, ansonsten ist er kein Admin.
	 *
	 * @param loginname ein Nutzername
	 * @param passwort  das Passwort des Benutzers
	 * @return {@code true}, wenn der Bennutzer Administratorrechte besitzt.
	 */
	public boolean isDAVAdmin(final String loginname, final String passwort) {
		try {
			final DataModel modell = factory.getDav().getDataModel();
			final UserAdministration userAdmin = modell.getUserAdministration();
			userAdmin.changeUserRights(loginname, passwort, loginname, true);
			return true;
		} catch (final ConfigurationTaskException ex) {
			return false;
		}
	}

	/**
	 * Prüft, ob die neuen {@link ZugriffsRegionNeu}, {@link ZugriffsRolleNeu} &
	 * {@link BerechtigungsklasseNeu} verwendet werden.
	 * 
	 * @return
	 */
	private boolean neueRechtepruefung() {
		boolean result = false;
		try {
			AccessControlMode accessControlMode = factory.getDav().getAccessControlMode();
			result = AccessControlMode.NewDataModel.equals(accessControlMode);
			if (!result) {
				result |= factory.bestimmeModellobjekte(BerechtigungsklasseNeu.PID).size() > 0;
				result |= factory.bestimmeModellobjekte(ZugriffsRegionNeu.PID).size() > 0;
				result |= factory.bestimmeModellobjekte(ZugriffsRolleNeu.PID).size() > 0;
			}
		} catch (Throwable t) {
			// API kleiner Dav 3.13
			return false;
		}
		return result;
	}

	/**
	 * Prüft ob ein bestimmter Benutzer zu einer bestimmten Berechtigungsklasse
	 * gehört.
	 *
	 * @param benutzer ein Benutzer.
	 * @param klasse   eine Berechtuigungsklasse.
	 * @return {@code true}, wenn der Benutzer zu der Berechtigungsklasse gehört.
	 * @deprecated use
	 *             {@link Benutzerverwaltung#isBerechtigungsklasse(Benutzer, BerechtigungsklasseNeu)}
	 */
	public boolean isBerechtigungsklasse(final Benutzer benutzer, final Berechtigungsklasse klasse) {
		return klasse.equals(getBerechtigungsklasse(benutzer));
	}

	public boolean isBerechtigungsklasse(final Benutzer benutzer, final BerechtigungsklasseNeu klasse) {
		return getBerechtigungsklasseNeu(benutzer).contains(klasse);
	}

	/**
	 * Bestimmt die aktuelle Berechtigungsklasse des Benutzers.
	 *
	 * @param benutzer ein Benutzer.
	 * @return die Berechtigungsklasse des Benutzers.
	 * 
	 * @deprecated use
	 *             {@link Benutzerverwaltung#getBerechtigungsklasseNeu(Benutzer)}
	 */
	public Berechtigungsklasse getBerechtigungsklasse(final Benutzer benutzer) {
		final PdBenutzerParameter parameter = benutzer.getPdBenutzerParameter();
		final PdBenutzerParameter.Daten datum = parameter.getDatum();

		if (datum.dContainsDaten()) {
			return datum.getBerechtigungsklasse();
		}

		return null;
	}

	public Collection<BerechtigungsklasseNeu> getBerechtigungsklasseNeu(final Benutzer benutzer) {
		Daten datum = benutzer.getPdBerechtigungsklassen().getDatum();
		if (datum != null && datum.dContainsDaten()) {
			return datum.getBerechtigungsklassen();
		}
		return Collections.emptyList();
	}

	/**
	 * Prüft ob ein bestimmter Benutzer eine bestimmte Rolle in einer Region hat.
	 *
	 * @param benutzer ein Benutzer.
	 * @param rolle    eine Zugriffsrolle.
	 * @param region   eine Zugriffsregion. Wenn {@code null}, wird die Region
	 *                 ignoriert und nur die Rolle geprüpft.
	 * @return {@code true}, wenn der Benutzer in der Region die angegebene Rolle
	 *         hat.
	 * @deprecated use
	 *             {@link Benutzerverwaltung#isRolleUndRegion(Benutzer, String, String)}
	 */
	public boolean isRolleUndRegion(final Benutzer benutzer, final ZugriffsRolle rolle, final ZugriffsRegion region) {
		if (benutzer == null) {
			throw new IllegalArgumentException("Benutzer darf nicht null sein.");
		}
		if (rolle == null) {
			throw new IllegalArgumentException("Zugriffsrolle darf nicht null sein.");
		}

		final PdBenutzerParameter paramBenutzer = benutzer.getPdBenutzerParameter();

		final PdBenutzerParameter.Daten datumBenutzer = paramBenutzer.getDatum();
		final Berechtigungsklasse klasse = datumBenutzer.getBerechtigungsklasse();
		if (klasse == null) {
			log.info("Dem Benutzer " + benutzer + " ist keine Berechtigungsklasse zugeordnet.");
			return false;
		}

		final PdRollenRegionenPaareParameter paramPaare = klasse.getPdRollenRegionenPaareParameter();
		final PdRollenRegionenPaareParameter.Daten datumPaare = paramPaare.getDatum();

		for (final AtlRollenRegionenPaar paar : datumPaare.getRollenRegionenPaare()) {
			if (region != null) {
				if (region.equals(paar.getRegion()) && rolle.equals(paar.getRolle())) {
					return true;
				}
			} else {
				if (rolle.equals(paar.getRolle())) {
					return true;
				}
			}
		}
		return false;
	}

	public boolean isRolleUndRegion(final Benutzer benutzer, final String pidRolle, final String pidRegion) {
		if (benutzer == null) {
			throw new IllegalArgumentException("Benutzer darf nicht null sein.");
		}
		if (pidRolle == null || pidRolle.isEmpty()) {
			throw new IllegalArgumentException("Zugriffsrolle darf nicht null sein.");
		}

		if (neueRechtepruefung()) {
			Collection<BerechtigungsklasseNeu> berechtigungsklassen = getBerechtigungsklasseNeu(benutzer);
			List<AtlRollenRegionenPaarNeu> rrPaare = berechtigungsklassen.stream()
					.map(b -> b.getPdRollenRegionenPaare().getDatum()).filter(d -> d != null && d.dContainsDaten())
					.flatMap(d -> d.getRollenRegionenPaare().stream()).collect(Collectors.toList());
			ZugriffsRolleNeu rolle = getRolleNeu(pidRolle);
			if (pidRegion != null && !pidRegion.isEmpty()) {
				return rrPaare.stream().filter(r -> Objects.equals(getRegionNeu(pidRegion), r.getRegion())
						&& Objects.equals(rolle, r.getRolle())).findFirst().isPresent();
			} else {
				return rrPaare.stream().filter(r -> Objects.equals(rolle, r.getRolle())).findFirst().isPresent();
			}

		} else {
			ZugriffsRolle rolle = getRolle(pidRolle);
			ZugriffsRegion region = null;
			if (pidRegion != null && !pidRegion.isEmpty()) {
				region = getRegion(pidRegion);
			}
			return isRolleUndRegion(benutzer, rolle, region);
		}
	}

	/**
	 * Sucht eine bestimmte Berechtigungsklasse.
	 *
	 * @param pid die PID einer Berechtigungsklasse.
	 * @return die Berechtigungsklasse oder {@code null}, wenn zu der PID keine
	 *         existiert.
	 * 
	 * @deprecated Use {@link Benutzerverwaltung#getBerechtigungsklasseNeu(String)}.
	 */
	public Berechtigungsklasse getBerechtigungsklasse(final String pid) {
		return (Berechtigungsklasse) factory.getModellobjekt(pid);
	}

	public BerechtigungsklasseNeu getBerechtigungsklasseNeu(final String pid) {
		return (BerechtigungsklasseNeu) factory.getModellobjekt(pid);
	}

	/**
	 * Sucht eine bestimmte Zugriffsrolle.
	 *
	 * @param pid die PID einer Zugriffsrolle.
	 * @deprecated Use {@link Benutzerverwaltung#getRolleNeu(String)}.
	 * @return die Zugriffsrolle oder {@code null}, wenn zu der PID keine existiert.
	 */
	public ZugriffsRolle getRolle(final String pid) {
		return (ZugriffsRolle) factory.getModellobjekt(pid);
	}

	public ZugriffsRolleNeu getRolleNeu(final String pid) {
		return (ZugriffsRolleNeu) factory.getModellobjekt(pid);
	}

	/**
	 * Gibt eine Liste aller Zugriffsrollen im System zurück.
	 *
	 * @return die Liste der Zugriffsrollen.
	 * @deprecated Use {@link Benutzerverwaltung#getRolleNeu()}.
	 */
	public List<ZugriffsRolle> getRolle() {
		final List<ZugriffsRolle> rollen = new ArrayList<>();

		for (final SystemObjekt so : factory.bestimmeModellobjekte(ZugriffsRolle.PID)) {
			rollen.add((ZugriffsRolle) so);
		}

		return rollen;
	}

	@SuppressWarnings("unchecked")
	public List<ZugriffsRolleNeu> getRolleNeu() {
		return (List<ZugriffsRolleNeu>) factory.bestimmeModellobjekte(ZugriffsRolleNeu.PID).stream()
				.collect(Collectors.toList());
	}

	/**
	 * Sucht eine bestimmte Zugriffsregion.
	 *
	 * @param pid die PID einer Zugriffsregion.
	 * @return die Zugriffsregion oder {@code null}, wenn zu der PID keine
	 *         existiert.
	 * 
	 * @deprecated Use {@link Benutzerverwaltung#getRegionNeu(String)}.
	 */
	public ZugriffsRegion getRegion(final String pid) {
		return (ZugriffsRegion) factory.getModellobjekt(pid);
	}

	public ZugriffsRegionNeu getRegionNeu(final String pid) {
		return (ZugriffsRegionNeu) factory.getModellobjekt(pid);
	}

	/**
	 * Gibt eine Liste aller Zugriffsregionen im System zurück.
	 *
	 * @return die Liste der Zugriffsregionen.
	 * @deprecated Use {@link Benutzerverwaltung#getRegionNeu()}.
	 */
	public List<ZugriffsRegion> getRegion() {
		final List<ZugriffsRegion> regionen = new ArrayList<>();

		for (final SystemObjekt so : factory.bestimmeModellobjekte(ZugriffsRegion.PID)) {
			regionen.add((ZugriffsRegion) so);
		}

		return regionen;
	}

	@SuppressWarnings("unchecked")
	public List<ZugriffsRegionNeu> getRegionNeu() {
		return (List<ZugriffsRegionNeu>) factory.bestimmeModellobjekte(ZugriffsRegionNeu.PID).stream()
				.collect(Collectors.toList());

	}

	/**
	 * Prüft ob ein bestimmter Benutzer existiert und gibt ihn zurück.
	 *
	 * @param loginname der eindeutige Benutzername (Loginname).
	 * @return der Benutzer oder {@code null}, wenn kein Benutzer mit dem
	 *         angegebenen Namen existiert.
	 */
	public Benutzer getBenutzer(final String loginname) {
		if ((loginname == null) || (loginname.length() == 0)) {
			throw new IllegalArgumentException("Der Benutzername darf weder null noch ein Leerstring sein.");
		}
		final String pid = DavTools.generierePID(loginname, PRAEFIX_PID);
		final Benutzer benutzer = (Benutzer) factory.getModellobjekt(pid);

		if (benutzer != null) {
			return benutzer;
		}

		// Suche per PID erfolglos, weiter nach Namen suchen
		for (final SystemObjekt so : factory.bestimmeModellobjekte(Benutzer.PID)) {
			final Benutzer b = (Benutzer) so;

			if (b.getName().equals(loginname)) {
				break;
			}
		}

		log.fine("Die PID des Benutzers " + benutzer + " beginnt nicht mit dem Standardprefix " + PRAEFIX_PID);
		return benutzer;
	}

	/**
	 * Sucht alle Benutzer auf die bestimmte Kriterien zutreffen. Die Suchkriterien
	 * dürfen auch {@code null} sein, dies wird als Wildcard "alle" gedeutet. Die
	 * einzelnen Kriterien werden "oder"-verknüpft.
	 *
	 * @param nachname       der Nachname der gesuchten Benutzer.
	 * @param vorname        der Vorname der gesuchten Benutzer.
	 * @param zweiterVorname der zweite Vorname der gesuchten Benutzer.
	 * @param organisation   die Organisation der gesuchten Benutzer.
	 * @param email          die E-Mail-Adresse der gesuchten Benutzer.
	 * @return die Liste der gefundenen Benutzer, niemals {@code null}.
	 */
	public List<Benutzer> sucheBenutzer(final String nachname, final String vorname, final String zweiterVorname,
			final String organisation, final String email) {
		final List<Benutzer> benutzer = new ArrayList<>();

		for (final SystemObjekt so : factory.bestimmeModellobjekte(Benutzer.PID)) {
			final Benutzer b = (Benutzer) so;
			final KdBenutzerEigenschaften.Daten benutzerDaten = b.getKdBenutzerEigenschaften().getDatum();
			boolean ok = false;

			if ((nachname == null) || nachname.equals(benutzerDaten.getNachname())) {
				ok = true;
			} else if ((vorname == null) || vorname.equals(benutzerDaten.getVorname())) {
				ok = true;
			} else if ((zweiterVorname == null) || zweiterVorname.equals(benutzerDaten.getZweiterVorname())) {
				ok = true;
			} else if ((organisation == null) || organisation.equals(benutzerDaten.getOrganisation())) {
				ok = true;
			} else if ((email == null) || email.equals(benutzerDaten.getEmailAdresse())) {
				ok = true;
			}

			if (ok) {
				benutzer.add(b);
			}
		}

		return benutzer;
	}

	/**
	 * Legt einen neuen Benutzer an. Das Anlegen erfolgt unabh&auml;ngig von
	 * Berechtigungsklassen nur unter Pr&uuml;fung der entsprechenden Rechte am DAV.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt.
	 * @param benutzerInfo   die Eigenschaften des neuen Benutzers.
	 * @param adminRechte    <code>true</code>, wenn der neue Benutzer die Rechte
	 *                       eines Administrators besitzen soll; <code>false</code>,
	 *                       wenn der Benutzer keine speziellen Rechte besitzen
	 *                       soll.
	 * @return der neue Benutzer
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Anlegen des Benutzers ein Fehler
	 *                                 eintrat.
	 */
	public Benutzer anlegenBenutzer(final String adminLoginname, final String adminPasswort,
			final BenutzerInfo benutzerInfo, final boolean adminRechte)
			throws KeineRechteException, BenutzerChangeException {
		if (!isDAVAdmin(adminLoginname, adminPasswort)) {
			throw new KeineRechteException(
					"Sie verfügen nicht über ausreichend Rechte zum Anlegen eines neuen Benutzer.");
		}

		final String pid;
		Benutzer benutzer;

		pid = DavTools.generierePID(benutzerInfo.getLoginname(), PRAEFIX_PID);
		try {
			benutzer = benutzerObjektAnlegen(pid, benutzerInfo.getLoginname(), benutzerInfo.getVorname(),
					benutzerInfo.getZweiterVorname(), benutzerInfo.getNachname(), benutzerInfo.getOrganisation(),
					benutzerInfo.getEmailAdresse());

			final DataModel modell = factory.getDav().getDataModel();
			final UserAdministration userAdmin = modell.getUserAdministration();

			final DynamicObjectType type = (DynamicObjectType) factory.getDav().getDataModel().getObject(Benutzer.PID);

			final ConfigurationArea konfigurationsBereich = DynamischeObjekte.getInstanz(factory.getDav())
					.getKonfigurationsBereich(type);

			userAdmin.createNewUser(adminLoginname, adminPasswort, benutzerInfo.getLoginname(), pid,
					benutzerInfo.getPasswort(), adminRechte, konfigurationsBereich.getPid());
		} catch (final ConfigurationTaskException | DynObjektException ex) {
			throw new BenutzerChangeException(
					"Der Benutzer '" + benutzerInfo.getLoginname() + "' konnte nicht angelegt werden.", ex);
		}

		return benutzer;
	}

	private Benutzer benutzerObjektAnlegen(final String pid, final String name, final String vorname,
			final String zweiterVorname, final String nachname, final String organisation, final String emailAdresse)
			throws DynObjektException {

		// Konfigurationsdatensatz erzeugen
		final KdBenutzerEigenschaften datensatz = new KdBenutzerEigenschaften(null, factory);
		final KdBenutzerEigenschaften.Daten datum = new KdBenutzerEigenschaften.Daten(datensatz,
				KdBenutzerEigenschaften.Aspekte.Eigenschaften);
		datum.setVorname(vorname != null ? vorname : "");
		datum.setZweiterVorname(zweiterVorname != null ? zweiterVorname : "");
		datum.setNachname(nachname != null ? nachname : "");
		datum.setOrganisation(organisation != null ? organisation : "");
		datum.setEmailAdresse(emailAdresse != null ? emailAdresse : "");

		return factory.createDynamischesObjekt(Benutzer.class, name, pid, datum);
	}

	/**
	 * Deaktiviert und löscht einen Benutzer. Das entsprechende Systemobjekt wird
	 * invalidiert.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt. <em>Wird derzeit ignoriert, da es nicht
	 *                       benötigt wird!</em>
	 * @param benutzer       der zu löschende Benutzers.
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Löschen des Benutzers ein Fehler
	 *                                 eintrat.
	 */
	public void entfernenBenutzer(final String adminLoginname, final String adminPasswort, final Benutzer benutzer)
			throws KeineRechteException, BenutzerChangeException {
		if (!isDAVAdmin(adminLoginname, adminPasswort)) {
			throw new KeineRechteException("Sie verfügen nicht über ausreichend Rechte zum Löschen eines Benutzer.");
		}

		try {
			benutzer.getSystemObject().invalidate();
			final UserAdministration userAdministration = factory.getDav().getDataModel().getUserAdministration();
			userAdministration.deleteUser(adminLoginname, adminPasswort, benutzer.getName());
		} catch (final ConfigurationChangeException ex) {
			throw new BenutzerChangeException(
					"Das Benutzerobjekt " + benutzer.getName() + " konnte nicht gelöscht werden.", ex);
		} catch (final ConfigurationTaskException ex) {
			throw new BenutzerChangeException(
					"Die Benutzerdaten " + benutzer.getName() + " konnten nicht gelöscht werden.", ex);
		}
	}

	/**
	 * Ändert die Berechtigungsklasse eines Benutzer.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt.
	 * @param benutzer       der zu deaktivierende Benutzers.
	 * @param klasse         die zu setzende Berechtigungsklasse.
	 * @param urlasser       der Nutzername des Urlassers.
	 * @param ursache        Ursache zur Urlasserinformation
	 * @param veranlasser    Veranlasser zur Urlasserinformation
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Ändern der Benutzerrechte ein
	 *                                 Fehler eintrat.
	 * @deprecated use
	 *             {@link Benutzerverwaltung#setBerechtigungsklasse(String, String, Benutzer, BerechtigungsklasseNeu, String, String, String)}
	 */
	public void setBerechtigungsklasse(final String adminLoginname, final String adminPasswort, final Benutzer benutzer,
			final Berechtigungsklasse klasse, final String urlasser, final String ursache, final String veranlasser)
			throws KeineRechteException, BenutzerChangeException {
		if (!isDAVAdmin(adminLoginname, adminPasswort)) {
			throw new KeineRechteException(
					"Sie verfügen nicht über ausreichend Rechte zum Ändern der Berechtigungsklasse eines Benutzers.");
		}

		final PdBenutzerParameter parameter = benutzer.getPdBenutzerParameter();
		final PdBenutzerParameter.Daten datum = parameter.createDatum();
		datum.setBerechtigungsklasse(klasse);

		datum.getUrlasser().setBenutzerReferenz(getBenutzer(adminLoginname));
		datum.getUrlasser().setUrsache(ursache);
		datum.getUrlasser().setVeranlasser(veranlasser);

		try {
			parameter.anmeldenSender();
			parameter.sendeDatum(datum);
		} catch (final AnmeldeException ex) {
			throw new BenutzerChangeException("Fehler beim Anmelden auf Parameter für Benutzer " + benutzer + ".", ex);
		} catch (final DatensendeException ex) {
			throw new BenutzerChangeException("Fehler beim Senden des Parameters für Benutzer " + benutzer + ".", ex);
		} finally {
			parameter.abmeldenSender();
		}
	}

	public void setBerechtigungsklasse(final String adminLoginname, final String adminPasswort, final Benutzer benutzer,
			final BerechtigungsklasseNeu klasse, final String urlasser, final String ursache, final String veranlasser)
			throws KeineRechteException, BenutzerChangeException {
		if (!isDAVAdmin(adminLoginname, adminPasswort)) {
			throw new KeineRechteException(
					"Sie verfügen nicht über ausreichend Rechte zum Ändern der Berechtigungsklasse eines Benutzers.");
		}

		final PdBerechtigungsklassen parameter = benutzer.getPdBerechtigungsklassen();
		final PdBerechtigungsklassen.Daten datum = parameter.getDatum();
		if (!datum.getBerechtigungsklassen().contains(klasse)) {

			datum.getBerechtigungsklassen().add(klasse);
			datum.getUrlasser().setBenutzerReferenz(getBenutzer(adminLoginname));
			datum.getUrlasser().setUrsache(ursache);
			datum.getUrlasser().setVeranlasser(veranlasser);

			try {
				parameter.anmeldenSender();
				parameter.sendeDatum(datum);
			} catch (final AnmeldeException ex) {
				throw new BenutzerChangeException("Fehler beim Anmelden auf Parameter für Benutzer " + benutzer + ".",
						ex);
			} catch (final DatensendeException ex) {
				throw new BenutzerChangeException("Fehler beim Senden des Parameters für Benutzer " + benutzer + ".",
						ex);
			} finally {
				parameter.abmeldenSender();
			}
		}
	}

	/**
	 * Ändert die Berechtigungsklasse eines Benutzer.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt.
	 * @param benutzer       der zu deaktivierende Benutzers.
	 * @param passworte      Liste der Einmalpassworte
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Ändern der Benutzerrechte ein
	 *                                 Fehler eintrat.
	 */
	public void setEinmalPassworte(final String adminLoginname, final String adminPasswort, final Benutzer benutzer,
			final List<String> passworte) throws KeineRechteException, BenutzerChangeException {
		if (!isDAVAdmin(adminLoginname, adminPasswort)) {
			throw new KeineRechteException(
					"Sie verfügen nicht über ausreichend Rechte zum Ändern der Berechtigungsklasse eines Benutzers.");
		}

		try {
			final DataModel modell = factory.getDav().getDataModel();
			final UserAdministration userAdmin = modell.getUserAdministration();

			for (final String pw : passworte) {
				userAdmin.createSingleServingPassword(adminLoginname, adminPasswort, benutzer.getName(), pw);
			}
		} catch (final ConfigurationTaskException ex) {
			throw new BenutzerChangeException(
					"Die Einmalpassworte für " + benutzer.getName() + " konnten nicht angelegt werden.", ex);
		}
	}

	/**
	 * Flag ob ein bestimmter Benutzer im Moment an irgendeiner Applikation
	 * angemeldet ist.
	 *
	 * @param benutzer ein Benutzer.
	 * @return {@code true}, wenn der Benutzer online ist.
	 */
	public boolean isOnline(final Benutzer benutzer) {
		return !getAnmeldungen(benutzer).isEmpty();
	}

	/**
	 * Prüft ein neues Passwort auf seine Sicherheit. Die Funktion wertet die für
	 * die Vergabe von Passworten definierten Parameter aus und prüft, ob das
	 * Passwort diesen Kriterien entspricht.
	 *
	 * @param passwort     das Passwort.
	 * @param benutzer     der Nutzer, für den das Passwort verwendet werden soll.
	 * @param passwortInfo die Sicherheitskriterien.
	 * @return {@code null}, wenn das Passwort sicher ist, sonst eine
	 *         Fehlerbeschreibung.
	 *
	 *         TODO Prüfen ob alle Sicherheitsaspekte geprüft werden.
	 */
	public String checkPasswort(final String passwort, final Benutzer benutzer, final PasswortInfo passwortInfo) {
		if (passwort == null) {
			throw new IllegalArgumentException("Passwort darf nicht null sein.");
		}

		final long minLaenge = passwortInfo.getMinLaenge();
		if (minLaenge > 0) {
			if (passwort.length() < minLaenge) {
				return "Das Passwort muss mindestens " + minLaenge + " Zeichen lang sein.";
			}
		}

		final String lowerPasswort = passwort.toLowerCase(Locale.getDefault());
		if (passwortInfo.isGemischteZeichen()) {
			final int laenge = lowerPasswort.length();
			int anzahl = 0;
			for (int i = 0; i < laenge; i++) {
				if (lowerPasswort.substring(i, i + 1).matches("[a-z]")) {
					anzahl++;
				}
			}

			if ((anzahl == 0) || (anzahl == laenge)) {
				return "Das Passwort muss außer Buchstaben auch Zahlen oder Sonderzeichen enthalten.";
			}
		}

		if ((benutzer != null) && passwortInfo.isVergleicheBenutzerdaten()) {
			final String loginname = benutzer.getName().toLowerCase(Locale.getDefault());
			final KdBenutzerEigenschaften.Daten benutzerDaten = benutzer.getKdBenutzerEigenschaften().getDatum();

			final String nachname = benutzerDaten.getNachname().toLowerCase(Locale.getDefault());
			final String vorname = benutzerDaten.getVorname().toLowerCase(Locale.getDefault());

			if (loginname.contains(lowerPasswort.subSequence(0, lowerPasswort.length() - 1))) {
				return "Das Passwort darf nicht im Loginnamen enthalten sein.";
			}
			if (lowerPasswort.contains(loginname.subSequence(0, loginname.length() - 1))) {
				return "Das Passwort darf nicht den Loginnamen enthalten.";
			}

			if (nachname.contains(lowerPasswort.subSequence(0, lowerPasswort.length() - 1))) {
				return "Das Passwort darf nicht im Nachnamen enthalten sein.";
			}
			if (lowerPasswort.contains(nachname.subSequence(0, nachname.length() - 1))) {
				return "Das Passwort darf nicht den Nachnamen enthalten.";
			}

			if (lowerPasswort.contains(vorname.subSequence(0, vorname.length() - 1))) {
				return "Das Passwort darf nicht im Vornamen enthalten sein.";
			}
			if (vorname.contains(lowerPasswort.subSequence(0, lowerPasswort.length() - 1))) {
				return "Das Passwort darf nicht den Vornamen enthalten.";
			}
		}

		// Alle Tests bestanden
		return null;
	}

	/**
	 * Liefert das gecachte Passwort eines Benutzers.
	 *
	 * @param b der Benutzer
	 * @return das gecachte Passwort oder null, wenn es keines gibt
	 */
	public String getCachedPasswort(final Benutzer b) {
		return cachedPasswords.get(b);
	}

	/**
	 * Löscht das gecachte Passwort für den Benutzer
	 *
	 * @param b der betroffene Benutzer
	 */
	public void invalidatePasswordCache(final Benutzer b) {
		cachedPasswords.remove(b);
	}

	/**
	 * Aktualisiert den Passwort-Cache des Benutzers
	 *
	 * @param b        der betroffene Benutzer
	 * @param password das neue Passwort
	 */
	public void updatePasswordCache(final Benutzer b, final String password) {
		cachedPasswords.put(b, password);
	}

	/**
	 * &Auml;ndert das Anmeldekennwort eines Benutzer. Das &Auml;ndern erfolgt
	 * unabh&auml;ngig von Berechtigungsklassen nur unter Pr&uuml;fung der
	 * entsprechenden Rechte am DAV.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt.
	 * @param benutzer       der Benutzer, dessen Passwort geändert werden soll.
	 * @param neuesPasswort  das neue Passwort des Benutzer.
	 * @return der neue Benutzer
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Anlegen des Benutzers ein Fehler
	 *                                 eintrat.
	 */
	public Benutzer aendernPasswort(final String adminLoginname, final String adminPasswort, final Benutzer benutzer,
			final String neuesPasswort) throws KeineRechteException, BenutzerChangeException {
		// wenn nicht das eigene Passwort geändert werden soll, prüfe ob der
		// Anfordernde ein Administrator ist, um eine ordentliche Fehlermeldung
		// zu erzeugen
		if (!benutzer.equals(getAngemeldetenBenutzer())) {
			if (!isDAVAdmin(adminLoginname, adminPasswort)) {
				throw new KeineRechteException(
						"Sie verfügen nicht über ausreichend Rechte zum Ändern des Passwortes eines Benutzer.");
			}
		}

		try {
			final DataModel modell = factory.getDav().getDataModel();
			final UserAdministration userAdmin = modell.getUserAdministration();
			userAdmin.changeUserPassword(adminLoginname, adminPasswort, benutzer.getName(), neuesPasswort);
		} catch (final ConfigurationTaskException ex) {
			throw new BenutzerChangeException(
					"Das Passwort des Benutzers " + benutzer.getName() + " konnte nicht geändert werden.", ex);
		}

		return benutzer;
	}

	/**
	 * Prüft, ob ein Bennutzer der Berechtigungsklasse
	 * {@link #PID_KLASSE_ADMINISTRATOR} zugeordnet ist.
	 *
	 * @param loginname ein beliebiger Nutzername
	 * @return {@code true}, wenn der Bennutzer Administratoraufgaben ausführen
	 *         darf.
	 */
	public boolean isAdmin(final String loginname) {
		final Benutzer benutzer = getAngemeldetenBenutzer();
		if (neueRechtepruefung()) {
			BerechtigungsklasseNeu klasse = getBerechtigungsklasseNeu(PID_KLASSE_ADMINISTRATOR);
			return isBerechtigungsklasse(benutzer, klasse);
		}
		final Berechtigungsklasse klasse = getBerechtigungsklasse(PID_KLASSE_ADMINISTRATOR);
		return isBerechtigungsklasse(benutzer, klasse);
	}

	/**
	 * Deaktiviert einen Benutzer. Seine Berechtigungsklasse wird auf "Kein Zugriff"
	 * gesetzt.
	 * <p>
	 * TODO Passwort überprüfen.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt. <em>Wird derzeit ignoriert, da es nicht
	 *                       benötigt wird!</em>
	 * @param benutzer       der zu deaktivierende Benutzers.
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn es beim deaktivieren einen Fehler gab.
	 */
	public void deaktiviereBenutzer(final String adminLoginname, final String adminPasswort, final Benutzer benutzer)
			throws KeineRechteException, BenutzerChangeException {
		final ObjektFactory factory = DefaultObjektFactory.getInstanz();
		final Berechtigungsklasse klasse = (Berechtigungsklasse) factory.getModellobjekt(PID_KLASSE_KEIN_ZUGRIFF);
		setBerechtigungsklasse(adminLoginname, adminPasswort, benutzer, klasse);
	}

	/**
	 * Ändert die Berechtigungsklasse eines Benutzer.
	 *
	 * @param adminLoginname der Name des Administrators der die Aktion ausführt.
	 * @param adminPasswort  das Anmeldekennwort des Administrators der die Aktion
	 *                       ausführt.
	 * @param benutzer       der zu deaktivierende Benutzers.
	 * @param klasse         die zu setzende Berechtigungsklasse.
	 * @throws KeineRechteException    wenn die Benutzerrechte für diese Aktion
	 *                                 nicht ausreichen.
	 * @throws BenutzerChangeException wenn beim Ändern der Benutzerrechte ein
	 *                                 Fehler eintrat.
	 * 
	 * @deprecated Use
	 *             {@link Benutzerverwaltung#setBerechtigungsklasse(String, String, Benutzer, BerechtigungsklasseNeu)}
	 */
	public void setBerechtigungsklasse(final String adminLoginname, final String adminPasswort, final Benutzer benutzer,
			final Berechtigungsklasse klasse) throws KeineRechteException, BenutzerChangeException {
		if (!isAdmin(adminLoginname)) {
			throw new KeineRechteException(
					"Sie verfügen nicht über ausreichend Rechte zum Ändern der Berechtigungsklasse eines Benutzers.");
		}

		final PdBenutzerParameter parameter = benutzer.getPdBenutzerParameter();
		final PdBenutzerParameter.Daten datum = parameter.createDatum();
		datum.setBerechtigungsklasse(klasse);
		try {
			parameter.anmeldenSender();
			parameter.sendeDatum(datum);

			try {
				if (PID_KLASSE_ADMINISTRATOR.equals(klasse.getPid())) {
					final DataModel modell = factory.getDav().getDataModel();
					final UserAdministration userAdmin = modell.getUserAdministration();
					userAdmin.changeUserRights(adminLoginname, adminPasswort, benutzer.getName(), true);
				} else {
					final DataModel modell = factory.getDav().getDataModel();
					final UserAdministration userAdmin = modell.getUserAdministration();
					userAdmin.changeUserRights(adminLoginname, adminPasswort, benutzer.getName(), false);
				}
			} catch (final ConfigurationTaskException ex) {
				throw new BenutzerChangeException(
						"Die Adminrechte für den Benutzer " + benutzer + " konnten nicht geändert werden.", ex);
			}
		} catch (final AnmeldeException ex) {
			throw new BenutzerChangeException("Fehler beim Anmelden auf Parameter für Benutzer " + benutzer + ".", ex);
		} catch (final DatensendeException ex) {
			throw new BenutzerChangeException("Fehler beim Senden des Parameters für Benutzer " + benutzer + ".", ex);
		} finally {
			parameter.abmeldenSender();
		}
	}

	public void setBerechtigungsklasse(final String adminLoginname, final String adminPasswort, final Benutzer benutzer,
			final BerechtigungsklasseNeu klasse) throws KeineRechteException, BenutzerChangeException {
		if (!isAdmin(adminLoginname)) {
			throw new KeineRechteException(
					"Sie verfügen nicht über ausreichend Rechte zum Ändern der Berechtigungsklasse eines Benutzers.");
		}

		setBerechtigungsklasse(adminLoginname, adminPasswort, benutzer, klasse, adminPasswort, "", "");
		try {
			final DataModel modell = factory.getDav().getDataModel();
			if (PID_KLASSE_ADMINISTRATOR.equals(klasse.getPid())) {
				final UserAdministration userAdmin = modell.getUserAdministration();
				userAdmin.changeUserRights(adminLoginname, adminPasswort, benutzer.getName(), true);
			} else {
				final UserAdministration userAdmin = modell.getUserAdministration();
				userAdmin.changeUserRights(adminLoginname, adminPasswort, benutzer.getName(), false);
			}
		} catch (final ConfigurationTaskException ex) {
			throw new BenutzerChangeException(
					"Die Adminrechte für den Benutzer " + benutzer + " konnten nicht geändert werden.", ex);
		}
	}

}
