/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.IMonitorProgressListener;

import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.util.monitor.DebugMonitor;

/**
 * Basisklasse für alle Caches.
 * 
 * Wenn die ableitende Klasse eine Methode preInit() hat, wird diese vor dem
 * Aufruf von {@link #doInit(IMonitor)} mit den ParentCahces aus dem Construktor
 * aufgerufen. Da die Methodenparameter von den Parent-Caches abhängig sind,
 * kann auch keine Superklassenmethode existieren.
 * 
 * <p>
 * Achtung (TODO): Es kommt hier gelegentlich zu Deadlocks, wegen der diversen
 * Synchronized Methoden bzw. sonstigen Synchonisierungen (Methoden
 * {@link AbstractCache#isInitialisiert()} und
 * {@link AbstractCache#setState(String)} ). Es wurde erstmal versucht des
 * Problem durch scharfes draufschauen und überlegen zu beheben -es besteht
 * jedoch die Möglichkeit, dass das Problem weiterhin besteht.
 * </p>
 * 
 * @author BitCtrl Systems GmbH, schnepel
 */
public abstract class AbstractCache {

	private static class BlockAbrufManager {

		private static class BlockAbrufKey {

			private final DataModel model;
			private final long typeId;
			private final long atgId;

			public BlockAbrufKey(final DataModel model, final long typeId, final long atgId) {
				this.model = model;
				this.typeId = typeId;
				this.atgId = atgId;
			}

			@Override
			public int hashCode() {
				final int prime = 31;
				int result = 1;
				result = prime * result + (int) (atgId ^ (atgId >>> 32));
				result = prime * result + (int) (typeId ^ (typeId >>> 32));
				return result;
			}

			@Override
			public boolean equals(final Object obj) {
				if (this == obj) {
					return true;
				}
				if (obj == null) {
					return false;
				}
				if (getClass() != obj.getClass()) {
					return false;
				}
				final BlockAbrufKey other = (BlockAbrufKey) obj;
				if (model != other.model) {
					return false;
				}
				if (atgId != other.atgId) {
					return false;
				}
				if (typeId != other.typeId) {
					return false;
				}
				return true;
			}
		}

		private final Set<BlockAbrufKey> abgerufen = new HashSet<>();

		public void blockAbruf(final DataModel model, final SystemObjectType type, final AttributeGroup[] atgs) {

			if ((model != null) && (type != null) && (atgs != null)) {
				List<SystemObject> elements = null;
				for (final AttributeGroup atg : atgs) {
					final BlockAbrufKey key = new BlockAbrufKey(model, type.getId(), atg.getId());
					if (!abgerufen.contains(key)) {
						if (elements == null) {
							elements = type.getElements();
						}
						model.getConfigurationData(elements, atg);
						abgerufen.add(key);
					}
				}
			}
		}
	}

	/**
	 * Flag, das steuert, ob der Abruf von Konfigurationsdaten als Blockabruf
	 * (f&uuml;r eine Liste von Systemobjekten) oder f&uuml;r jedes Objekt einzeln
	 * durchgef&uuml;hrt werden soll.
	 * <p>
	 * In der gegenw&auml;rtigen Implementation der DAV-API wird beim Blockabruf der
	 * lokale Cache nicht benutzt, so dass bei vorhandenem Cache die Einzelabfragen
	 * deutlich schneller sind.
	 * </p>
	 */
	private boolean benutzeBlockAbruf = false;

	private final BlockAbrufManager blockAbrufManager = new BlockAbrufManager();

	public final void blockAbruf(final DataModel model, final SystemObjectType type, final AttributeGroup atg) {
		blockAbruf(model, type, new AttributeGroup[] { atg });
	}

	public void blockAbruf(final DataModel model, final SystemObjectType type, final AttributeGroup[] atgs) {
		blockAbrufManager.blockAbruf(model, type, atgs);
	}

	public boolean isBenutzeBlockAbruf() {
		return benutzeBlockAbruf;
	}

	public void setBenutzeBlockAbruf(final boolean benutzerBlockAbruf) {
		benutzeBlockAbruf = benutzerBlockAbruf;
	}

	public interface CacheMonitor {
		public void monitorCache(AbstractCache cache);
	}

	private final PropertyChangeListener parentListener = new PropertyChangeListener() {
		public void propertyChange(final PropertyChangeEvent evt) {
			if (AbstractCache.PROP_STATE.equals(evt.getPropertyName())) {
				if (AbstractCache.PROP_STATE_INITIALIZED.equals(evt.getNewValue())) {
					testAndCallDoInit();
				} else if (AbstractCache.PROP_STATE_CANCELED.equals(evt.getNewValue())) {
					cancel();
				} else if (AbstractCache.PROP_STATE_STARTED.equals(evt.getNewValue()) && null != evt.getOldValue()) {
					System.out.println(AbstractCache.this.getClass().getName() + " needs a redoInit()");
					// throw new UnsupportedOperationException();
				}
			}
		}
	};

	public static final String PROP_STATE = "state";
	public static final String PROP_STATE_STARTED = "started";
	public static final String PROP_STATE_INITIALIZED = "initialized";
	public static final String PROP_STATE_CANCELED = "canceled";

	public static final String PROP_PROGRESS = "progress";

	private final PropertyChangeSupport pcsDelegate = new PropertyChangeSupport(this);
	private String state = null;

	private final AbstractCache[] parentCaches;

	private final IMonitorProgressListener monitorListener = new IMonitorProgressListener() {
		public void progress(final double progress) {
			firePropertyChanged(PROP_PROGRESS, null, progress);
		}
	};

	private static CacheMonitor cacheMonitor;

	private String name;

	protected AbstractCache(final AbstractCache... parentCaches) {
		this(null, parentCaches);
	}

	protected AbstractCache(final String name, final AbstractCache... parentCaches) {
		setName(null == name ? getClass().getSimpleName() : name);
		this.parentCaches = parentCaches;

		if (cacheMonitor != null) {
			cacheMonitor.monitorCache(this);
		}

		testAndCallDoInit();
	}

	private void testAndCallDoInit() {
		boolean ok = true;
		synchronized (this) {
			if (null != parentCaches) {
				for (final AbstractCache parentCache : parentCaches) {
					parentCache.removePropertyChangeListener(AbstractCache.PROP_STATE, parentListener);
					parentCache.addPropertyChangeListener(AbstractCache.PROP_STATE, parentListener);
					if (!parentCache.isInitialisiert()) {
						ok = false;
					}
				}
			}
		}
		if (ok && callPreInit()) {
			callDoInit(new DebugMonitor());
		}

	}

	private boolean callPreInit() {
		boolean preInitOk = true;

		Class<?> clazz = getClass();
		loopClasses: while (null != clazz) {
			final Method[] methods = clazz.getDeclaredMethods();
			loopMethods: for (final Method method : methods) {
				if (method.getName().equals("preInit")) {
					final Class<?>[] parameterTypes = method.getParameterTypes();
					if (parentCaches.length != parameterTypes.length) {
						continue;
					}
					for (int i = 0; i < parameterTypes.length; i++) {
						if (!parameterTypes[i].isInstance(parentCaches[i])) {
							continue loopMethods;
						}
					}
					try {
						method.setAccessible(true);
						method.invoke(this, (Object[]) parentCaches);
					} catch (final Exception ex) {
						ex.printStackTrace();
						preInitOk = false;
					}
					break loopClasses;
				}
			}
			clazz = clazz.getSuperclass();
		}
		return preInitOk;
	}

	protected void callDoInit(final IMonitor monitor) {
		boolean ret;
		try {
			monitor.addMonitorListener(monitorListener);
			setState(AbstractCache.PROP_STATE_STARTED);

			ret = doInit(monitor);

		} catch (final Exception e) {
			e.printStackTrace();
			ret = false;
		} finally {
			monitor.removeMonitorListener(monitorListener);
		}
		if (ret) {
			done();
		} else {
			monitor.setCanceled(true);
			cancel();
		}
	}

	protected abstract boolean doInit(IMonitor monitor);

	/**
	 * Registriert einen {@link PropertyChangeListener} für alle Properties.
	 * 
	 * @param listener der zu registrierende Listener.
	 */
	public final void addPropertyChangeListener(final PropertyChangeListener listener) {
		pcsDelegate.addPropertyChangeListener(listener);
	}

	/**
	 * Registriert einen {@link PropertyChangeListener} nur für eine bestimmte
	 * Property.
	 * 
	 * @param propertyName der Name der Property, die beobachtet werden soll.
	 * @param listener     der zu registrierende Listener.
	 */
	public final void addPropertyChangeListener(final String propertyName, final PropertyChangeListener listener) {
		pcsDelegate.addPropertyChangeListener(propertyName, listener);
	}

	/**
	 * Deregistriert einen {@link PropertyChangeListener} für alle Properties.
	 * 
	 * @param listener der zu deregistrierende Listener.
	 */
	public final void removePropertyChangeListener(final PropertyChangeListener listener) {
		pcsDelegate.removePropertyChangeListener(listener);
	}

	/**
	 * Deregistriert einen {@link PropertyChangeListener} nur für eine bestimmte
	 * Property.
	 * 
	 * @param propertyName der Name der Property, die nicht mehr beobachtet werden
	 *                     soll.
	 * @param listener     der zu deregistrierende Listener.
	 */
	public final void removePropertyChangeListener(final String propertyName, final PropertyChangeListener listener) {
		pcsDelegate.removePropertyChangeListener(propertyName, listener);
	}

	/**
	 * Verteilt die Nachricht über die Änderung einer Property.
	 * 
	 * @param propertyName der Name der geänderten Property.
	 * @param oldValue     der alte Propertywert.
	 * @param newValue     der neue Propertywert.
	 */
	protected final synchronized void firePropertyChanged(final String propertyName, final Object oldValue,
			final Object newValue) {
		pcsDelegate.firePropertyChange(propertyName, oldValue, newValue);
	}

	protected final synchronized void ensureInit() {
		if (!isInitialisiert()) {
			throw new IllegalStateException(getClass().getSimpleName() + " wurde nicht initialisiert.");
		}
	}

	/**
	 * Flag ob der Netzcache initialisiert wurde. Will man auf das warten bis der
	 * Cache initialisiert wurde, kann ein {@link PropertyChangeListener} für die
	 * Property {@link #PROP_STATE} registriert werden.
	 * 
	 * @return <code>true</code>, wenn der Cache initialister ist, sonst
	 *         <code>false</code>.
	 */
	public final boolean isInitialisiert() {
		return AbstractCache.PROP_STATE_INITIALIZED.equals(state);
	}

	private void cancel() {
		setState(AbstractCache.PROP_STATE_CANCELED);
	}

	private void done() {
		setState(AbstractCache.PROP_STATE_INITIALIZED);
	}

	public static final void setCacheMonitor(final CacheMonitor cacheMonitor) {
		if (AbstractCache.cacheMonitor != null) {
			throw new IllegalArgumentException("CacheMonitor already set!");
		}
		AbstractCache.cacheMonitor = cacheMonitor;
	}

	public final synchronized void redoInit() {
		setState(null);
		if (cacheMonitor != null) {
			cacheMonitor.monitorCache(this);
		}
		callDoInit(new DebugMonitor());
	}

	public String getState() {
		return state;
	}

	public synchronized void setState(final String newState) {
		if ((newState != null) && (!newState.equals(state))) {
			final String oldState = state;
			state = newState;
			firePropertyChanged(AbstractCache.PROP_STATE, oldState, state);
		}
	}

	public String getName() {
		return name;
	}

	public void setName(final String name) {
		this.name = name;
	}

}