/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobal.objekte.AnzeigeQuerschnitt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobal.objekte.WvzInhalt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobalzwischenschicht.konfigurationsdaten.KdWvzInhaltUmsetzung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmanzeigenglobalzwischenschicht.objekte.WvzInhaltUmsetzung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdPunktLiegtAufLinienObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Linie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Cache zur Lokalisierung von {@link AnzeigeQuerschnitt}en auf
 * {@link StrassenSegment}en.
 *
 * @author BitCtrl Systems GmbH, Hoesel
 * @version $Id$
 */
public class AqCache extends AbstractCache {

	private final Map<StrassenSegment, Set<AnzeigeQuerschnitt>> strassenAqMap = new HashMap<>();
	private final Map<AnzeigeQuerschnitt, StrassenSegment> aqStrassenMap = new HashMap<>();
	private final Map<WvzInhalt, WvzInhaltUmsetzung> wvzInhaltwvzInhaltUmsetzungMap = new HashMap<>();

	private AbstractDavVerbindungsCache verbindungsCache;

	/**
	 * @param verbindungsCache der verwendete Verbindungs-Cache
	 *
	 */
	public AqCache(final AbstractDavVerbindungsCache verbindungsCache) {
		super("Anzeigequerschnitt-Cache", verbindungsCache);
	}

	public void preInit(final AbstractDavVerbindungsCache verbindungsCache) {
		this.verbindungsCache = verbindungsCache;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see de.bsvrz.sys.funclib.bitctrl.modell.util.cache.AbstractCache#doInit(com
	 * .bitctrl.util.monitor.IMonitor)
	 */
	@Override
	protected boolean doInit(final IMonitor monitor) {
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();
		final List<StrassenSegment> strassenSegmente = (List<StrassenSegment>) objektFactory
				.bestimmeModellobjekte(StrassenSegment.PID);

		final long start = System.currentTimeMillis();

		final List<AnzeigeQuerschnitt> aqList = (List<AnzeigeQuerschnitt>) objektFactory
				.bestimmeModellobjekte(AnzeigeQuerschnitt.PID);
		subMonitor.beginTask("Lade Anzeigequerschnitte", aqList.size());

		final List<KdPunktLiegtAufLinienObjekt.Daten> daten = objektFactory.getDaten(aqList,
				KdPunktLiegtAufLinienObjekt.class);
		for (int i = 0; i < aqList.size(); i++) {
			try {
				final AnzeigeQuerschnitt mqa = aqList.get(i);
				final KdPunktLiegtAufLinienObjekt.Daten datum = daten.get(i);
				subMonitor.setTaskName(mqa.getName());
				if (null == datum) {
					continue;
				}
				final Linie l = datum.getLinienReferenz();
				if (l instanceof StrassenSegment) {
					final StrassenSegment mqSegment = (StrassenSegment) l;
					if (!strassenSegmente.contains(mqSegment)) {
						continue;
					}
					aqStrassenMap.put(mqa, mqSegment);
					Set<AnzeigeQuerschnitt> mqs = strassenAqMap.get(mqSegment);
					if (null == mqs) {
						mqs = new HashSet<>();
						strassenAqMap.put(mqSegment, mqs);
					}
					mqs.add(mqa);

				}
			} finally {
				subMonitor.worked(1);
			}
		}

		final SubMonitor subMonitor2 = SubMonitor.convert(monitor);
		final List<WvzInhaltUmsetzung> wvzInhaltUmsetzungen = (List<WvzInhaltUmsetzung>) objektFactory
				.bestimmeModellobjekte(WvzInhaltUmsetzung.PID);

		subMonitor2.beginTask("Lade WvzInhalte", wvzInhaltUmsetzungen.size());
		final List<KdWvzInhaltUmsetzung.Daten> datenWvz = objektFactory.getDaten(wvzInhaltUmsetzungen,
				KdWvzInhaltUmsetzung.class);
		for (int i = 0; i < wvzInhaltUmsetzungen.size(); i++) {
			try {
				final WvzInhaltUmsetzung umsetzung = wvzInhaltUmsetzungen.get(i);
				final KdWvzInhaltUmsetzung.Daten datum = datenWvz.get(i);
				subMonitor.setTaskName(umsetzung.getName());
				if (null == datum) {
					continue;
				}
				final WvzInhalt l = datum.getWvzInhaltReferenz();
				if (l != null) {
					wvzInhaltwvzInhaltUmsetzungMap.put(l, umsetzung);
				}
			} finally {
				subMonitor.worked(1);
			}
		}

		Debug.getLogger().info("Benötigte Zeit zum Initialisieren des BitCtrl Anzeigequerschnitt-Caches: "
				+ (System.currentTimeMillis() - start) + " ms");
		return true;
	}

	/**
	 * Liefert das {@link StrassenSegment}, zu dem der übergebene
	 * {@link AnzeigeQuerschnitt} gehört. Oder <code>null</code>, wenn der
	 * {@link AnzeigeQuerschnitt} zu keinem {@link StrassenSegment} gehört. das ist
	 * immer dann der Fall, wenn die Atg {@link KdPunktLiegtAufLinienObjekt} nicht
	 * versorgt ist.
	 *
	 * @param aq der AQ für den das Straßensegment ermittelt werden soll
	 * @return das {@link StrassenSegment}, auf dem der {@link AnzeigeQuerschnitt}
	 *         liegt.
	 */
	public StrassenSegment getStrassenSegment(final AnzeigeQuerschnitt aq) {
		ensureInit();
		return aqStrassenMap.get(aq);
	}

	/**
	 * Liefert alle {@link AnzeigeQuerschnitt}e, die zu dem übergebenen
	 * {@link StrassenSegment} gehören.
	 *
	 * @param strassenSegment das Straßensegment
	 * @return die Menge der AQ
	 */
	public Set<AnzeigeQuerschnitt> getAnzeigeQuerschnitt(final StrassenSegment strassenSegment) {
		ensureInit();
		if (strassenAqMap.containsKey(strassenSegment)) {
			return Collections.unmodifiableSet(strassenAqMap.get(strassenSegment));
		}
		return new HashSet<>();
	}

	/**
	 * Sucht die {@link WvzInhaltUmsetzung} eines {@link WvzInhalt}es.
	 *
	 * @param inhalt Der WvzInhalt, dessen {@link WvzInhaltUmsetzung} gesucht wird.
	 * @return Liefert die passende {@link WvzInhaltUmsetzung} zu dem übergebenen
	 *         {@link WvzInhalt}, oder <code>null</code>.
	 */
	public WvzInhaltUmsetzung getWvzInhaltUmsetzung(final WvzInhalt inhalt) {
		return wvzInhaltwvzInhaltUmsetzungMap.get(inhalt);
	}

}
