/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.Collection;
import java.util.List;
import java.util.TreeMap;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.mifimportobjektetypen.objekte.Gewaesser;
import de.bsvrz.sys.funclib.bitctrl.modell.mifimportobjektetypen.objekte.Kreis;
import de.bsvrz.sys.funclib.bitctrl.modell.mifimportobjektetypen.objekte.Ortslage;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdKomplexKoordinaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.KomplexXY;
import de.bsvrz.sys.funclib.bitctrl.modell.util.DavCacheLader;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Cache fuer Gewaesser, Kreise und Orte.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 * 
 *         XXX Prüfen, warum der Serializable implemntiert hatte
 */
public class MifCache extends AbstractCache {

	private final TreeMap<Long, KomplexXYCacheObject> kreisMap = new TreeMap<>();

	private final TreeMap<Long, KomplexXYCacheObject> gewaesserMap = new TreeMap<>();

	private final TreeMap<Long, KomplexXYCacheObject> ortsMap = new TreeMap<>();

	private AbstractDavVerbindungsCache davVerbindungsCache;

	public MifCache(final AbstractDavVerbindungsCache davVerbCache) {
		super("Mif-Cache", davVerbCache);
	}

	public void preInit(final AbstractDavVerbindungsCache davVerbCache) {
		davVerbindungsCache = davVerbCache;
	}

	@Override
	protected boolean doInit(final IMonitor monitor) {
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		try {
			DavCacheLader.lade(davVerbindungsCache.getClientDavInterface(), KomplexXY.PID, KdKomplexKoordinaten.PID);

			final List<? extends SystemObjekt> kreise = davVerbindungsCache.getObjektFactory()
					.bestimmeModellobjekte(Kreis.PID);
			final List<? extends SystemObjekt> orte = davVerbindungsCache.getObjektFactory()
					.bestimmeModellobjekte(Ortslage.PID);
			final List<? extends SystemObjekt> gewaesserList = davVerbindungsCache.getObjektFactory()
					.bestimmeModellobjekte(Gewaesser.PID);
			subMonitor.beginTask("Initialisiere MIF-Cache ...", kreise.size() + orte.size() + gewaesserList.size());

			final SubMonitor kreisMonitor = subMonitor.newChild(kreise.size());
			kreisMonitor.beginTask("Lade Kreise", kreise.size());
			for (final SystemObjekt kreis : kreise) {
				kreisMap.put(kreis.getId(), new KomplexXYCacheObject((KomplexXY) kreis));
				kreisMonitor.worked(1);
				if (kreisMonitor.isCanceled() || subMonitor.isCanceled()) {
					return false;
				}
			}

			final SubMonitor ortsMonitor = subMonitor.newChild(orte.size());
			ortsMonitor.beginTask("Lade Orte", orte.size());
			for (final SystemObjekt ort : orte) {
				ortsMap.put(ort.getId(), new KomplexXYCacheObject((KomplexXY) ort));
				ortsMonitor.worked(1);

				if (ortsMonitor.isCanceled() || subMonitor.isCanceled()) {
					return false;
				}
			}

			final SubMonitor gewaesserMonitor = subMonitor.newChild(gewaesserList.size());
			gewaesserMonitor.beginTask("Lade Gewaesser", gewaesserList.size());

			for (final SystemObjekt gewaesser : gewaesserList) {
				gewaesserMap.put(gewaesser.getId(), new KomplexXYCacheObject((KomplexXY) gewaesser));
				gewaesserMonitor.worked(1);
				if (gewaesserMonitor.isCanceled() || subMonitor.isCanceled()) {
					return false;
				}
			}
			return true;
		} catch (final Exception ex) {
			ex.printStackTrace();
			Debug.getLogger().error("Initialisierung fehlgeschlagen.", ex);
		}
		return false;
	}

	public Collection<KomplexXYCacheObject> getKreise() {
		return kreisMap.values();
	}

	public Collection<KomplexXYCacheObject> getOrte() {
		return ortsMap.values();
	}

	public Collection<KomplexXYCacheObject> getGewaesser() {
		return kreisMap.values();
	}

	/**
	 * @param komplexXY das Cache-Objekt zum übergebenen Objekt
	 * @return das ermittelte Cache-Objekt oder null
	 */
	public KomplexXYCacheObject get(final KomplexXY komplexXY) {
		if (komplexXY instanceof Gewaesser) {
			return gewaesserMap.get(komplexXY.getId());
		} else if (komplexXY instanceof Kreis) {
			return kreisMap.get(komplexXY.getId());
		} else if (komplexXY instanceof Ortslage) {
			return ortsMap.get(komplexXY.getId());
		}
		throw new IllegalArgumentException(
				komplexXY + " ist nicht vom Typ " + KomplexXY.PID + " (sondern: " + komplexXY.getTyp().getPid() + ")");
	}

}
