/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.NullMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdAeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrasse.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.VerkehrsModellNetz;

/**
 * <p>
 * Diese Hilfsklasse cacht sämtliche für dieses Plugin benötigten Informationen
 * über das Straßennetz. Es werden dabei sämtliche dem System bekannte Straßen,
 * Straßenknoten, innere und äußere Straßensegmente sowie Straßenteilsegmente
 * geladen und in entsprechenden Maps für spätere Berechnungen vorgehalten.
 * </p>
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 * @version $Id: StrassenCache.java 22179 2010-03-16 18:31:09Z meissner $
 */
public class NetzCacheExtended extends AbstractCache {

	/** Liste aller im System definierten Straßen. */
	private List<Strasse> strassen;

	/**
	 * Liste aller im System definierten Straßen, welche mindestens in einer
	 * Strassenrichtung ein äußeres Straßensegment besitzen.
	 */
	private List<Strasse> strassenMitSegmenten;

	/**
	 * Map, welche zu einer Straßennummer alle dazugehörigen Straßen (LCL-Segmente)
	 * speichert. Die Liste der Straßen ist dabei in Durchfahrtrichtung sortiert.
	 */
	private Map<String, List<Strasse>> strNummer2strMap;

	/**
	 * Map, welche zu einer (positiv- bzw. negativ gerichteteten) Straße eine Liste
	 * aller zu der Straße gehörende äußere Straßensegmente speichert.
	 */
	private Map<AttTmcRichtung, Map<Strasse, List<AeusseresStrassenSegment>>> str2assMap;

	/**
	 * Map, welche zu einer (positiv- bzw. negativ gerichteteten) Straße eine Liste
	 * aller zu der Straße gehörende Straßenknoten speichert.
	 */
	private Map<AttTmcRichtung, Map<Strasse, List<StrassenKnoten>>> str2skMap;

	/**
	 * Map, welche (abhängig von Straße und Straßenrichtung) von einem Straßenknoten
	 * die direkten hinführenden äußeren Straßensegmente speichert.
	 */
	private Map<AttTmcRichtung, Map<Strasse, Map<StrassenKnoten, List<AeusseresStrassenSegment>>>> str2assVorSkMap;

	/**
	 * Map, welche (abhängig von Straße und Straßenrichtung) von einem Straßenknoten
	 * die direkten wegführenden äußeren Straßensegmente speichert.
	 */
	private Map<AttTmcRichtung, Map<Strasse, Map<StrassenKnoten, List<AeusseresStrassenSegment>>>> str2assNachSk;

	/**
	 * Der TMC-Cache von dem dieser Cache abhängig ist.
	 */
	private TmcCache tmcCache;

	/**
	 * Der Netz-Cache von dem dieser Cache indirekt abhängig ist.
	 */
	private NetzCache netzCache;

	/**
	 * Der Konstruktor.
	 *
	 * @param tmcCache der TmcCache, von welchem dieser Cache abhängig ist
	 */
	public NetzCacheExtended(final TmcCache tmcCache) {
		super(tmcCache);
	}

	/**
	 * Initialisiert diesen Cache.
	 *
	 * @param parent der TmcCache, von welchem dieser Cache abhängig ist
	 */
	protected void preInit(final TmcCache parent) {
		tmcCache = parent;
		netzCache = tmcCache.getNetzCache();
	}

	/**
	 * Liefert den TmcCache zurück, von welchem dieser Cache abhängig ist.
	 *
	 * @return den TmcCache, von welchem dieser Cache abhängig ist
	 */
	protected final TmcCache getTmcCache() {
		return tmcCache;
	}

	/**
	 * Liefert den NetzCache zurück, von welchem dieser Cache abhängig ist.
	 *
	 * @return den NetzCache, von welchem dieser Cache abhängig ist
	 */
	public final NetzCache getNetzCache() {
		return tmcCache.getNetzCache();
	}

	@Override
	protected boolean doInit(final IMonitor monitor) {
		setName(getNetzCache().getName());

		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere Straßennetz", 110000);

		final SubMonitor subMonitor2 = subMonitor.newChild(10000);
		subMonitor2.beginTask("Sortiere Straßen", 1);
		ordneStrassen();
		subMonitor2.worked(1);
		if (subMonitor2.isCanceled()) {
			subMonitor2.done();
			return false;
		}

		return ordneStrassenSegmenteUndKnoten(subMonitor.newChild(10000));
	}

	/**
	 * Sortiert die Liste der Straßen und ordnet den Straßennummern eine Liste der
	 * dazughörigen Straßen zu.
	 */
	private void ordneStrassen() {
		strNummer2strMap = new HashMap<>();
		for (final Strasse strasse : getStrassenInternal()) {
			final String strassenNummer = getStrassenNummer(strasse);
			if (strassenNummer != null) {
				List<Strasse> liste = strNummer2strMap.get(strassenNummer);
				if (liste == null) {
					liste = new ArrayList<>();
					strNummer2strMap.put(strassenNummer, liste);
				}
				liste.add(strasse);
			}
		}
	}

	/**
	 * Ermittelt zu einer Straße die dazugehörige Straßennummer. Diese Methode ist
	 * nur für interne Verwendung gedacht.
	 *
	 * @param strasse die Strasse
	 *
	 * @return zu einer Straße die dazugehörige Straßennummer
	 */
	/* package */String getStrassenNummer(final Strasse strasse) {
		if ((strasse != null) && (strasse.getKdStrasse() != null)) {
			final Daten daten = strasse.getKdStrasse().getDatum();
			if (daten != null) {
				return getStrassenNummer(daten);
			}
		}
		return null;
	}

	/* package */String getStrassenNummer(final Daten daten) {
		if (daten != null) {
			return daten.getTyp().toString() + daten.getNummer().byteValue() + daten.getZusatz();
		}
		return null;
	}

	/**
	 * Liefert die Liste aller im System definierten Straßen zurück.
	 *
	 * <p>
	 * Die zurückgegebene Liste der Straßen ist bereits sortiert nach Typ, Nummer
	 * und Zusatz.
	 * </p>
	 *
	 * @return die Liste aller im System definierten Straßen, niemals
	 *         <code>null</code>
	 */
	public List<Strasse> getStrassen() {
		ensureInit();
		return Collections.unmodifiableList(getStrassenInternal());
	}

	private List<Strasse> getStrassenInternal() {
		if (strassen == null) {
			strassen = new ArrayList<>(getNetzCache().getStrassen());

			Collections.sort(strassen, new Comparator<Strasse>() {
				@Override
				public int compare(final Strasse s1, final Strasse s2) {
					return compareStrassen(s1, s2);
				}
			});
		}
		return strassen;
	}

	/**
	 * Liefert eine (in Durchfahrtsrichtung) sortierte Liste der Straßen zurück,
	 * welche dieselbe Straßennummer haben (auf derselben logischen Straßen liegen
	 * bzw. diese bilden).
	 *
	 * @param daten die Daten einer Staße (eines LCL-Segments), womit auf die
	 *              Straßennummer geschlossen werden kann, nicht <code>null</code>
	 *
	 * @return eine (in Durchfahrtsrichtung) sortierte Liste der Straßen zurück,
	 *         welche dieselbe Straßennummer haben
	 */
	public List<Strasse> getStrassen(final Daten daten) {
		ensureInit();
		if (daten == null) {
			throw new IllegalArgumentException("Übergebene Straßen-Daten dürfen nicht null sein.");
		}

		final String strassenNummer = getStrassenNummer(daten);
		if (strassenNummer != null) {
			return strNummer2strMap.get(strassenNummer);
		}
		return null;
	}

	/**
	 * Vergleicht die zwei gegebenen Straßen miteinander. Diese Methode ist dafür
	 * gedacht, Straßen zu sortieren.
	 *
	 * @param strasse1 die erste Strasse, darf nicht <code>null</code> sein
	 * @param strasse2 die erste Strasse, darf nicht <code>null</code> sein
	 *
	 * @return 0 bei Gleichheit; negativer Wert, wenn erste Straße vor zweiter
	 *         Straße liegt soll, ansonsten positiver Wert
	 */
	public int compareStrassen(final Strasse strasse1, final Strasse strasse2) {

		final KdStrasse.Daten s1 = strasse1.getKdStrasse().getDatum();
		final KdStrasse.Daten s2 = strasse2.getKdStrasse().getDatum();

		int compare = s1.getTyp().toString().compareTo(s2.getTyp().toString());
		if (compare == 0) {
			compare = new Integer(s1.getNummer().intValue()).compareTo(new Integer(s2.getNummer().intValue()));
		}
		if (compare == 0) {
			compare = s1.getZusatz().toString().compareTo(s2.getZusatz().toString());
		}
		if (compare == 0) {
			return tmcCache.compareStrasse(strasse1, strasse2);
		}
		return compare;
	}

	/**
	 * Liefert die Liste aller im System definierten Straßen zurück, welche
	 * mindestens in einer Strassenrichtung ein äußeres Straßensegment besitzen..
	 *
	 * <p>
	 * Die zurückgegebene Liste der Straßen ist bereits sortiert nach Typ, Nummer
	 * und Zusatz.
	 * </p>
	 *
	 * @return die Liste aller im System definierten Straßen, welche mindestens in
	 *         einer Strassenrichtung ein äußeres Straßensegment besitzen, niemals
	 *         <code>null</code>
	 */
	public List<Strasse> getStrassenMitSegmenten() {
		ensureInit();
		return getStrassenMitSegmentenInternal();
	}

	private List<Strasse> getStrassenMitSegmentenInternal() {
		if (strassenMitSegmenten == null) {
			strassenMitSegmenten = new ArrayList<>(getStrassenInternal());
			for (final Iterator<Strasse> iter = strassenMitSegmenten.iterator(); iter.hasNext();) {
				final Strasse strasse = iter.next();
				final List<StrassenKnoten> positiveKnoten = getStrassenKnotenInternal(strasse,
						AttTmcRichtung.ZUSTAND_1_POSITIV);
				if ((positiveKnoten != null) && (positiveKnoten.size() > 1)) {
					continue;
				}
				final List<StrassenKnoten> negativeKnoten = getStrassenKnotenInternal(strasse,
						AttTmcRichtung.ZUSTAND_1N_NEGATIV);
				if ((negativeKnoten != null) && (negativeKnoten.size() > 1)) {
					continue;
				}
				iter.remove();
			}
		}
		return strassenMitSegmenten;
	}

	/**
	 * Ordnet den Straßen sämtliche Straßensegmente und Straßenknoten zu.
	 *
	 * @param monitor der Monitor
	 *
	 * @return <code>false</code>, wenn der Monitor gecancelt wurde, ansonsten
	 *         <code>true</code>
	 */
	private boolean ordneStrassenSegmenteUndKnoten(final IMonitor monitor) {
		final List<AeusseresStrassenSegment> aess = getNetzCache().getAeussereStrassenSegmente();
		final int anzahlSegmente = aess.size();

		monitor.beginTask("Ordne äußere Straßensegmente und Straßenknoten", anzahlSegmente);

		// initialisiere Maps
		str2assMap = new HashMap<>();
		str2skMap = new HashMap<>();
		str2assVorSkMap = new HashMap<>();
		str2assNachSk = new HashMap<>();

		for (final AttTmcRichtung r : AttTmcRichtung.getZustaende()) {
			str2assMap.put(r, new HashMap<>());
			str2skMap.put(r, new HashMap<>());
			str2assVorSkMap.put(r, new HashMap<>());
			str2assNachSk.put(r, new HashMap<>());
			for (final Strasse str : getStrassenInternal()) {
				str2assVorSkMap.get(r).put(str, new HashMap<>());
				str2assNachSk.get(r).put(str, new HashMap<>());
			}
		}

		int i = 0;

		for (final AeusseresStrassenSegment segment : aess) {

			monitor.subTask("Analysiere Segment: " + segment.getName() + " (" + i + "/" + anzahlSegmente + ")");

			// lese Konfigurationsdaten aus
			final KdAeusseresStrassenSegment.Daten datum = segment.getKdAeusseresStrassenSegment().getDatum();
			final Strasse gehoertZuStrasse = segment.getKdStrassenSegment().getDatum().getGehoertZuStrasse();
			final StrassenKnoten vonKnoten = datum.getVonKnoten();
			final StrassenKnoten nachKnoten = datum.getNachKnoten();
			final AttTmcRichtung richtung = datum.getTmcRichtung();
			if (richtung == null) {
				continue;
			}

			// füge Segmente und Knoten den jeweiligen Maps hinzu
			List<AeusseresStrassenSegment> segmente = str2assMap.get(richtung).get(gehoertZuStrasse);
			if (segmente == null) {
				segmente = new ArrayList<>();
				str2assMap.get(richtung).put(gehoertZuStrasse, segmente);
			}
			segmente.add(segment);

			List<StrassenKnoten> knoten = str2skMap.get(richtung).get(gehoertZuStrasse);
			if (knoten == null) {
				knoten = new ArrayList<>();
				str2skMap.get(richtung).put(gehoertZuStrasse, knoten);
			}
			if ((vonKnoten != null) && !knoten.contains(vonKnoten)) {
				knoten.add(vonKnoten);
			}
			if ((nachKnoten != null) && !knoten.contains(nachKnoten)) {
				knoten.add(nachKnoten);
			}

			// Wir können Vorgänger- und Nachfolger-Segmente nur bestimmen, wenn
			// das aktuell betrachtete Segment sich auf einer Strasse befindet.
			if (gehoertZuStrasse != null) {
				// Speichert die Informationen Vorgänger- und
				// Nachfolger-Segmente zu Staßenknoten an. Da hier im Einzelfall
				// (bei Verzweigungen einer Straße) vor bzw. nach einem Knoten
				// auch mehrere Segmente auftreten können, ist das Abspeichern
				// einer Liste von Segmenten nötig.
				List<AeusseresStrassenSegment> segmenteNachKnoten = str2assNachSk.get(richtung).get(gehoertZuStrasse)
						.get(vonKnoten);
				if (segmenteNachKnoten == null) {
					segmenteNachKnoten = new ArrayList<>();
					str2assNachSk.get(richtung).get(gehoertZuStrasse).put(vonKnoten, segmenteNachKnoten);
				}
				if (!segmenteNachKnoten.contains(segment)) {
					segmenteNachKnoten.add(segment);
				}

				List<AeusseresStrassenSegment> segmenteVorKnoten = str2assVorSkMap.get(richtung).get(gehoertZuStrasse)
						.get(nachKnoten);
				if (segmenteVorKnoten == null) {
					segmenteVorKnoten = new ArrayList<>();
					str2assVorSkMap.get(richtung).get(gehoertZuStrasse).put(nachKnoten, segmenteVorKnoten);
				}
				if (!segmenteVorKnoten.contains(segment)) {
					segmenteVorKnoten.add(segment);
				}
			}

			monitor.worked(1);
			if (monitor.isCanceled()) {
				monitor.done();
				return false;
			}
			i++;
		}
		return true;
	}

	/**
	 * Liefert die Liste der zu einer gegebenen Staßen gehörigen äußeren
	 * Straßensegmente zurück.
	 *
	 * <p>
	 * <b>Achtung:</b> Bei der Straße kann es sich auch lediglich um ein LCL-Segment
	 * einer logischen Straße handeln. Eine logische Straße wäre z.B. die A5 mit den
	 * LCL-Segmenten 'A5 Basel nach Karlsruhe', 'A5 Karlsruhe nach Heidelberg' und
	 * 'A5 Heidelberg nach Darmstadt'.
	 * </p>
	 *
	 * @param strasse  die Staße, nicht <code>null</code>
	 * @param richtung die Richtung der Straße (positiv oder negativ)
	 *
	 * @return die Liste der zu der gegebenen Staßen gehörigen Straßensegmente
	 */
	public List<AeusseresStrassenSegment> getAeussereStrassenSegmente(final Strasse strasse,
			final AttTmcRichtung richtung) {
		ensureInit();
		if (strasse == null) {
			throw new IllegalArgumentException("Übergebene Straße darf nicht null sein.");
		}

		// gibt eine leere Liste zurück, falls keine äußeren Straßensegemente
		// für eine gegebene Straße in der gegebenen Richtung gefunden wurden
		final Map<Strasse, List<AeusseresStrassenSegment>> map = str2assMap.get(richtung);
		List<AeusseresStrassenSegment> ergebnis = map.get(strasse);
		if (ergebnis == null) {
			ergebnis = new ArrayList<>();
			map.put(strasse, ergebnis);
		}
		return ergebnis;
	}

	/**
	 * Liefert die Liste der zu einer gegebenen Staßennummer (logischen Straße)
	 * gehörigen äußeren Straßensegmente zurück.
	 *
	 * <p>
	 * <b>Achtung:</b> Eine Straßennummer definiert eine logischen Straße. Eine
	 * logische Straße wäre z.B. die 'A5' mit den LCL-Segmenten 'A5 Basel nach
	 * Karlsruhe', 'A5 Karlsruhe nach Heidelberg' und 'A5 Heidelberg nach
	 * Darmstadt'.
	 * </p>
	 *
	 * @param daten    die Daten einer Staße (eines LCL-Segments), womit auf die
	 *                 Straßennummer geschlossen werden kann, nicht
	 *                 <code>null</code>
	 * @param richtung die Richtung der Straße (positiv oder negativ)
	 *
	 * @return die Liste der zu einer gegebenen Staßennummer (logischen Straße)
	 *         gehörigen äußeren Straßensegmente, niemals <code>null</code>
	 *
	 * @see #getAeussereStrassenSegmente(Strasse, AttTmcRichtung)
	 */
	public List<AeusseresStrassenSegment> getAeussereStrassenSegmente(final Daten daten,
			final AttTmcRichtung richtung) {
		ensureInit();
		if (daten == null) {
			throw new IllegalArgumentException("Übergebene Straßen-Daten dürfen nicht null sein.");
		}

		final List<AeusseresStrassenSegment> ergebnis = new ArrayList<>();
		final String strassenNummer = getStrassenNummer(daten);
		if (strassenNummer == null) {
			return ergebnis;
		}
		final List<Strasse> strassenZurNummer = strNummer2strMap.get(strassenNummer);
		if ((strassenZurNummer == null) || strassenZurNummer.isEmpty()) {
			return ergebnis;
		}

		for (final Strasse strasse : strassenZurNummer) {
			final List<AeusseresStrassenSegment> segmente = getAeussereStrassenSegmente(strasse, richtung);
			if (segmente != null) {
				ergebnis.addAll(segmente);
			}
		}
		return ergebnis;
	}

	/**
	 * Liefert eine Liste der äußeren Staßensegmente zurück, welche auf der
	 * entgegengesetzten Straßenrichtung zu dem gegebenen äußeren Straßensegment
	 * liegen.
	 *
	 * @param segment das äußere Straßensegmente, nicht <code>null</code>
	 *
	 * @return eine Liste der äußeren Staßensegmente, welche auf der
	 *         entgegengesetzten Straßenrichtung zu dem gegebenen äußeren
	 *         Straßensegment liegen, niemals <code>null</code>
	 */
	public List<AeusseresStrassenSegment> getAeussereStrassenSegmenteInEntgegengesetzerRichtung(
			final AeusseresStrassenSegment segment) {
		ensureInit();
		return getAeussereStrassenSegmenteInEntgegengesetzerRichtungInternal(segment);
	}

	private List<AeusseresStrassenSegment> getAeussereStrassenSegmenteInEntgegengesetzerRichtungInternal(
			final AeusseresStrassenSegment segment) {
		if (segment == null) {
			throw new IllegalArgumentException("Übergebenes äußeres Straßensegment darf nicht null sein.");
		}

		final Strasse strasse = segment.getKdStrassenSegment().getDatum().getGehoertZuStrasse();
		final AttTmcRichtung richtung = segment.getKdAeusseresStrassenSegment().getDatum().getTmcRichtung();

		if (richtung == AttTmcRichtung.ZUSTAND_0_OHNE_RICHTUNG) {
			return new ArrayList<>();
		}
		final AttTmcRichtung entgegengesetzteRichtung = (richtung == AttTmcRichtung.ZUSTAND_1_POSITIV)
				? AttTmcRichtung.ZUSTAND_1N_NEGATIV
				: AttTmcRichtung.ZUSTAND_1_POSITIV;

		final StrassenKnoten nachKnoten = segment.getKdAeusseresStrassenSegment().getDatum().getNachKnoten();
		if (nachKnoten != null) {
			final List<AeusseresStrassenSegment> ergebnis = getAeussereStrassenSegmenteNachStrassenKnotenInternal(
					nachKnoten, strasse, entgegengesetzteRichtung);
			if ((ergebnis != null) && !ergebnis.isEmpty()) {
				return ergebnis;
			}
		}

		final StrassenKnoten vonKnoten = segment.getKdAeusseresStrassenSegment().getDatum().getVonKnoten();
		if (vonKnoten != null) {
			final List<AeusseresStrassenSegment> ergebnis = getAeussereStrassenSegmenteVorStrassenKnotenInternal(
					vonKnoten, strasse, entgegengesetzteRichtung);
			if (ergebnis != null) {
				return ergebnis;
			}
		}
		return new ArrayList<>();
	}

	/**
	 * Liefert die Liste der zu der gegebenen Staßen gehörigen Straßenknoten zurück.
	 *
	 * <p>
	 * <b>Achtung:</b> Bei der Straße kann es sich auch lediglich um ein LCL-Segment
	 * einer logischen Straße handeln. Eine logische Straße wäre z.B. die A5 mit den
	 * LCL-Segmenten 'A5 Basel nach Karlsruhe', 'A5 Karlsruhe nach Heidelberg' und
	 * 'A5 Heidelberg nach Darmstadt'.
	 * </p>
	 *
	 * @param strasse  die Staße, nicht <code>null</code>
	 * @param richtung die Richtung der Straße (positiv oder negativ)
	 *
	 * @return die Liste der zu der gegebenen Staßen gehörigen Straßenknoten
	 */
	public List<StrassenKnoten> getStrassenKnoten(final Strasse strasse, final AttTmcRichtung richtung) {
		ensureInit();
		return getStrassenKnotenInternal(strasse, richtung);
	}

	private List<StrassenKnoten> getStrassenKnotenInternal(final Strasse strasse, final AttTmcRichtung richtung) {

		if (strasse == null) {
			throw new IllegalArgumentException("Die übergebene Straße darf nicht null sein.");
		}
		AttTmcRichtung dieRichtung = richtung;

		// sind die Maps (für positive und negative Richtung) noch leer, werden
		// diese zunächst einmalig initialisiert
		if ((str2skMap == null) || (str2assVorSkMap == null) || (str2assNachSk == null)) {

			ordneStrassenSegmenteUndKnoten(new NullMonitor());
		}

		if (null == dieRichtung) {
			dieRichtung = AttTmcRichtung.ZUSTAND_1_POSITIV;
			if (!str2skMap.get(dieRichtung).containsKey(strasse)) {
				dieRichtung = AttTmcRichtung.ZUSTAND_1N_NEGATIV;
			}
		}
		// gibt eine leere Liste zurück, falls keine Straßenknoten für eine
		// gegebene Straße in der gegebenen Richtung gefunden wurden
		final Map<Strasse, List<StrassenKnoten>> map = str2skMap.get(dieRichtung);
		List<StrassenKnoten> ergebnis = map.get(strasse);
		if (ergebnis == null) {
			ergebnis = new ArrayList<>();
			map.put(strasse, ergebnis);
		}
		return ergebnis;
	}

	/**
	 * Liefert die Liste der zu einer gegebenen Staßennummer (logischen Straße)
	 * gehörigen Straßenknoten zurück.
	 *
	 * <p>
	 * <b>Achtung:</b> Eine Straßennummer definiert eine logischen Straße. Eine
	 * logische Straße wäre z.B. die 'A5' mit den LCL-Segmenten 'A5 Basel nach
	 * Karlsruhe', 'A5 Karlsruhe nach Heidelberg' und 'A5 Heidelberg nach
	 * Darmstadt'.
	 * </p>
	 *
	 * @param daten    die Daten einer Staße (eines LCL-Segments), womit auf die
	 *                 Straßennummer geschlossen werden kann, nicht
	 *                 <code>null</code>
	 * @param richtung die Richtung der Straße (positiv oder negativ)
	 *
	 * @return die Liste der zu einer gegebenen Staßennummer (logischen Straße)
	 *         gehörigen Straßenknoten zurück
	 *
	 * @see #getStrassenKnoten(Strasse, AttTmcRichtung)
	 */
	public List<StrassenKnoten> getStrassenKnoten(final Daten daten, final AttTmcRichtung richtung) {
		ensureInit();
		if (daten == null) {
			throw new IllegalArgumentException("Übergebene Straßen-Daten dürfen nicht null sein.");
		}

		final List<StrassenKnoten> ergebnis = new ArrayList<>();
		final String strassenNummer = getStrassenNummer(daten);
		if (strassenNummer == null) {
			return ergebnis;
		}
		final List<Strasse> strassenZurNummer = strNummer2strMap.get(strassenNummer);
		if ((strassenZurNummer == null) || strassenZurNummer.isEmpty()) {
			return ergebnis;
		}

		final Set<StrassenKnoten> ergebnisSet = new HashSet<>();
		for (final Strasse strasse : strassenZurNummer) {
			final List<StrassenKnoten> knoten = getStrassenKnotenInternal(strasse, richtung);
			if (knoten != null) {
				ergebnisSet.addAll(knoten);
			}
		}
		return new ArrayList<>(ergebnisSet);
	}

	/**
	 * Bestimmt, ob eine Richtung fuer eine Strasse im Modell definiert ist.
	 *
	 * @param strasse  die Strasse
	 * @param richtung die Richtung
	 * @return wahr, wenn die Richtung definiert ist.
	 */
	public boolean istRichtungDefiniert(final Strasse strasse, final AttTmcRichtung richtung) {
		ensureInit();
		if ((str2skMap == null) || (str2assVorSkMap == null) || (str2assNachSk == null)) {
			getStrassenKnotenInternal(strasse, richtung);
		}
		return str2skMap.get(richtung).containsKey(strasse);
	}

	/**
	 * Liefert eine Liste der äußeren Straßensegmente zurück, welche <u>direkt</u>
	 * vor dem gegebenen Straßenknoten auf der Straße (bzw. dem LCL-Segment) liegen.
	 * Handelt es sich bei dem gegebenen Straßenknoten um einen Startknoten der
	 * Straße (bzw. des LCL-Segments), ist der Rückgabewert <code>null</code>.
	 *
	 * <p>
	 * <b>Achtung:</b> Bei der Straße kann es sich auch lediglich um ein LCL-Segment
	 * einer logischen Straße handeln. Eine logische Straße wäre z.B. die A5 mit den
	 * LCL-Segmenten 'A5 Basel nach Karlsruhe', 'A5 Karlsruhe nach Heidelberg' und
	 * 'A5 Heidelberg nach Darmstadt'.
	 * </p>
	 *
	 * @param knoten   der Knoten, nicht <code>null</code>
	 * @param strasse  die Strasse, nicht <code>null</code>
	 * @param richtung die Richtung (positiv oder negativ)
	 *
	 * @return Liste der äußeren Straßensegmente, welche <u>direkt</u> vor dem
	 *         gegebenen Straßenknoten auf der Straße (bzw. dem LCL-Segment) liegen,
	 *         kann <code>null</code> sein (gegebener Straßenknoten ist Startknoten
	 *         der Straße)
	 */
	public List<AeusseresStrassenSegment> getAeussereStrassenSegmenteVorStrassenKnoten(final StrassenKnoten knoten,
			final Strasse strasse, final AttTmcRichtung richtung) {
		ensureInit();
		return getAeussereStrassenSegmenteVorStrassenKnotenInternal(knoten, strasse, richtung);
	}

	/**
	 * Liefert eine Liste der äußeren Straßensegmente zurück, welche <u>direkt</u>
	 * vor dem gegebenen Straßenknoten auf einer gegebenen logischen Straße
	 * (Staßennummer) liegen. handelt es sich bei dem gegebenen Straßenknoten um
	 * einen Startknoten der logischen Straße, ist der Rückgabewert
	 * <code>null</code>.
	 *
	 * <p>
	 * <b>Achtung:</b> Eine Straßennummer definiert eine logischen Straße. Eine
	 * logische Straße wäre z.B. die 'A5' mit den LCL-Segmenten 'A5 Basel nach
	 * Karlsruhe', 'A5 Karlsruhe nach Heidelberg' und 'A5 Heidelberg nach
	 * Darmstadt'.
	 * </p>
	 *
	 * @param knoten   der Knoten, nicht <code>null</code>
	 * @param daten    die Daten einer Staße (eines LCL-Segments), womit auf die
	 *                 Straßennummer geschlossen werden kann, nicht
	 *                 <code>null</code>
	 * @param richtung die Richtung (positiv oder negativ)
	 *
	 * @return Liste der äußeren Straßensegmente zurück, welche <u>direkt</u> vor
	 *         dem gegebenen Straßenknoten auf einer gegebenen logischen Straße
	 *         (Staßennummer) liegen, kann <code>null</code> sein (gegebener
	 *         Straßenknoten ist Startknoten der logischen Straße)
	 *
	 * @see #getAeussereStrassenSegmenteVorStrassenKnoten(StrassenKnoten, Strasse,
	 *      AttTmcRichtung)
	 */
	public List<AeusseresStrassenSegment> getAeussereStrassenSegmenteVorStrassenKnoten(final StrassenKnoten knoten,
			final Daten daten, final AttTmcRichtung richtung) {
		ensureInit();
		if (daten == null) {
			throw new IllegalArgumentException("Übergebene Straßen-Daten dürfen nicht null sein.");
		}

		final List<AeusseresStrassenSegment> ergebnis = new ArrayList<>();
		final String strassenNummer = getStrassenNummer(daten);
		if (strassenNummer == null) {
			return ergebnis;
		}
		final List<Strasse> strassenZurNummer = strNummer2strMap.get(strassenNummer);
		if ((strassenZurNummer == null) || strassenZurNummer.isEmpty()) {
			return ergebnis;
		}

		for (final Strasse strasse : strassenZurNummer) {
			final List<AeusseresStrassenSegment> segmente = getAeussereStrassenSegmenteVorStrassenKnotenInternal(knoten,
					strasse, richtung);
			if (segmente != null) {
				ergebnis.addAll(segmente);
			}
		}
		return ergebnis;
	}

	/**
	 * Liefert eine Liste der äußeren Straßensegmente zurück, welche <u>direkt</u>
	 * nach dem gegebenen Straßenknoten auf der Straße (bzw. dem LCL-Segment)
	 * liegen. Handelt es sich bei dem gegebenen Straßenknoten um einen Endknoten
	 * der Straße (bzw. des LCL-Segments), ist der Rückgabewert <code>null</code>.
	 *
	 * <p>
	 * <b>Achtung:</b> Bei der Straße kann es sich auch lediglich um ein LCL-Segment
	 * einer logischen Straße handeln. Eine logische Straße wäre z.B. die A5 mit den
	 * LCL-Segmenten 'A5 Basel nach Karlsruhe', 'A5 Karlsruhe nach Heidelberg' und
	 * 'A5 Heidelberg nach Darmstadt'.
	 * </p>
	 *
	 * @param knoten   der Knoten, nicht <code>null</code>
	 * @param strasse  die Strasse, nicht <code>null</code>
	 * @param richtung die Richtung (positiv oder negativ)
	 *
	 * @return Liste der äußeren Straßensegmente, welche <u>direkt</u> nach dem
	 *         gegebenen Straßenknoten auf der Straße (bzw. dem LCL-Segment) liegen,
	 *         kann <code>null</code> sein (gegebener Straßenknoten ist Endknoten
	 *         der Straße)
	 */
	public List<AeusseresStrassenSegment> getAeussereStrassenSegmenteNachStrassenKnoten(final StrassenKnoten knoten,
			final Strasse strasse, final AttTmcRichtung richtung) {
		ensureInit();
		return getAeussereStrassenSegmenteNachStrassenKnotenInternal(knoten, strasse, richtung);
	}

	/**
	 * Liefert eine Liste der äußeren Straßensegmente zurück, welche <u>direkt</u>
	 * nach dem gegebenen Straßenknoten auf einer gegebenen logischen Straße
	 * (Staßennummer) liegen. Handelt es sich bei dem gegebenen Straßenknoten um
	 * einen Endknoten der logischen Straße, ist Rückgabewert <code>null</code>.
	 *
	 * <p>
	 * <b>Achtung:</b> Eine Straßennummer definiert eine logischen Straße. Eine
	 * logische Straße wäre z.B. die 'A5' mit den LCL-Segmenten 'A5 Basel nach
	 * Karlsruhe', 'A5 Karlsruhe nach Heidelberg' und 'A5 Heidelberg nach
	 * Darmstadt'.
	 * </p>
	 *
	 * @param knoten   der Knoten, nicht <code>null</code>
	 * @param daten    die Daten einer Staße (eines LCL-Segments), womit auf die
	 *                 Straßennummer geschlossen werden kann, nicht
	 *                 <code>null</code>
	 * @param richtung die Richtung (positiv oder negativ)
	 *
	 * @return Liste der äußeren Straßensegmente, welche nach dem gegebenen
	 *         Straßenknoten liegen, kann <code>null</code> sein (gegebener
	 *         Straßenknoten ist Endknoten der logischen Straße)
	 *
	 * @see #getAeussereStrassenSegmenteNachStrassenKnoten(StrassenKnoten, Strasse,
	 *      AttTmcRichtung)
	 */
	public List<AeusseresStrassenSegment> getAeussereStrassenSegmenteNachStrassenKnoten(final StrassenKnoten knoten,
			final Daten daten, final AttTmcRichtung richtung) {
		ensureInit();

		ensureInit();
		if (daten == null) {
			throw new IllegalArgumentException("Übergebene Straßen-Daten dürfen nicht null sein.");
		}

		final List<AeusseresStrassenSegment> ergebnis = new ArrayList<>();
		final String strassenNummer = getStrassenNummer(daten);
		if (strassenNummer == null) {
			return ergebnis;
		}
		final List<Strasse> strassenZurNummer = strNummer2strMap.get(strassenNummer);
		if ((strassenZurNummer == null) || strassenZurNummer.isEmpty()) {
			return ergebnis;
		}

		for (final Strasse strasse : strassenZurNummer) {
			final List<AeusseresStrassenSegment> segmente = getAeussereStrassenSegmenteNachStrassenKnotenInternal(
					knoten, strasse, richtung);
			if (segmente != null) {
				ergebnis.addAll(segmente);
			}
		}
		return ergebnis;
	}

	/**
	 * Liefert eine Liste von äußeren Straßensegmente zurück, an welchen der
	 * gegebenen Straßenknoten direkt liegt (also entweder Von- oder
	 * Nach-Straßenknoten ist).
	 *
	 * @param knoten der Knoten, nicht <code>null</code>
	 *
	 * @return eine Liste von äußeren Straßensegmente zurück, an welchen der
	 *         gegebenen Straßenknoten direkt liegt, niemals <code>null</code>
	 */
	public List<AeusseresStrassenSegment> getAeussereStrassenSegmente(final StrassenKnoten knoten) {
		ensureInit();
		if (knoten == null) {
			throw new IllegalArgumentException("Übergebener Straßenknoten darf nicht null sein.");
		}

		final Set<AeusseresStrassenSegment> ergebnis = new HashSet<>();
		for (final AttTmcRichtung r : AttTmcRichtung.getZustaende()) {
			for (final Strasse str : getStrassenInternal()) {
				final List<AeusseresStrassenSegment> segmente1 = getAeussereStrassenSegmenteNachStrassenKnotenInternal(
						knoten, str, r);
				if ((segmente1 != null) && !segmente1.isEmpty()) {
					ergebnis.addAll(segmente1);
				}
				final List<AeusseresStrassenSegment> segmente2 = getAeussereStrassenSegmenteVorStrassenKnotenInternal(
						knoten, str, r);
				if ((segmente2 != null) && !segmente2.isEmpty()) {
					ergebnis.addAll(segmente2);
				}
			}
		}
		return new ArrayList<>(ergebnis);
	}

	private List<AeusseresStrassenSegment> getAeussereStrassenSegmenteVorStrassenKnotenInternal(
			final StrassenKnoten knoten, final Strasse strasse, final AttTmcRichtung richtung) {
		if (knoten == null) {
			throw new IllegalArgumentException("Übergebener Straßenknoten darf nicht null sein.");
		}
		if (strasse == null) {
			throw new IllegalArgumentException("Übergebene Straße darf nicht null sein.");
		}
		return str2assVorSkMap.get(richtung).get(strasse).get(knoten);
	}

	private List<AeusseresStrassenSegment> getAeussereStrassenSegmenteNachStrassenKnotenInternal(
			final StrassenKnoten knoten, final Strasse strasse, final AttTmcRichtung richtung) {
		if (knoten == null) {
			throw new IllegalArgumentException("Übergebener Straßenknoten darf nicht null sein.");
		}
		if (strasse == null) {
			throw new IllegalArgumentException("Übergebene Straße darf nicht null sein.");
		}
		return str2assNachSk.get(richtung).get(strasse).get(knoten);
	}

	/**
	 * Gibt die inneren Straßensegmente eines Straßenknotens zurück, dessen
	 * vonSegment das gegebene äußere Straßensegment ist.
	 *
	 * @param sk  ein Straßenknoten.
	 * @param ass das äußere Straßensegment, darf nicht <code>null</code> sein
	 *
	 * @return die inneren Straßensegmente oder eine leere Liste, wenn der
	 *         Straßenknoten keine Straßensegmente hat oder unbekannt ist.
	 */
	public List<InneresStrassenSegment> getInneresStrassenSegmente(final StrassenKnoten sk,
			final AeusseresStrassenSegment ass) {

		if (sk == null) {
			throw new IllegalArgumentException("Übergebener Straßenknoten darf nicht null sein.");
		}
		if (ass == null) {
			throw new IllegalArgumentException("Übergebenes äußeres Straßensegment darf nicht null sein.");
		}

		// (1) Ermittel die Straße bzw. Straßennummer des gegebene äußeren
		// Straßensegments
		List<InneresStrassenSegment> ergebnis = new ArrayList<>();
		final Strasse assStrasse = ass.getKdStrassenSegment().getDatum().getGehoertZuStrasse();
		if (assStrasse == null) {
			return ergebnis;
		}
		final String assStrassenNr = getStrassenNummer(assStrasse);
		if (assStrassenNr == null) {
			return ergebnis;
		}

		// (2) Setze Ergebnis-Liste zunächst auf sämtliche inneren
		// Straßensegmente des gegebenen Straßenknotens
		ergebnis = new ArrayList<>(netzCache.getInneresStrassenSegmente(sk));

		// (3) Filtere alle inneren Straßensegmente, die nicht auf derselben
		// Straße, wie das gegebene äußere Straßensegment, liegen bzw. deren
		// vonSegment nicht dem gegebenen äußeren Straßensegment entspricht
		for (final Iterator<InneresStrassenSegment> iter = ergebnis.iterator(); iter.hasNext();) {
			final InneresStrassenSegment iss = iter.next();
			final AeusseresStrassenSegment vonSegment = iss.getKdInneresStrassenSegment().getDatum()
					.getVonStrassenSegment();
			if (vonSegment != null) {
				final Strasse issStrasse = iss.getKdStrassenSegment().getDatum().getGehoertZuStrasse();
				if (issStrasse != null) {
					final String issStrassenNr = getStrassenNummer(issStrasse);
					if ((issStrassenNr != null) && issStrassenNr.equals(assStrassenNr)
							&& vonSegment.getPid().equals(ass.getPid())) {
						continue;
					}
				}
			}
			iter.remove();
		}
		return ergebnis;
	}

	/**
	 * Liefert das zugrundeliegende Verkehrsmodellnetz zurück.
	 *
	 * @return das zugrundeliegende Verkehrsmodellnetz
	 */
	public VerkehrsModellNetz getNetz() {
		return netzCache.getNetz();
	}

	/**
	 * Liefert für ein gegebenes inneres Straßensegment die abhängig von dessen
	 * äußeren Von- und Nach-Straßensegment bestimmte Richtung zurück.
	 *
	 * <p>
	 * Die bestimmte Richtung entspricht (sofern vorhanden) der Richtung des
	 * Nach-Segments des inneren Straßensegments. Existiert für das gegebene innere
	 * Straßensegment kein nachfolgendes äußeres Straßensegment, so wird die
	 * Richtung des Von-Segements verwendet. Existiert auch dieses nicht, so liefert
	 * diese Methode <code>null</code> zurück.
	 * </p>
	 *
	 * @param segment das innere Straßensegment, darf nicht <code>null</code> sein
	 *
	 * @return die abhängig vom äußeren Von- und Nach-Straßensegment bestimmte
	 *         Richtung, kann <code>null</code> sein
	 */
	public AttTmcRichtung getRichtung(final InneresStrassenSegment segment) {
		if (segment == null) {
			throw new IllegalArgumentException("Übergebenes inneres Straßensegment darf nicht null sein.");
		}

		final de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdInneresStrassenSegment.Daten datum = segment
				.getKdInneresStrassenSegment().getDatum();
		if (datum != null) {
			AeusseresStrassenSegment ass = datum.getNachStrassenSegment();
			if (ass == null) {
				ass = datum.getVonStrassenSegment();
			}
			if (ass != null) {
				return ass.getKdAeusseresStrassenSegment().getDatum().getTmcRichtung();
			}
		}
		return null;
	}

	/**
	 * Liefert die Straße zurück, auf welcher das gegebenes inneres Straßensegment
	 * liegt. Existiert für das gegebene innere Straßensegment keine
	 * Straßeninformation, so wird die Straße abhängig von den äußeren Von- und
	 * Nach-Straßensegment bestimmt.
	 *
	 * <p>
	 * Die bestimmte Straße entspricht dann (sofern vorhanden) der Straße des
	 * Nach-Segments des inneren Straßensegments. Existiert für das gegebene innere
	 * Straßensegment kein nachfolgendes äußeres Straßensegment, so wird die
	 * Straßeninformation des Von-Segements verwendet. Existiert auch dieses nicht,
	 * so liefert diese Methode <code>null</code> zurück.
	 * </p>
	 *
	 * @param segment das innere Straßensegment, darf nicht <code>null</code> sein
	 *
	 * @return die Straße, auf welcher das gegebenes inneres Straßensegment liegt,
	 *         kann <code>null</code> sein
	 */
	public Strasse getStrasse(final InneresStrassenSegment segment) {
		if (segment == null) {
			throw new IllegalArgumentException("Übergebenes inneres Straßensegment darf nicht null sein.");
		}

		final Strasse strasse = segment.getKdStrassenSegment().getDatum().getGehoertZuStrasse();
		if (strasse != null) {
			return strasse;
		}

		final de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdInneresStrassenSegment.Daten datum = segment
				.getKdInneresStrassenSegment().getDatum();
		if (datum != null) {
			AeusseresStrassenSegment ass = datum.getNachStrassenSegment();
			if (ass == null) {
				ass = datum.getVonStrassenSegment();
			}
			if (ass != null) {
				return ass.getKdStrassenSegment().getDatum().getGehoertZuStrasse();
			}
		}
		return null;
	}
}
