/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.InvalidationListener;
import de.bsvrz.sys.funclib.bitctrl.modell.AenderbareMenge;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.MengenEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.MengenListener;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.OnlineDatum;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.objekte.LandesMeldeStelle;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.objekte.RdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.parameter.PdRdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.parameter.PdRdsMeldung.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.RdsMeldungWrapper;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.RdsMeldungenChangedListener;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * <p>
 * Diese Hilfsklasse verwaltet sämtliche für dieses Plugin benötigten
 * Informationen bzgl. RDS-Meldungen.
 * </p>
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 */
public class RdsMeldungenCache extends AbstractCache
		implements MengenListener, DatensatzUpdateListener, InvalidationListener {

	/**
	 * Das Lock-Objekt.
	 */
	private static Object lock = new Object();

	/**
	 * Flag, welches angibt, ob der MengeListener bereits hinzugefügt wurde.
	 */
	private boolean mengenListenerHinzugefuegt;

	/**
	 * Flag, welches angibt, ob auf DatensatzUpdateListener reagiert werden soll.
	 */
	private boolean listenerAktiv = true;

	/**
	 * Liste von Listener, welche auf Änderungen von RDS-Meldungen warten.
	 */
	private List<RdsMeldungenChangedListener> listeners;

	/**
	 * Map, welche den entsprechenden RdsMeldungWrapper zu einer RdsMeldung
	 * speichert.
	 */
	private Map<RdsMeldung, RdsMeldungWrapper> rdsMeldungenWrapperMap;

	/**
	 * Der TMC-LocationCode-Cache, von welchem dieser Cache abhängig ist.
	 */
	private TmcLocationCodeCache tmcLocationCodeCache;

	/**
	 * Der RDS-Cache, von welchem dieser Cache abhängig ist.
	 */
	private RdsCache rdsCache;

	/**
	 * Der Netz-Cache, von welchem dieser Cache implizit abhängig ist.
	 */
	private NetzCache netzCache;

	/**
	 * Der ModellObjectCache, von welchem dieser Cache abhängig ist.
	 */
	private ModellObjektCache moCache;

	/**
	 * Der AbstractDavVerbindungsCache, von welchem dieser Cache abhängig ist.
	 */
	private AbstractDavVerbindungsCache verbindungsCache;

	/**
	 * Die zugrundeliegende Landesmeldestelle.
	 */
	private LandesMeldeStelle landesMeldeStelle;

	/**
	 * ThreadPool, der die Threads zur Bearbeitung der Ergebnisse der
	 * Parameteranmeldung bereit stellt.
	 */
	private final ExecutorService threadPool = Executors.newFixedThreadPool(30);

	/**
	 * Der Konstruktor.
	 *
	 * @param moc                  der ModellObject-Cache
	 * @param verbindungsCache     der Verbindungs-Cache
	 * @param rdsCache             der RDS-Cache, von welchem dieser Cache abhängig
	 *                             ist
	 * @param tmcLocationCodeCache der TmcLocationCode-Cache, von welchem dieser
	 *                             Cache abhängig ist
	 */
	public RdsMeldungenCache(final ModellObjektCache moc, final AbstractDavVerbindungsCache verbindungsCache,
			final RdsCache rdsCache, final TmcLocationCodeCache tmcLocationCodeCache) {
		super(tmcLocationCodeCache.getName(), moc, verbindungsCache, rdsCache, tmcLocationCodeCache);
	}

	/**
	 * Initialisiert den Cache.
	 *
	 * @param mocache              der Modell-Object-Cache mit der Landesmeldestelle
	 * @param verbindungscache     der Verbindungs-Cache
	 * @param rdscache             der RDS-Cache
	 * @param tmcLocationCodecache der TmcLocationCodeCache
	 */
	protected void preInit(final ModellObjektCache mocache, final AbstractDavVerbindungsCache verbindungscache,
			final RdsCache rdscache, final TmcLocationCodeCache tmcLocationCodecache) {
		moCache = mocache;
		verbindungsCache = verbindungscache;
		rdsCache = rdscache;
		tmcLocationCodeCache = tmcLocationCodecache;
	}

	/**
	 * Liefert den RDS-Cache zurück, von welchem dieser Cache abhängig ist.
	 *
	 * @return den RDS-Cache, von welchem dieser Cache abhängig ist
	 */
	public RdsCache getRdsCache() {
		return rdsCache;
	}

	/**
	 * Liefert den TmcLocationCodeCache-Cache zurück, von welchem dieser Cache
	 * abhängig ist.
	 *
	 * @return den TmcLocationCodeCache-Cache, von welchem dieser Cache abhängig ist
	 */
	public TmcLocationCodeCache getTmcLocationCodeCache() {
		return tmcLocationCodeCache;
	}

	/**
	 * Liefert den Netz-Cache zurück, von welchem dieser Cache abhängig ist.
	 *
	 * @return den Netz-Cache, von welchem dieser Cache abhängig ist
	 */
	public NetzCache getNetzCache() {
		return netzCache;
	}

	@Override
	protected boolean doInit(final IMonitor monitor) {

		landesMeldeStelle = (LandesMeldeStelle) moCache.getModellObjekt();

		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		final int work = 2000;
		subMonitor.beginTask("Lade RDS-Meldungen", work);

		if (erzeugeRdsMeldungenWrapper(subMonitor.newChild(work))) {
			registerListener();
			return true;
		}
		return false;
	}

	/**
	 * Fügt eine RdsMeldungenChangedListener der RDS-Meldungen-Verwaltung hinzu.
	 *
	 * @param listener der RdsMeldungenChangedListener, nicht <code>null</code>
	 */
	public void addRdsMeldungenChangedListener(final RdsMeldungenChangedListener listener) {
		if (listener == null) {
			throw new IllegalArgumentException("Übergebener Listener darf nicht null sein.");
		}
		if (listeners == null) {
			listeners = new ArrayList<>();
		}
		if (!listeners.contains(listener)) {
			listeners.add(listener);
		}
	}

	/**
	 * Entfernt den gegebenen RdsMeldungenChangedListener von der
	 * RDS-Meldungen-Verwaltung.
	 *
	 * @param listener der RdsMeldungenChangedListener, nicht <code>null</code>
	 */
	public void removeRdsMeldungenChangedListener(final RdsMeldungenChangedListener listener) {
		if (listeners != null) {
			listeners.remove(listener);
			if (listeners.isEmpty()) {
				listeners = null;
			}
		}
	}

	/**
	 * Registriert die Update- und MengenListener für RdsMeldungen.
	 */
	private void registerListener() {

		// Füge Update- und MengenListener hinzu
		listenerAktiv = false;
		final AenderbareMenge<RdsMeldung> rdsMeldungen = landesMeldeStelle.getRDSMeldungen();
		if (rdsMeldungen != null) {
			if (!mengenListenerHinzugefuegt) {
				rdsMeldungen.addMengenListener(this);
				mengenListenerHinzugefuegt = true;
			}
			for (final RdsMeldung rdsMeldung : rdsMeldungen) {
				addUpdateListener(rdsMeldung);
				addInvalidationListener(rdsMeldung);
			}
		}
		listenerAktiv = true;
	}

	/**
	 * Fügt der gegebenen RDS-Meldung sämtliche UpdateListener hinzu.
	 *
	 * @param rdsMeldung die RDS-Meldung
	 */
	private void addUpdateListener(final RdsMeldung rdsMeldung) {
		rdsMeldung.getPdRdsMeldung().addUpdateListener(PdRdsMeldung.Aspekte.RdsGeneriert, this);
		rdsMeldung.getPdRdsMeldung().addUpdateListener(PdRdsMeldung.Aspekte.RdsSenden, this);
		rdsMeldung.getPdRdsMeldung().addUpdateListener(PdRdsMeldung.Aspekte.RdsVersendet, this);
		rdsMeldung.getPdRdsMeldung().addUpdateListener(PdRdsMeldung.Aspekte.RdsEmpfangen, this);
	}

	/**
	 * Entfernt von der gegebenen RDS-Meldung sämtliche UpdateListener.
	 *
	 * @param rdsMeldung die RDS-Meldung
	 */
	private void removeUpdateListener(final RdsMeldung rdsMeldung) {
		rdsMeldung.getPdRdsMeldung().removeUpdateListener(PdRdsMeldung.Aspekte.RdsGeneriert, this);
		rdsMeldung.getPdRdsMeldung().removeUpdateListener(PdRdsMeldung.Aspekte.RdsSenden, this);
		rdsMeldung.getPdRdsMeldung().removeUpdateListener(PdRdsMeldung.Aspekte.RdsVersendet, this);
		rdsMeldung.getPdRdsMeldung().removeUpdateListener(PdRdsMeldung.Aspekte.RdsEmpfangen, this);
	}

	/**
	 * Fügt der gegebenen RDS-Meldung einen InvalidationListener hinzu.
	 *
	 * @param rdsMeldung die RDS-Meldung
	 */
	private void addInvalidationListener(final RdsMeldung rdsMeldung) {
		((DynamicObject) rdsMeldung.getSystemObject()).addListenerForInvalidation(this);
	}

	/**
	 * Entfernt von der gegebenen RDS-Meldung einen InvalidationListener.
	 *
	 * @param rdsMeldung die RDS-Meldung
	 */
	private void removeInvalidationListener(final RdsMeldung rdsMeldung) {
		((DynamicObject) rdsMeldung.getSystemObject()).removeListenerForInvalidation(this);
	}

	/**
	 * Erzeugt sämtliche RDSMeldungenWrapper.
	 *
	 * @param monitor der Monitor zur Visualisierung des Fortschritts
	 *
	 * @return <code>true</code>, wenn die Methode erfolgreich ausgeführt werden
	 *         konnte, ansonsten <code>false</code>
	 */
	private boolean erzeugeRdsMeldungenWrapper(final IMonitor monitor) {

		synchronized (lock) {

			if (rdsMeldungenWrapperMap == null) {
				rdsMeldungenWrapperMap = new HashMap<>();

				final int totalWork = 1000000;
				final String taskName = "Lade RDS-Meldungen";
				monitor.setTaskName(taskName);
				monitor.beginTask(taskName, totalWork);

				final AenderbareMenge<RdsMeldung> rdsMeldungen = landesMeldeStelle.getRDSMeldungen();
				final int anzahlRdsMeldungen = rdsMeldungen.size();
				if (anzahlRdsMeldungen == 0) {
					return true;
				}
				final int work = totalWork / anzahlRdsMeldungen;

				int i = 0;
				for (final RdsMeldung rdsMeldung : rdsMeldungen) {
					try {
						monitor.subTask(new StringBuffer("Initialisiere RDS-Meldung ").append(i).append(" von ")
								.append(anzahlRdsMeldungen).toString());

						final RdsMeldungWrapper wrapper = erzeugeRdsMeldungWrapper(rdsMeldung);
						if (wrapper != null) {
							rdsMeldungenWrapperMap.put(rdsMeldung, wrapper);
						}

					} catch (final RuntimeException e) {
						if (rdsMeldungenWrapperMap.get(rdsMeldung) != null) {
							rdsMeldungenWrapperMap.remove(rdsMeldung);
						}
						Debug.getLogger().warning("Es gab Probleme beim Laden einer RDS-Meldung.", e);
					} finally {
						monitor.worked(work);
						if (monitor.isCanceled()) {
							monitor.done();
							return false;
						}
						i++;
					}
				}
			}
		}
		return true;
	}

	/**
	 * Erzeugt den RDSMeldungenWrapper für das gegebene Systemobjekt.
	 *
	 * @param rdsMeldung das Systemobjekt
	 *
	 * @return den RDSMeldungenWrapper
	 */
	private RdsMeldungWrapper erzeugeRdsMeldungWrapper(final RdsMeldung rdsMeldung) {

		if (!rdsMeldung.getSystemObject().isValid()) {
			Debug.getLogger().warning("Ungültige (Invalide) RDS-Meldung in Menge: " + rdsMeldung.getPid());
			return null;
		}

		// (1) Ermittel zunächst sämtliche Daten die über die unterstützten
		// Aspekte empfangen werden
		final PdRdsMeldung pdRdsMeldung = rdsMeldung.getPdRdsMeldung();
		final Daten datumEmpfangen = pdRdsMeldung.getDatum(PdRdsMeldung.Aspekte.RdsEmpfangen);
		final Daten datumVersendet = pdRdsMeldung.getDatum(PdRdsMeldung.Aspekte.RdsVersendet);
		final Daten datumSenden = pdRdsMeldung.getDatum(PdRdsMeldung.Aspekte.RdsSenden);
		final Daten datumGeneriert = pdRdsMeldung.getDatum(PdRdsMeldung.Aspekte.RdsGeneriert);

		// (2) Analysiere und bewerte die Daten und ermittel das Datum mit der
		// höchsten Wertigkeit. Die höchste Wertigkeit hat das Datum mit der
		// höchsten Versionsnummer. Haben Daten dieselbe Versionsnummer so wird
		// ein "empfangenes" Datum, einem "versendetem", einem "gesendetem" und
		// dieses wiederum einem "generierten" Datum vorgezogen.
		int bewertung = -1;
		Daten datum = null;
		if ((datumEmpfangen != null) && (datumEmpfangen.dGetDatenStatus() == OnlineDatum.Status.DATEN)) {

			bewertung = (datumEmpfangen.getVersion().getVerwaltungsInformationen().getVersionsNummer().getValue() * 10)
					+ 4;
			datum = datumEmpfangen;
		}
		if ((datumVersendet != null) && (datumVersendet.dGetDatenStatus() == OnlineDatum.Status.DATEN)) {

			final int aktuelleBewertung = (datumVersendet.getVersion().getVerwaltungsInformationen().getVersionsNummer()
					.getValue() * 10) + 3;
			if (aktuelleBewertung > bewertung) {
				bewertung = aktuelleBewertung;
				datum = datumVersendet;
			}
		}
		if ((datumSenden != null) && (datumSenden.dGetDatenStatus() == OnlineDatum.Status.DATEN)) {

			final int aktuelleBewertung = (datumSenden.getVersion().getVerwaltungsInformationen().getVersionsNummer()
					.getValue() * 10) + 2;
			if (aktuelleBewertung > bewertung) {
				bewertung = aktuelleBewertung;
				datum = datumSenden;
			}
		}
		if ((datumGeneriert != null) && (datumGeneriert.dGetDatenStatus() == OnlineDatum.Status.DATEN)) {

			final int aktuelleBewertung = (datumGeneriert.getVersion().getVerwaltungsInformationen().getVersionsNummer()
					.getValue() * 10) + 1;
			if (aktuelleBewertung > bewertung) {
				bewertung = aktuelleBewertung;
				datum = datumGeneriert;
			}
		}

		// (3) Erstelle einen RdsMeldungWrapper von dem höchstwertigen Datum.
		if (datum != null) {
			return new RdsMeldungWrapper(this, rdsMeldung, datum);
		}
		Debug.getLogger().warning("RDS-Meldung enthält unter keinem Aspekt Nutzdaten: " + rdsMeldung.getPid());
		return null;
	}

	/**
	 * Liefert eine Collection sämtlicher RDS-Meldungen zurück.
	 *
	 * @return eine Collection sämtlicher RDS-Meldungen zurück, niemals
	 *         <code>null</code>
	 */
	public Collection<RdsMeldung> getRdsMeldungen() {
		ensureInit();
		if (rdsMeldungenWrapperMap != null) {
			return rdsMeldungenWrapperMap.keySet();
		}
		return new ArrayList<>();
	}

	/**
	 * Liefert eine Collection sämtlicher RdsMeldungWrapper zurück.
	 *
	 * @return eine Collection sämtlicher RdsMeldungWrapper zurück, niemals
	 *         <code>null</code>
	 */
	public Collection<RdsMeldungWrapper> getRdsMeldungenWrapper() {
		ensureInit();
		if (rdsMeldungenWrapperMap != null) {
			return rdsMeldungenWrapperMap.values();
		}
		return new ArrayList<>();
	}

	/**
	 * Liefert den zu der RdsMeldung den passenden RdsMeldungWrapper zurück.
	 *
	 * @param rdsMeldung die RdsMeldung, darf nicht <code>null</code> sein
	 *
	 * @return den zu der RdsMeldung den passenden RdsMeldungWrapper zurück
	 */
	public RdsMeldungWrapper getRdsMeldungWrapper(final RdsMeldung rdsMeldung) {
		ensureInit();
		return getRdsMeldungWrapperInternal(rdsMeldung);
	}

	/**
	 * Liefert den zu der RdsMeldung den passenden RdsMeldungWrapper zurück.
	 *
	 * @param rdsMeldung die RdsMeldung, darf nicht <code>null</code> sein
	 *
	 * @return den zu der RdsMeldung den passenden RdsMeldungWrapper zurück
	 */
	private RdsMeldungWrapper getRdsMeldungWrapperInternal(final RdsMeldung rdsMeldung) {
		if (rdsMeldung == null) {
			throw new IllegalArgumentException("Übergebene RDS-Meldung darf nicht null sein.");
		}
		return rdsMeldungenWrapperMap.get(rdsMeldung);
	}

	/**
	 * Entfernt sämtliche registrierten Listener sowie sämtliche gecachten
	 * Referenzen auf RDS-Meldungen.
	 *
	 * @param entferneGlobaleListener true, wenn auch sämtliche globalen Listener
	 *                                entfernt werden sollen
	 */
	public void dispose(final boolean entferneGlobaleListener) {

		// Entferne UpdateListener
		if (landesMeldeStelle != null) {
			final AenderbareMenge<RdsMeldung> rdsMeldungen = landesMeldeStelle.getRDSMeldungen();
			if (rdsMeldungen != null) {
				for (final RdsMeldung rdsMeldung : rdsMeldungen) {
					removeUpdateListener(rdsMeldung);
					removeInvalidationListener(rdsMeldung);
				}
			}

			// Entferne MengenListener
			if ((rdsMeldungen != null) && entferneGlobaleListener) {
				rdsMeldungen.removeMengenListener(this);
			}
		}
		synchronized (lock) {
			rdsMeldungenWrapperMap = null;
		}
	}

	@Override
	public void mengeAktualisiert(final MengenEvent e) {
		final Collection<? extends SystemObjekt> entfernt = e.getEntfernt();
		final Collection<? extends SystemObjekt> hinzugefuegt = e.getHinzugefuegt();
		threadPool.execute(new MengeAktualisiertRunner(entfernt, hinzugefuegt));
	}

	/**
	 * Runnable zur Behandlung einer Mengenänderung.
	 *
	 * @author BitCtrl Systems GmbH, Andreas Meissner
	 * @version $Id: RdsMeldungenCache.java 25379 2010-08-05 13:02:03Z meissner $
	 */
	private class MengeAktualisiertRunner implements Runnable {

		/** Die entfernten Meldungen. */
		private final Collection<? extends SystemObjekt> entfernt;

		/** Die neu hinzugefügten Meldungen. */
		private final Collection<? extends SystemObjekt> hinzugefuegt;

		/**
		 * Der Konstruktor.
		 *
		 * @param entfernt     die entfernten Meldungen
		 * @param hinzugefuegt die neu hinzugefügten Meldungen
		 */
		public MengeAktualisiertRunner(final Collection<? extends SystemObjekt> entfernt,
				final Collection<? extends SystemObjekt> hinzugefuegt) {
			this.entfernt = entfernt;
			this.hinzugefuegt = hinzugefuegt;
		}

		@Override
		public void run() {

			final List<RdsMeldungWrapper> entfernteRdsMeldungen = new ArrayList<>();
			final List<RdsMeldungWrapper> hinzugefuegteRdsMeldungen = new ArrayList<>();

			synchronized (lock) {

				// Bearbeite entfernte RDS-Meldungen
				for (final SystemObjekt obj : entfernt) {

					if (obj instanceof RdsMeldung) {
						final RdsMeldung rdsMeldung = (RdsMeldung) obj;
						removeUpdateListener(rdsMeldung);
						removeInvalidationListener(rdsMeldung);

						final RdsMeldungWrapper wrapper = getRdsMeldungWrapperInternal(rdsMeldung);
						if ((wrapper != null) && !entfernteRdsMeldungen.contains(wrapper)) {
							entfernteRdsMeldungen.add(wrapper);
							rdsMeldungenWrapperMap.remove(rdsMeldung);
						}
					}
				}

				// Bearbeite hinzugefügte RDS-Meldungen
				for (final SystemObjekt obj : hinzugefuegt) {

					if (obj instanceof RdsMeldung) {

						try {

							final RdsMeldung rdsMeldung = (RdsMeldung) obj;
							addUpdateListener(rdsMeldung);
							addInvalidationListener(rdsMeldung);

							final RdsMeldungWrapper wrapper = erzeugeRdsMeldungWrapper(rdsMeldung);
							if ((wrapper != null) && !rdsMeldungenWrapperMap.containsKey(rdsMeldung)
									&& !hinzugefuegteRdsMeldungen.contains(wrapper)) {
								hinzugefuegteRdsMeldungen.add(wrapper);
								rdsMeldungenWrapperMap.put(rdsMeldung, wrapper);
							}

						} catch (final RuntimeException e) {
							Debug.getLogger().warning("Es gab Probleme beim Laden einer neu hinzugefügten RDS-Meldung.",
									e);
						}
					}
				}
			}

			// Benachrichtige Listener
			if ((listeners != null) && !listeners.isEmpty()
					&& (!entfernteRdsMeldungen.isEmpty() || !hinzugefuegteRdsMeldungen.isEmpty())) {
				final List<RdsMeldungenChangedListener> tempList = new ArrayList<>(listeners);
				for (final RdsMeldungenChangedListener listener : tempList) {
					if (!entfernteRdsMeldungen.isEmpty()) {
						listener.rdsMeldungenRemoved(entfernteRdsMeldungen);
					}
					if (!hinzugefuegteRdsMeldungen.isEmpty()) {
						listener.rdsMeldungenAdded(hinzugefuegteRdsMeldungen);
					}
				}
			}
		}
	}

	@Override
	public void datensatzAktualisiert(final DatensatzUpdateEvent event) {
		if (!listenerAktiv) {
			return;
		}

		final SystemObjekt systemObjekt = event.getObjekt();
		if ((systemObjekt != null) && event.getDatum().dContainsDaten()) {
			threadPool.execute(new DatensatzAktualisiertRunner(systemObjekt));
		}
	}

	/**
	 * Runnable zur Behandlung einer Datensatzänderung.
	 *
	 * @author BitCtrl Systems GmbH, Andreas Meissner
	 * @version $Id: RdsMeldungenCache.java 25379 2010-08-05 13:02:03Z meissner $
	 */
	private class DatensatzAktualisiertRunner implements Runnable {

		/** Das Systemobjekt dessen Daten sich geändert haben. */
		private final SystemObjekt systemObjekt;

		/**
		 * Der Konstruktor.
		 *
		 * @param systemObjekt das Systemobjekt dessen Daten sich geändert haben.
		 */
		public DatensatzAktualisiertRunner(final SystemObjekt systemObjekt) {
			this.systemObjekt = systemObjekt;
		}

		@Override
		public void run() {

			RdsMeldungWrapper alterWrapper = null;
			RdsMeldungWrapper neuerWrapper = null;

			synchronized (lock) {

				if (systemObjekt instanceof RdsMeldung) {
					final RdsMeldung rdsMeldung = (RdsMeldung) systemObjekt;
					alterWrapper = getRdsMeldungWrapperInternal(rdsMeldung);
					neuerWrapper = erzeugeRdsMeldungWrapper(rdsMeldung);
					if (neuerWrapper != null) {
						rdsMeldungenWrapperMap.put(rdsMeldung, neuerWrapper);
					}
				}
			}

			// Benachrichtige Listener
			if ((neuerWrapper != null) && (listeners != null) && !listeners.isEmpty()) {
				final List<RdsMeldungenChangedListener> tempList = new ArrayList<>(listeners);
				for (final RdsMeldungenChangedListener listener : tempList) {
					listener.rdsMeldungChanged(neuerWrapper, alterWrapper);
				}
			}
		}
	}

	@Override
	public void invalidObject(final DynamicObject dynamicObject) {
		if (!listenerAktiv) {
			return;
		}

		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();
		if ((objektFactory != null) && objektFactory.isVerbunden()) {
			final SystemObjekt systemObjekt = objektFactory.getModellobjekt(dynamicObject);
			if (systemObjekt instanceof RdsMeldung) {
				threadPool.execute(new MeldungInvalidiertRunner((RdsMeldung) systemObjekt));
			}
		}
	}

	/**
	 * Runnable zur Behandlung einer Invalidierung.
	 *
	 * @author BitCtrl Systems GmbH, Andreas Meissner
	 * @version $Id: RdsMeldungenCache.java 25379 2010-08-05 13:02:03Z meissner $
	 */
	private class MeldungInvalidiertRunner implements Runnable {

		/** Die RDS-Meldung. */
		private final RdsMeldung rdsMeldung;

		/**
		 * Der Konstruktor.
		 *
		 * @param rdsMeldung die RDS-Meldung
		 */
		public MeldungInvalidiertRunner(final RdsMeldung rdsMeldung) {
			this.rdsMeldung = rdsMeldung;
		}

		@Override
		public void run() {

			final List<RdsMeldungWrapper> entfernteRdsMeldungen = new ArrayList<>();

			synchronized (lock) {

				removeUpdateListener(rdsMeldung);
				removeInvalidationListener(rdsMeldung);

				final RdsMeldungWrapper wrapper = getRdsMeldungWrapperInternal(rdsMeldung);
				if (wrapper != null) {
					entfernteRdsMeldungen.add(wrapper);
					rdsMeldungenWrapperMap.remove(rdsMeldung);
				}
			}

			// Benachrichtige Listener
			if ((listeners != null) && !listeners.isEmpty() && !entfernteRdsMeldungen.isEmpty()) {
				final List<RdsMeldungenChangedListener> tempList = new ArrayList<>(listeners);
				for (final RdsMeldungenChangedListener listener : tempList) {
					listener.rdsMeldungenRemoved(entfernteRdsMeldungen);
				}
			}
		}
	}

	/**
	 * Entfernt sämtliche nicht-globalen Listener und führt die Initialisierung
	 * dieses Caches erneut durch.
	 */
	public void reInit() {
		dispose(false);
		redoInit();
	}
}
