/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.konfigurationsdaten.KdBestehtAusLinienObjekten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmgeoreferenzierungglobal.objekte.Linie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.attribute.AtlBetriebsKilometer;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdBetriebsKilometer;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrassenTeilSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenTeilSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.StreckenAbschnitt.Intervall;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Cached die Beziehung {@link StrassenSegment} &lt;=&gt;
 * {@link StrassenTeilSegment} sowie {@link AtlBetriebsKilometer
 * BetriebsKilometer} und {@link StreckenAbschnitt StreckenAbschnitte}.
 *
 * @author BitCtrl Systems GmbH, schnepel
 * @version $Id$
 */
public class StrassenTeilSegmenteCache extends AbstractCache {

	/**
	 * Map, welche zu einem äußeren Straßensegment die sortierte Liste aller
	 * enthaltenen Straßenteilsegmente speichert.
	 */
	private Map<StrassenTeilSegment, Set<StrassenSegment>> strassenTeilSegmenteMap;

	/**
	 * Map, welche die entsprechenden Wrapper zu Straßenteilsegmenten speichert.
	 */
	private Map<StrassenTeilSegment, StrassenTeilSegmentWrapper> strassenTeilSegmentWrapperMap;

	/**
	 * Map, welche zu äußeren und inneren Straßensegment die gesetzten
	 * Betriebskilometer speichert.
	 */
	private Map<StrassenSegment, Map<String, List<AtlBetriebsKilometer>>> ss2bkMap;

	/**
	 * Map, welche zu äußeren und inneren Straßensegment ein Intervall für die
	 * Betriebskilometer speichert.
	 */
	private Map<StrassenSegment, Map<String, StreckenAbschnitt.Intervall>> ss2bkIntervallMap;

	/**
	 * Map, welche zu einem äußeren Straßensegment abhängig vom folgenden inneren
	 * Straßensegment das Intervall für die Betriebskilometer speichert.
	 */
	private Map<AeusseresStrassenSegment, Map<InneresStrassenSegment, Map<String, StreckenAbschnitt.Intervall>>> ass2iss2bkIntervallMap;

	/** Der Netz-Cache, von welchem dieser Cache abhängig ist. */
	private NetzCache netzCache;

	/** Der erweiterte Netz-Cache, von welchem dieser Cache abhängig ist. */
	private NetzCacheExtended netzCacheExtended;

	/** Der Verbindungs-Cache, von welchem dieser Cache abhängig ist. */
	private AbstractDavVerbindungsCache verbindungsCache;

	/**
	 * Der Konstruktor.
	 *
	 * @param verbindungsCache  der AbstractDavVerbindungsCache, von welchem dieser
	 *                          Cache abhängig ist
	 * @param netzCacheExtended der erweiterte Netz-Cache, von welchem dieser Cache
	 *                          abhängig ist
	 */
	public StrassenTeilSegmenteCache(final AbstractDavVerbindungsCache verbindungsCache,
			final NetzCacheExtended netzCacheExtended) {
		super(verbindungsCache, netzCacheExtended);
	}

	/**
	 * Initialisiert den Cache.
	 *
	 * @param vc  der AbstractDavVerbindungsCache, von welchem dieser Cache abhängig
	 *            ist
	 * @param nce der erweiterte Netz-Cache, von welchem dieser Cache abhängig ist
	 */
	protected void preInit(final AbstractDavVerbindungsCache vc, final NetzCacheExtended nce) {
		verbindungsCache = vc;
		netzCacheExtended = nce;
		netzCache = nce.getNetzCache();
	}

	@Override
	protected boolean doInit(final IMonitor monitor) {
		final long start = System.currentTimeMillis();
		boolean result = false;
		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Lade Straßenteilsegmente", 90000);
		strassenTeilSegmenteMap = new HashMap<>();

		if (ladeStrassenTeilSegmente(subMonitor.newChild(80000))) {
			result = ermittelBetriebsKilometer(subMonitor.newChild(10000));
		}
		if (result) {
			Debug.getLogger().info("Benötigte Zeit zum Initialisieren des StrassenTeilSegmenteCaches: "
					+ (System.currentTimeMillis() - start) + " ms");
		}
		return result;
	}

	/**
	 * Liefert eine sortierte Liste der {@link StrassenTeilSegment
	 * StrassenTeilSegmente} für das gegebene äußere Straßensegment zurück.
	 *
	 * @param strassenSegment das Straßensegment, nicht <code>null</code>
	 *
	 * @return eine sortierte Liste der {@link StrassenTeilSegment
	 *         StrassenTeilSegmente} für das gegebene äußere Straßensegment
	 */
	public List<StrassenTeilSegment> getStrassenTeilSegmente(final StrassenSegment strassenSegment) {
		ensureInit();
		return getStrassenTeilSegmenteInternal(strassenSegment);
	}

	@SuppressWarnings("unchecked")
	private List<StrassenTeilSegment> getStrassenTeilSegmenteInternal(final StrassenSegment strassenSegment) {
		if (strassenSegment == null) {
			throw new IllegalArgumentException("Übergebenes äußeres Straßensegment darf nicht null sein.");
		}

		return (List) strassenSegment.getKdBestehtAusLinienObjekten().getDatum().getLinienReferenz();
	}

	/**
	 * Liefert ein Set aller {@link StrassenTeilSegment} im Netz.
	 *
	 * @return ein Set aller {@link StrassenTeilSegment} im Netz
	 */
	public Set<StrassenTeilSegment> getStrassenTeilSegmente() {
		ensureInit();
		return strassenTeilSegmenteMap.keySet();
	}

	/**
	 * Liefert zu einem {@link StrassenTeilSegment} die assoziierten
	 * {@link StrassenSegment StrassenSegmente} im Netz.
	 *
	 * @param sts das Straßenteilsegement
	 *
	 * @return die assoziierten {@link StrassenSegment StrassenSegmente} zu einem
	 *         {@link StrassenTeilSegment} im Netz
	 */
	public Set<StrassenSegment> getStrassenSegmente(final StrassenTeilSegment sts) {
		ensureInit();
		return Collections.unmodifiableSet(strassenTeilSegmenteMap.get(sts));
	}

	/**
	 * Erzeugt für sämtliche Straßenteilsegmente entsprechende Wrapper, welche die
	 * Länge, Anzahl der Fahrstreifen sowie die gesetzten Betriebskilometer bereits
	 * aufgelöst vorhalten.
	 *
	 * @param monitor der Monitor
	 * @return true, bei erfolgreicher Ausführung
	 */
	@SuppressWarnings("unchecked")
	private boolean ladeStrassenTeilSegmente(final IMonitor monitor) {

		strassenTeilSegmentWrapperMap = new HashMap<>();

		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();
		final DataModel dataModel = objektFactory.getDav().getDataModel();

		final int totalWork = 120000;
		final SubMonitor subMonitor = SubMonitor.convert(monitor, totalWork);

		final int work = totalWork / 6;

		// (1)
		subMonitor.subTask("Lade Daten der Straßensegmente");
		final Set<StrassenSegment> segmente = netzCache.getStrassenSegmente();
		// final Collection<SystemObject> segmenteSO =
		// resolveSystemObjects(segmente);

		blockAbruf(dataModel, dataModel.getType(StrassenSegment.PID),
				new AttributeGroup[] { dataModel.getAttributeGroup(KdBestehtAusLinienObjekten.PID),
						dataModel.getAttributeGroup(KdBetriebsKilometer.PID),
						dataModel.getAttributeGroup(KdStrassenTeilSegment.PID) });

		// if (isBenutzeBlockAbruf()) {
		// dataModel.getConfigurationData(segmenteSO, dataModel
		// .getAttributeGroup(KdBestehtAusLinienObjekten.PID));
		// } else {
		// for (final SystemObject so : segmenteSO) {
		// so.getConfigurationData(dataModel
		// .getAttributeGroup(KdBestehtAusLinienObjekten.PID));
		// }
		// }

		subMonitor.worked(work);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (2)
		subMonitor.subTask("Lade Straßenteilsegmente");
		final Collection<? extends StrassenTeilSegment> teilSegmente = (Collection<? extends StrassenTeilSegment>) objektFactory
				.bestimmeModellobjekte(StrassenTeilSegment.PID);
		final Collection<SystemObject> teilSegmenteSO = resolveSystemObjects(teilSegmente);
		subMonitor.worked(work);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (3)
		subMonitor.subTask("Lade Daten der Straßenteilsegmente");

		blockAbruf(dataModel, dataModel.getType(StrassenTeilSegment.PID),
				dataModel.getAttributeGroup(KdStrassenTeilSegment.PID));

		// if (isBenutzeBlockAbruf()) {
		// dataModel.getConfigurationData(teilSegmenteSO,
		// dataModel.getAttributeGroup(KdStrassenTeilSegment.PID));
		// } else {
		// for (final SystemObject so : teilSegmenteSO) {
		// so.getConfigurationData(dataModel
		// .getAttributeGroup(KdStrassenTeilSegment.PID));
		// }
		// }

		subMonitor.worked(work);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (4)
		subMonitor.subTask("Lade Betriebskilometer der Straßenteilsegmente");

		blockAbruf(dataModel, dataModel.getType(StrassenTeilSegment.PID),
				dataModel.getAttributeGroup(KdBetriebsKilometer.PID));

		// if (isBenutzeBlockAbruf()) {
		// dataModel.getConfigurationData(teilSegmenteSO,
		// dataModel.getAttributeGroup(KdBetriebsKilometer.PID));
		// } else {
		// for (final SystemObject so : teilSegmenteSO) {
		// so.getConfigurationData(dataModel
		// .getAttributeGroup(KdBetriebsKilometer.PID));
		// }
		// }

		subMonitor.worked(work);
		if (subMonitor.isCanceled()) {
			subMonitor.done();
			return false;
		}

		// (5)
		final String taskName = "Ermittel relevante Straßenteilsegmente";
		subMonitor.subTask(taskName);
		SubMonitor subMonitor2 = subMonitor.newChild(work);
		subMonitor2.beginTask(taskName, segmente.size());
		final List<StrassenTeilSegment> strassenTeilSegmente = new ArrayList<>();
		for (final StrassenSegment segment : segmente) {
			final Feld<Linie> linien = segment.getKdBestehtAusLinienObjekten().getDatum().getLinienReferenz();
			if ((linien != null) && !linien.isEmpty()) {
				for (final Linie linie : linien) {
					if ((linie instanceof StrassenTeilSegment) && !strassenTeilSegmente.contains(linie)) {
						strassenTeilSegmente.add((StrassenTeilSegment) linie);
						if (linie instanceof StrassenTeilSegment) {
							Set<StrassenSegment> stsSet = strassenTeilSegmenteMap.get(linie);
							if (stsSet == null) {
								stsSet = new HashSet<>();
								strassenTeilSegmenteMap.put((StrassenTeilSegment) linie, stsSet);
							}
							stsSet.add(segment);
						}
					}
				}
			}
			subMonitor2.worked(1);
			if (subMonitor2.isCanceled()) {
				subMonitor2.done();
				return false;
			}
		}

		// (6)
		int i = 0;
		final int anzahlTeilSegmente = strassenTeilSegmente.size();
		if (anzahlTeilSegmente == 0) {
			return true;
		}
		monitor.subTask("Lade Straßenteilsegmente");
		subMonitor2 = subMonitor.newChild(work);
		subMonitor2.beginTask("Lade Straßenteilsegmente", strassenTeilSegmente.size());
		for (final StrassenTeilSegment strassenTeilSegment : strassenTeilSegmente) {

			subMonitor2.setTaskName("Initialisiere Teilsegment " + i + " von " + anzahlTeilSegmente);

			final StrassenTeilSegmentWrapper wrapper = new StrassenTeilSegmentWrapper(strassenTeilSegment);
			strassenTeilSegmentWrapperMap.put(strassenTeilSegment, wrapper);
			subMonitor2.worked(1);
			if (subMonitor2.isCanceled()) {
				subMonitor2.done();
				return false;
			}
			i++;
		}
		return true;
	}

	/**
	 * Ermittelt für sämtliche äußeren StraßenSegmente die
	 * Betriebskilometer-Informationen und speichert diese in entsprechenden Maps
	 * ab.
	 *
	 * @param monitor der Monitor
	 * @return true bei erfolgreicher Ausführung
	 *
	 * @see #ss2bkMap
	 * @see #ss2bkIntervallMap
	 * @see #ass2iss2bkIntervallMap
	 */
	private boolean ermittelBetriebsKilometer(final IMonitor monitor) {

		ss2bkMap = new HashMap<>();
		ss2bkIntervallMap = new HashMap<>();
		ass2iss2bkIntervallMap = new HashMap<>();

		final List<AeusseresStrassenSegment> aeussereSegmente = netzCache.getAeussereStrassenSegmente();
		final Set<StrassenKnoten> strassenKnoten = netzCache.getStrassenKnoten();

		final int anzahlKnoten = strassenKnoten.size();
		final int anzahlSegmente = aeussereSegmente.size();
		final int anzahlGesamt = anzahlKnoten + (2 * anzahlSegmente);
		if (anzahlGesamt == 0) {
			return true;
		}

		final SubMonitor subMonitor = SubMonitor.convert(monitor, anzahlGesamt);
		subMonitor.subTask("Ermittle Betriebskilometer");

		int i = 0;
		for (final StrassenKnoten knoten : strassenKnoten) {

			subMonitor.setTaskName(new StringBuffer("Analysiere Straßenknoten ").append(i).append(" von ")
					.append(anzahlKnoten).toString());

			for (final InneresStrassenSegment segment : knoten.getInnereStrassenSegmente()) {

				ermittelBetriebsKilometer(segment);

				if (subMonitor.isCanceled()) {
					subMonitor.done();
					return false;
				}
			}

			subMonitor.worked(1);
			i++;
		}

		i = 0;
		for (final AeusseresStrassenSegment segment : aeussereSegmente) {

			subMonitor.setTaskName(new StringBuffer("Analysiere äußeres Straßensegment ").append(i).append(" von ")
					.append(anzahlSegmente).toString());

			ermittelBetriebsKilometer(segment);

			subMonitor.worked(1);
			if (subMonitor.isCanceled()) {
				subMonitor.done();
				return false;
			}
			i++;
		}

		i = 0;
		for (final AeusseresStrassenSegment segment : aeussereSegmente) {
			subMonitor.setTaskName(
					new StringBuffer("Analysiere nachfolgende innere Straßensegmente des äußeres Straßensegment ")
							.append(i).append(" von ").append(anzahlSegmente).toString());

			ass2iss2bkIntervallMap.put(segment, new HashMap<InneresStrassenSegment, Map<String, Intervall>>());

			final StrassenKnoten nachKnoten = segment.getKdAeusseresStrassenSegment().getDatum().getNachKnoten();
			if (nachKnoten != null) {
				final List<InneresStrassenSegment> inneresSegmente = netzCacheExtended
						.getInneresStrassenSegmente(nachKnoten, segment);
				for (final InneresStrassenSegment inneresSegment : inneresSegmente) {
					for (final String blockNummer : ss2bkMap.get(segment).keySet()) {
						ermittelBetriebsKilometerIntervall(segment, inneresSegment, blockNummer, true);
						if (subMonitor.isCanceled()) {
							subMonitor.done();
							return false;
						}
					}
				}
			}

			subMonitor.worked(1);
			i++;
		}
		return true;
	}

	private void ermittelBetriebsKilometer(final StrassenSegment segment) {

		final Map<String, List<AtlBetriebsKilometer>> bkMap = new HashMap<>();
		ss2bkMap.put(segment, bkMap);
		final List<StrassenTeilSegment> teilSegmente = getStrassenTeilSegmenteInternal(segment);
		if (teilSegmente != null) {
			for (final StrassenTeilSegment teilSegment : teilSegmente) {
				if (teilSegment == null) {
					continue;
				}
				final StrassenTeilSegmentWrapper wrapper = getStrassenTeilSegmentWrapperInternal(teilSegment);
				if (wrapper != null) {
					for (final String blockNummer : wrapper.getBlockNummern()) {
						List<AtlBetriebsKilometer> bkList = bkMap.get(blockNummer);
						if (bkList == null) {
							bkList = new ArrayList<>();
							bkMap.put(blockNummer, bkList);
						}
						bkList.addAll(wrapper.getBetriebsKilometer(blockNummer));
					}
				}
			}
		}

		if ((ss2bkMap.get(segment) != null) && (ss2bkMap.get(segment).keySet() != null)) {
			for (final String blockNummer : ss2bkMap.get(segment).keySet()) {
				ermittelBetriebsKilometerIntervall(segment, null, blockNummer, true);
			}
		}
	}

	/**
	 * Ermittel das Intervall der Betriebskilometer für das gegebene Straßensegment
	 * und ein optionales folgendes inneres Straßensegment.
	 *
	 * @param segment                            das Straßensegment
	 * @param folgeSegment                       das optional folgende innere
	 *                                           Straßensegment, kann
	 *                                           <code>null</code> sein
	 * @param blockNummer                        die Blocknummer
	 * @param untersucheEntgegengesetzteSegmente <code>true</code>, wenn bei der
	 *                                           Analyse der Betriebskilometer ggf.
	 *                                           auch die entgegengesetzte Richtung
	 *                                           mitberücksichtigt werden soll,
	 *                                           ansonsten <code>false</code>
	 *
	 * @return das Intervall der Betriebskilometer für das gegebene Straßensegment
	 *         und ein optionales folgendes inneres Straßensegment
	 */
	private StreckenAbschnitt.Intervall ermittelBetriebsKilometerIntervall(final StrassenSegment segment,
			final InneresStrassenSegment folgeSegment, final String blockNummer,
			final boolean untersucheEntgegengesetzteSegmente) {

		if (segment == null) {
			return null;
		}

		Map<String, Intervall> bk2intervallMap = null;
		if (folgeSegment == null) {
			bk2intervallMap = ss2bkIntervallMap.get(segment);
			if (bk2intervallMap == null) {
				bk2intervallMap = new HashMap<>();
				ss2bkIntervallMap.put(segment, bk2intervallMap);
			}
		} else if (segment instanceof AeusseresStrassenSegment) {
			bk2intervallMap = ass2iss2bkIntervallMap.get(segment).get(folgeSegment);
			if (bk2intervallMap == null) {
				bk2intervallMap = new HashMap<>();
				ass2iss2bkIntervallMap.get(segment).put(folgeSegment, bk2intervallMap);
			}
		} else {
			return null;
		}

		final StreckenAbschnitt.Intervall intervall = bk2intervallMap.get(blockNummer);
		if (intervall != null) {
			return intervall;
		}

		List<AtlBetriebsKilometer> betriebsKilometer = getBetriebsKilometerInternal(segment, blockNummer);
		if (folgeSegment != null) {
			final List<AtlBetriebsKilometer> folgeBk = getBetriebsKilometerInternal(folgeSegment, blockNummer);
			if (folgeBk != null) {
				if (betriebsKilometer == null) {
					betriebsKilometer = new ArrayList<>();
				}
				betriebsKilometer.addAll(folgeBk);
			}
		}

		List<StrassenTeilSegment> teilSegmente = getStrassenTeilSegmenteInternal(segment);
		if (folgeSegment != null) {
			final List<StrassenTeilSegment> folgeTS = getStrassenTeilSegmenteInternal(folgeSegment);
			if (folgeTS != null) {
				if (teilSegmente == null) {
					teilSegmente = new ArrayList<>();
				}
				teilSegmente.addAll(folgeTS);
			}
		}

		if ((betriebsKilometer == null) || betriebsKilometer.isEmpty() || (teilSegmente == null)
				|| teilSegmente.isEmpty()) {
			return null;
		}

		final AtlBetriebsKilometer startBK = betriebsKilometer.get(0);
		final AtlBetriebsKilometer endBK = betriebsKilometer.get(betriebsKilometer.size() - 1);
		boolean aufsteigendeBK = true;

		if (betriebsKilometer.size() > 1) {

			// Bestimme Richtung
			aufsteigendeBK = endBK.getWert().getValue() >= startBK.getWert().getValue();

			// Überprüfe, ob Sprünge existieren (also die
			// Betriebskilometerangaben für dieses Segment fehlerhaft sind) und
			// somit kein Intervall bestimmt werden kann
			byte ascending = 0;
			double alterWert = -1;
			for (final AtlBetriebsKilometer bk : betriebsKilometer) {
				final double wert = bk.getWert().doubleValue();
				if (((wert > alterWert) && (ascending == -1)) || ((wert < alterWert) && (ascending == 1))) {
					return null;
				}
				if ((alterWert != -1) && (ascending == 0)) {
					if (alterWert < wert) {
						ascending = 1;
					} else if (alterWert > wert) {
						ascending = -1;
					}
				}
				alterWert = wert;
			}

		} else if (untersucheEntgegengesetzteSegmente && (segment instanceof AeusseresStrassenSegment)) {

			// Sonderfall: Richtung lässt sich ggf. über entgegengesetztes
			// äußeres Straßensegment bestimmen
			boolean richtungKonnteBestimmtWerden = false;
			final List<AeusseresStrassenSegment> segmente = netzCacheExtended
					.getAeussereStrassenSegmenteInEntgegengesetzerRichtung((AeusseresStrassenSegment) segment);
			for (final AeusseresStrassenSegment seg : segmente) {
				final StreckenAbschnitt.Intervall interval = ermittelBetriebsKilometerIntervall(seg, null, blockNummer,
						false);
				if (interval != null) {
					aufsteigendeBK = !interval.istAufsteigend();
					richtungKonnteBestimmtWerden = true;
					break;
				}
			}
			if (!richtungKonnteBestimmtWerden) {
				return null;
			}
		} else {
			return null;
		}

		StrassenTeilSegment startTeilSegment = null;
		StrassenTeilSegment endTeilSegment = null;
		for (final Iterator<StrassenTeilSegment> iter = teilSegmente.iterator(); iter.hasNext();) {
			final StrassenTeilSegment teilSegment = iter.next();
			final StrassenTeilSegmentWrapper wrapper = getStrassenTeilSegmentWrapperInternal(teilSegment);
			if (wrapper != null) {
				if (wrapper.getBetriebsKilometer(blockNummer).contains(startBK)) {
					startTeilSegment = teilSegment;
				}
				if (wrapper.getBetriebsKilometer(blockNummer).contains(endBK)) {
					endTeilSegment = teilSegment;
				}
			} else {
				iter.remove();
			}
		}
		final int startTeilSegmentIndex = teilSegmente.indexOf(startTeilSegment);
		final int endTeilSegmentIndex = teilSegmente.indexOf(endTeilSegment);

		// Start-Betriebskilometer
		long startBKWert = startBK.getWert().getValue();
		if (aufsteigendeBK) {
			startBKWert -= startBK.getOffset().getValue();
		} else {
			startBKWert += startBK.getOffset().getValue();
		}
		if (startTeilSegmentIndex > 0) {
			for (int index = startTeilSegmentIndex - 1; index >= 0; index--) {
				final StrassenTeilSegmentWrapper wrapper = getStrassenTeilSegmentWrapperInternal(
						teilSegmente.get(index));
				if (aufsteigendeBK) {
					startBKWert -= wrapper.getLaenge();
				} else {
					startBKWert += wrapper.getLaenge();
				}
			}
		}

		// End-Betriebskilometer
		long endBKWert = endBK.getWert().getValue();
		if (aufsteigendeBK) {
			endBKWert -= endBK.getOffset().getValue();
		} else {
			endBKWert += endBK.getOffset().getValue();
		}
		if (endTeilSegmentIndex < teilSegmente.size()) {
			for (int index = endTeilSegmentIndex; index < teilSegmente.size(); index++) {
				final StrassenTeilSegmentWrapper wrapper = getStrassenTeilSegmentWrapperInternal(
						teilSegmente.get(index));
				if (aufsteigendeBK) {
					endBKWert += wrapper.getLaenge();
				} else {
					endBKWert -= wrapper.getLaenge();
				}
			}
		}

		final StreckenAbschnitt.Intervall ergebnis = new StreckenAbschnitt.Intervall(startBKWert / 1000d,
				endBKWert / 1000d);
		bk2intervallMap.put(blockNummer, ergebnis);
		if (ergebnis.getLaenge() > 0) {
			return ergebnis;
		}
		return null;
	}

	/**
	 * Liefert die für das gegebene Straßensegment gesetzte Liste aller
	 * Betriebskilometer zurück.
	 *
	 * @param segment     das Straßensegment, nicht <code>null</code>
	 * @param blockNummer die Blocknummer, darf <code>null</code> sein
	 *
	 * @return die für das gegebene Straßensegment gesetzte Liste aller
	 *         Betriebskilometer, niemals <code>null</code>
	 */
	public List<AtlBetriebsKilometer> getBetriebsKilometer(final StrassenSegment segment, final String blockNummer) {
		ensureInit();
		return getBetriebsKilometerInternal(segment, blockNummer);
	}

	private List<AtlBetriebsKilometer> getBetriebsKilometerInternal(final StrassenSegment segment,
			final String blockNummer) {
		if (segment == null) {
			throw new IllegalArgumentException("Übergebenes Straßensegment darf nicht null sein.");
		}
		if (blockNummer == null) {
			return null;
		}
		if (ss2bkMap != null) {
			final Map<String, List<AtlBetriebsKilometer>> map = ss2bkMap.get(segment);
			if ((map != null) && (map.get(blockNummer) != null)) {
				return new ArrayList<>(map.get(blockNummer));
			}
		}
		return null;
	}

	/**
	 * Liefert das für das gegebene Straßensegment gesetzte Intervall der
	 * Betriebskilometer der gegebenen Blocknummer zurück.
	 *
	 * @param segment     das Straßensegment, darf nicht <code>null</code> sein
	 * @param blockNummer die Blocknummer, darf <code>null</code> sein
	 *
	 * @return das für das gegebene Straßensegment gesetzte Intervall der
	 *         Betriebskilometer, niemals <code>null</code>
	 */
	public StreckenAbschnitt.Intervall getBetriebsKilometerIntervall(final StrassenSegment segment,
			final String blockNummer) {
		ensureInit();
		return getBetriebsKilometerIntervallInternal(segment, blockNummer);
	}

	/* package */StreckenAbschnitt.Intervall getBetriebsKilometerIntervallInternal(final StrassenSegment segment,
			final String blockNummer) {
		if (segment == null) {
			throw new IllegalArgumentException("Übergebenes Straßensegment darf nicht null sein.");
		}
		if (blockNummer == null) {
			return null;
		}
		final Map<String, Intervall> map = ss2bkIntervallMap.get(segment);
		if (map != null) {
			return map.get(blockNummer);
		}
		return null;
	}

	/**
	 * Liefert das für das gegebene äußere sowie nachfolgende innere Straßensegment
	 * gesetzte Intervall der Betriebskilometer der gegebenen Blocknummer zurück.
	 *
	 * @param segment      das äußere Straßensegment, darf nicht <code>null</code>
	 *                     sein
	 * @param folgeSegment das innere Straßensegment, welches dem gegebenen
	 *                     Straßensegment nachfolgt, darf nicht <code>null</code>
	 *                     sein
	 * @param blockNummer  die Blocknummer, darf <code>null</code> sein
	 *
	 * @return das für das gegebene Straßensegment gesetzte Intervall der
	 *         Betriebskilometer, niemals <code>null</code>
	 */
	public StreckenAbschnitt.Intervall getBetriebsKilometerIntervall(final AeusseresStrassenSegment segment,
			final InneresStrassenSegment folgeSegment, final String blockNummer) {
		ensureInit();
		return getBetriebsKilometerIntervallInternal(segment, folgeSegment, blockNummer);
	}

	/* package */StreckenAbschnitt.Intervall getBetriebsKilometerIntervallInternal(
			final AeusseresStrassenSegment segment, final InneresStrassenSegment folgeSegment,
			final String blockNummer) {
		if (segment == null) {
			throw new IllegalArgumentException("Übergebenes Straßensegment darf nicht null sein.");
		}
		if (folgeSegment == null) {
			throw new IllegalArgumentException("Übergebenes inneres Straßensegment darf nicht null sein.");
		}
		if (blockNummer == null) {
			return null;
		}
		final Map<InneresStrassenSegment, Map<String, Intervall>> map = ass2iss2bkIntervallMap.get(segment);
		if (map != null) {
			final Map<String, Intervall> map2 = map.get(folgeSegment);
			if (map2 != null) {
				return map2.get(blockNummer);
			}
		}
		return null;
	}

	/**
	 * Liefert die Menge der Blocknummern zurück, für welche Betriebskilometer für
	 * das gegegebene Straßensegment definiert sind.
	 *
	 * @param segment das Straßensegment, darf nicht <code>null</code> sein
	 *
	 * @return die Menge der Blocknummern, für welche Betriebskilometer für das
	 *         gegebene Straßensegment definiert sind, niemals <code>null</code>
	 */
	public List<String> getBlockNummern(final StrassenSegment segment) {
		ensureInit();
		if (segment == null) {
			throw new IllegalArgumentException("Übergebenes Straßensegment darf nicht null sein.");
		}
		final List<String> ergebnis = new ArrayList<>(ss2bkMap.get(segment).keySet());
		for (final Iterator<String> iter = ergebnis.iterator(); iter.hasNext();) {
			final String bk = iter.next();
			if (getBetriebsKilometerIntervallInternal(segment, bk) == null) {
				iter.remove();
			}
		}
		return ergebnis;
	}

	/**
	 * Liefert den entsprechenden Wrapper zu dem gegebenen Straßenteilsegment
	 * zurück. Der Wrapper hat bereits die Werte Länge und Fahrstreifenanzahl
	 * aufgelöst vorliegen.
	 *
	 * @param strassenTeilSegment das Straßenteilsegment, nicht <code>null</code>
	 *
	 * @return den entsprechenden Wrapper zu dem gegebenen Straßenteilsegment
	 */
	public StrassenTeilSegmentWrapper getStrassenTeilSegmentWrapper(final StrassenTeilSegment strassenTeilSegment) {
		ensureInit();
		return getStrassenTeilSegmentWrapperInternal(strassenTeilSegment);
	}

	private StrassenTeilSegmentWrapper getStrassenTeilSegmentWrapperInternal(
			final StrassenTeilSegment strassenTeilSegment) {
		if (strassenTeilSegment == null) {
			throw new IllegalArgumentException("Übergebenes Straßenteilsegment darf nicht null sein.");
		}
		return strassenTeilSegmentWrapperMap.get(strassenTeilSegment);
	}

	/**
	 * Liefert den erweiterten Netz-Cache zurück, von dem dieser Cache abhängig ist.
	 *
	 * @return den erweiterten Netz-Cache, von dem dieser Cache abhängig ist
	 */
	public NetzCacheExtended getNetzCacheExtended() {
		return netzCacheExtended;
	}

	private Collection<SystemObject> resolveSystemObjects(final Collection<? extends SystemObjekt> objekte) {
		final ArrayList<SystemObject> objects = new ArrayList<>(objekte.size());
		for (final SystemObjekt objekt : objekte) {
			objects.add(objekt.getSystemObject());
		}
		return objects;
	}
}
