/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;

import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdInneresStrassenSegment.Daten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.InneresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenTeilSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.util.comparator.InnereSegmenteComparator;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Ein Streckenabschnitt entspricht einem Weg zwischen zwei (nicht identischen)
 * Straßenknoten auf einer Straße in gegebener Fahrtrichtung. Somit definiert
 * ein Streckenabschnitt eine Sequenz von äußeren und inneren Straßensegmenten,
 * welche nötig ist, um die Straßenknoten zu verbinden.
 *
 * Beachte: Während die sortierte Liste der den Streckenabschnitt definierenden
 * äußeren Straßensegmente eindeutig ist, sind alle zu einem Straßenknoten
 * gehörenden (möglichenweise auch parallelen) inneren Straßensegmente im
 * Streckenabschnitt abgelegt. Hier wird standardmäßig das innere Straßensegment
 * ausgewählt, welches die niedrigste PID besitzt. Es kann aber auch
 * nachträglich ein alternatives inneres Straßensegment gesetzt werden.
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 * @version $Id: StreckenAbschnitt.java 22169 2010-03-16 16:53:23Z meissner $
 */
public class StreckenAbschnitt implements Cloneable {

	/**
	 * Intervall-Hilfsklasse.
	 *
	 * Es können beliebige Start- und Endwerte gesetzt werden, also sowohl auf- als
	 * auch absteigende Intervalle definiert werden.
	 *
	 * @author BitCtrl Systems GmbH, Andreas Meissner
	 * @version $Id: Intervall.java 21582 2010-02-17 14:07:52Z meissner $
	 */
	public static class Intervall {

		/** Der Startwert. */
		private double start;

		/** Der Endwert. */
		private double ende;

		/**
		 * Der Konstruktor.
		 *
		 * @param start der Startwert
		 * @param ende  der Endwert
		 */
		public Intervall(final double start, final double ende) {
			this.start = start;
			this.ende = ende;
		}

		/**
		 * Der Konstruktor.
		 *
		 * @param intervall ein anderes Intervall
		 */
		public Intervall(final Intervall intervall) {
			if (intervall == null) {
				throw new IllegalArgumentException("Übergebenes Intervall darf nicht null sein.");
			}
			start = intervall.start;
			ende = intervall.ende;
		}

		/**
		 * Liefert den Startwert dieses Intervalls zurück.
		 *
		 * @return den Startwert dieses Intervalls
		 */
		public double getStart() {
			return start;
		}

		/**
		 * Setzt den Startwert dieses Intervalls.
		 *
		 * @param start der neue Startwert.
		 */
		public void setStart(final double start) {
			this.start = start;
		}

		/**
		 * Liefert den Endwert dieses Intervalls zurück.
		 *
		 * @return den Endwert dieses Intervalls
		 */
		public double getEnde() {
			return ende;
		}

		/**
		 * Setzt den Endwert dieses Intervalls.
		 *
		 * @param ende der neue Endwert
		 */
		public void setEnde(final double ende) {
			this.ende = ende;
		}

		/**
		 * Gibt <code>true</code> zurück, wenn es sich bei diesem Intervall um ein
		 * aufsteigendes handelt (Endwert &gt;= Startwert), ansonsten
		 * <code>false</code>.
		 *
		 * @return <code>true</code>, wenn es sich bei diesem Intervall um ein
		 *         aufsteigendes handelt (Endwert &gt;= Startwert), ansonsten
		 *         <code>false</code>
		 */
		public boolean istAufsteigend() {
			return ende >= start;
		}

		/**
		 * Liefert die Laenge (bzw. den Betrag) dieses Intervalls zurück.
		 *
		 * @return die Laenge (bzw. den Betrag) dieses Intervalls
		 */
		public double getLaenge() {
			return Math.abs(ende - start);
		}

		/**
		 * Liefert den minimalen Wert dieses Intervalls zurück. Dieser Wert entspricht
		 * abhängig davon, ob es sich um ein auf- oder absteigendes Intervall handelt,
		 * den Start- oder Endwert zurück.
		 *
		 * @return den minimalen Wert dieses Intervalls
		 */
		public double getMin() {
			return Math.min(start, ende);
		}

		/**
		 * Liefert den maximalen Wert dieses Intervalls zurück. Dieser Wert entspricht
		 * abhängig davon, ob es sich um ein auf- oder absteigendes Intervall handelt,
		 * den End- oder Startwert zurück.
		 *
		 * @return den minimalen Wert dieses Intervalls
		 */
		public double getMax() {
			return Math.max(start, ende);
		}

		@Override
		public String toString() {
			return "(" + start + ", " + ende + ")";
		}
	}

	/** JavaBeans PropertyChangeSupport. */
	private final PropertyChangeSupport changeSupport = new PropertyChangeSupport(this);

	/** Comparator zum Vergleichen/Sortieren von inneren Straßensegmenten. */
	private final InnereSegmenteComparator innereSegmenteComparator = new InnereSegmenteComparator();

	/** Der erweiterte Netz-Cache. */
	private final NetzCacheExtended netzCache;

	/** Der Straßenteilsegmente-Cache. */
	private final StrassenTeilSegmenteCache teilSegmenteCache;

	/** Straßenrichtung des Streckenabschnitts. */
	private final AttTmcRichtung richtung;

	/**
	 * Straßennummer der logische Straße auf welcher sich der Streckenabschnitt
	 * befindet.
	 */
	private final String strassenNummer;

	/**
	 * Sortierte Liste aller auf dem Streckenabschnitt liegenden Straßenknoten.
	 */
	private List<StrassenKnoten> strassenKnoten;

	/**
	 * Sortierte Liste aller auf dem Streckenabschnitt liegenden Straßensegmente.
	 * Diese Liste stellt die Folge der äußeren und inneren Straßensegmenten
	 * zwischen Startstraßenknoten und Endstraßenknoten dar.
	 */
	private List<StrassenSegment> strassenSegmente;

	/**
	 * Sortierte Liste aller auf dem Streckenabschnitt liegenden äußeren
	 * Straßensegmente. Diese Liste stellt die Folge der äußeren Straßensegmenten
	 * zwischen Startstraßenknoten und Endstraßenknoten dar.
	 */
	private List<AeusseresStrassenSegment> aeussereStrassenSegmente;

	/**
	 * Map, welche zu einem Straßenknoten alle inneren Straßensegmente auf diesem
	 * Streckenabschnitt speichert.
	 *
	 * <p>
	 * Achtung: Dem Start- und Endknoten des Streckenabschnitts sind keine inneren
	 * Straßensegmente zugeordnet. Außerdem kann es vorkommen, dass für einen
	 * Straßenknoten kein inneres Straßensegment existiert, welches das Von- und
	 * Nachsegment miteinander verbindet.
	 * </p>
	 */
	private Map<StrassenKnoten, List<InneresStrassenSegment>> innereStrassenSegmenteMap;

	/**
	 * Position des Streckenabschnittanfang (in Kilometer) im ersten äußeren
	 * Straßensegment.
	 */
	private double startOffset;

	/**
	 * Position des Streckenabschnittendes (in Kilometer) im letzten äußeren
	 * Straßensegment.
	 */
	private double endOffset;

	/**
	 * Maximal möglicher StartOffset (Einheit: Kilometer).
	 *
	 * @see #startOffset
	 */
	private double maxStartOffset;

	/**
	 * Maximal möglicher EndOffset (Einheit: Kilometer).
	 *
	 * @see #endOffset
	 */
	private double maxEndOffset;

	/**
	 * Anzahl der Fahrstreifen am Beginn des Streckenabschnitts. Entspricht der
	 * Anzahl der Fahrstreifen auf dem Straßenteilsegment an der Position des
	 * StartOffsets.
	 */
	private byte anzahlFahrstreifen;

	/**
	 * Betriebskilometerangabe (in Kilometer) am Beginn des Streckenabschnitts.
	 */
	private double startBetriebsKilometer;

	/**
	 * Möglicher Bereich für den Betriebskilometer am Beginn des Streckenabschnitts.
	 *
	 * @see #startBetriebsKilometer
	 */
	private Intervall startBetriebsKilometerBereich;

	/**
	 * Blocknummer des Bereichs für den Betriebskilometer am Beginn des
	 * Streckenabschnitts.
	 *
	 * @see #startBetriebsKilometerBereich
	 */
	private String startBetriebsKilometerBlockNummer;

	/**
	 * Betriebskilometerangabe (in Kilometer) am Ende des Streckenabschnitts (unter
	 * Berücksichtigung des EndOffsets).
	 */
	private double endBetriebsKilometer;

	/**
	 * Möglicher Bereich für den Betriebskilometer am Ende des Streckenabschnitts.
	 *
	 * @see #startBetriebsKilometer
	 */
	private Intervall endBetriebsKilometerBereich;

	/**
	 * Blocknummer des Bereichs für den Betriebskilometer am Ende des
	 * Streckenabschnitts.
	 *
	 * @see #startBetriebsKilometerBereich
	 */
	private String endBetriebsKilometerBlockNummer;

	/**
	 * Erzeugt einen leere Streckenabschnitt.
	 *
	 * Der Streckenabschnitt kann im Folgenden durch Hinzufügen von äußeren
	 * Straßensegmenten sukzessive aufgebaut werden.
	 *
	 * @param cache          der Cache, welcher diesen Streckenabschnitt verwaltet,
	 *                       darf nicht <code>null</code> sein
	 * @param strassenNummer die Strasse auf welchem der Streckenabschnitt liegt,
	 *                       nicht <code>null</code>
	 * @param richtung       die Straßenrichtung des Streckenabschnitts
	 *
	 * @see #addSegment(AeusseresStrassenSegment)
	 */
	/* package */ StreckenAbschnitt(final StreckenAbschnittCache cache, final String strassenNummer,
			final AttTmcRichtung richtung) {

		assert null != cache;
		if (strassenNummer == null) {
			throw new IllegalArgumentException("Übergebenene Straßennummer darf nicht null sein.");
		}

		netzCache = cache.getNetzCacheExtended();
		teilSegmenteCache = cache.getStrassenTeilSegmenteCache();
		this.strassenNummer = strassenNummer;
		this.richtung = richtung;
		strassenKnoten = new ArrayList<>();
		strassenSegmente = new ArrayList<>();
		aeussereStrassenSegmente = new ArrayList<>();
		innereStrassenSegmenteMap = new HashMap<>();

		setStandardStartBetriebsKilometerBlockNummer();
		setStandardEndBetriebsKilometerBlockNummer();
		setStartOffset(0d);
		setEndOffset(getMaxEndOffset());
	}

	/**
	 * Erzeugt einen Streckenabschnitt aus einem anderen.
	 *
	 * @param streckenAbschnitt der Streckenabschnitt, nicht <code>null</code>
	 */
	/* package */ StreckenAbschnitt(final StreckenAbschnitt streckenAbschnitt) {

		if (streckenAbschnitt == null) {
			throw new IllegalArgumentException("Übergebenener Streckenabschnitt darf nicht null sein.");
		}

		netzCache = streckenAbschnitt.netzCache;
		teilSegmenteCache = streckenAbschnitt.teilSegmenteCache;
		strassenNummer = new String(streckenAbschnitt.strassenNummer);
		richtung = streckenAbschnitt.richtung;
		strassenKnoten = new ArrayList<>(streckenAbschnitt.strassenKnoten);
		strassenSegmente = new ArrayList<>(streckenAbschnitt.strassenSegmente);
		aeussereStrassenSegmente = new ArrayList<>(streckenAbschnitt.aeussereStrassenSegmente);
		innereStrassenSegmenteMap = new HashMap<>(streckenAbschnitt.innereStrassenSegmenteMap);
		startOffset = streckenAbschnitt.startOffset;
		maxStartOffset = streckenAbschnitt.maxStartOffset;
		endOffset = streckenAbschnitt.endOffset;
		maxEndOffset = streckenAbschnitt.maxEndOffset;
		anzahlFahrstreifen = streckenAbschnitt.anzahlFahrstreifen;
		startBetriebsKilometer = streckenAbschnitt.startBetriebsKilometer;
		if (streckenAbschnitt.startBetriebsKilometerBlockNummer != null) {
			startBetriebsKilometerBlockNummer = new String(streckenAbschnitt.startBetriebsKilometerBlockNummer);
		} else {
			startBetriebsKilometerBlockNummer = null;
		}
		if (streckenAbschnitt.startBetriebsKilometerBereich != null) {
			startBetriebsKilometerBereich = new Intervall(streckenAbschnitt.startBetriebsKilometerBereich);
		} else {
			startBetriebsKilometerBereich = null;
		}
		endBetriebsKilometer = streckenAbschnitt.endBetriebsKilometer;
		if (streckenAbschnitt.endBetriebsKilometerBlockNummer != null) {
			endBetriebsKilometerBlockNummer = new String(streckenAbschnitt.endBetriebsKilometerBlockNummer);
		} else {
			endBetriebsKilometerBlockNummer = null;
		}
		if (streckenAbschnitt.endBetriebsKilometerBereich != null) {
			endBetriebsKilometerBereich = new Intervall(streckenAbschnitt.endBetriebsKilometerBereich);
		} else {
			endBetriebsKilometerBereich = null;
		}
	}

	@Override
	public StreckenAbschnitt clone() {
		try {
			super.clone();
		} catch (final CloneNotSupportedException e) {
			Debug.getLogger().finest("Es gab ein Problem beim Klonen eines Streckenabschnitts.", e);
		}
		return new StreckenAbschnitt(this);
	}

	/**
	 * Fügt dem Streckenabschnitt ein weiteres äußeres Straßensegment hinzu.
	 *
	 * Achtung: Das äußere Straßensegment muss auf derselben Straße sowie Richtung
	 * liegen, auf welchem dieser Streckenabschnitt definiert ist. Außerdem muss
	 * sich das gegebene äußere Straßensegment dem aktuell letzten äußeren
	 * Straßensegment direkt anschließen und es muss ein inneres Straßensegment
	 * existieren, welches das letzte äußere Straßensegment mit dem gegebenen
	 * äußeren Straßensegment verbindet.
	 *
	 * @param segment das äußere Straßensegment, nicht <code>null</code>
	 *
	 * @see #getStrassenNummer()
	 * @see #getRichtung()
	 */
	public void addSegment(final AeusseresStrassenSegment segment) {

		if (aeussereStrassenSegmente.contains(segment)) {
			return;
		}

		// Teste auf null
		if (segment == null) {
			throw new IllegalArgumentException("Übergebenenes äußeres Straßensegment darf nicht null sein.");
		}

		// Teste auf selbe Straße und Richtung
		final Strasse strasseDesSegments = segment.getKdStrassenSegment().getDatum().getGehoertZuStrasse();

		final String strassenNummerDesSegments = netzCache.getStrassenNummer(strasseDesSegments);

		if (!getStrassenNummer().equals(strassenNummerDesSegments)
				|| (segment.getKdAeusseresStrassenSegment().getDatum().getTmcRichtung() != getRichtung())) {
			throw new IllegalArgumentException(
					"Übergebenenes äußeres Straßensegment muss auf der für diesen Streckenabschnitt definierten logischen Straße (Straßennummer) sowie Fahrrichtung liegen.");
		}

		// Teste auf Einhaltung der Reihenfolge bzw. des Anschlusses
		final StrassenKnoten vonKnoten = segment.getKdAeusseresStrassenSegment().getDatum().getVonKnoten();
		final AeusseresStrassenSegment endSegment = getEndASS();
		if (endSegment != null) {
			final StrassenKnoten nachKnoten = endSegment.getKdAeusseresStrassenSegment().getDatum().getNachKnoten();
			if (!nachKnoten.equals(vonKnoten)) {
				throw new IllegalArgumentException(
						"Übergebenenes äußeres Straßensegment muss sich dem letzten äußeren Straßensegment direkt anschließen.");
			}
		}

		if (strassenKnoten.isEmpty()) {
			// ermittel Startknoten
			if ((vonKnoten != null) && !strassenKnoten.contains(vonKnoten)) {
				strassenKnoten.add(vonKnoten);
			}
		} else {
			// ermittel innere Straßensegmente
			final List<InneresStrassenSegment> innereSegmente = new ArrayList<>();
			Collection<InneresStrassenSegment> innereStrassenSegmente = null;
			innereStrassenSegmente = getEndKnoten().getInnereStrassenSegmente();
			if ((innereStrassenSegmente == null) || innereStrassenSegmente.isEmpty()) {
				Debug.getLogger().fine("Folgender Straßenknoten enthält keine inneren Straßensegmente: "
						+ getEndKnoten().getName() + " (PID: " + getEndKnoten().getPid() + ")");
			}
			if (innereStrassenSegmente != null) {
				for (final InneresStrassenSegment inneresSegment : innereStrassenSegmente) {
					final Daten datum = inneresSegment.getKdInneresStrassenSegment().getDatum();
					final AeusseresStrassenSegment vonSegment = datum.getVonStrassenSegment();
					final AeusseresStrassenSegment nachSegment = datum.getNachStrassenSegment();
					if ((vonSegment != null) && vonSegment.equals(getEndASS()) && (nachSegment != null)
							&& nachSegment.equals(segment)) {
						innereSegmente.add(inneresSegment);
					}
				}
			}

			if (!innereSegmente.isEmpty()) {
				// sortiere Liste der inneren Straßensegmente nach PID
				Collections.sort(innereSegmente, innereSegmenteComparator);

				// füge das erste innere Straßensegment der Segmentliste
				// hinzu (also das Segment mit der niedrigsten PID, sollte
				// dem Segment entsprechen, welches auf '0' endet)
				strassenSegmente.add(innereSegmente.get(0));
			} else {
				Debug.getLogger().warning("Streckenabschnitt enthält Straßenknoten ohne ISS!");
			}
			innereStrassenSegmenteMap.put(getEndKnoten(), innereSegmente);
		}

		strassenSegmente.add(segment);
		aeussereStrassenSegmente.add(segment);
		if (aeussereStrassenSegmente.size() <= 2) {
			setStandardStartBetriebsKilometerBlockNummer();
			setStartBetriebsKilometerBereich();
			setMaxStartOffset();
		}
		setStandardEndBetriebsKilometerBlockNummer();
		setEndBetriebsKilometerBereich();
		setMaxEndOffset();

		final StrassenKnoten nachKnoten = segment.getKdAeusseresStrassenSegment().getDatum().getNachKnoten();
		if ((nachKnoten != null) && !strassenKnoten.contains(nachKnoten)) {
			strassenKnoten.add(nachKnoten);
		}
	}

	/**
	 * Setzt das gegebene innere Straßensegment.
	 *
	 * Achtung: Das gegebene innere Straßensegment muss zu einem auf diesem
	 * Streckenabschnitt liegenden Straßenknoten gehören.
	 *
	 * @param segment das innere Straßensegment, darf nicht <code>null</code> sein
	 */
	public void setInneresStrassenSegment(final InneresStrassenSegment segment) {

		if (strassenSegmente.contains(segment)) {
			return;
		}

		// Teste auf null
		if (segment == null) {
			throw new IllegalArgumentException("Übergebenenes inneres Straßensegment darf nicht null sein.");
		}

		final InneresStrassenSegment tempStartISS = getStartISS();
		final InneresStrassenSegment tempEndISS = getEndISS();

		final List<StrassenSegment> tempStrassenSegmente = new ArrayList<>(strassenSegmente);
		for (final Entry<StrassenKnoten, List<InneresStrassenSegment>> entry : innereStrassenSegmenteMap.entrySet()) {
			final List<InneresStrassenSegment> segmente = entry.getValue();
			if (segmente.contains(segment)) {
				for (final InneresStrassenSegment inneresSegment : segmente) {
					final int index = strassenSegmente.indexOf(inneresSegment);
					if (index >= 0) {
						strassenSegmente.set(index, segment);

						if (tempStartISS != getStartISS()) {
							setStandardStartBetriebsKilometerBlockNummer();
							setStartBetriebsKilometerBereich();
							setMaxStartOffset();
							setStartOffset(0d);
							setAnzahlFahrstreifen();
						}
						if (tempEndISS != getEndISS()) {
							setStandardEndBetriebsKilometerBlockNummer();
							setEndBetriebsKilometerBereich();
							setMaxEndOffset();
							setEndOffset(getMaxEndOffset());
						}

						firePropertyChange("strassenSegmente", tempStrassenSegmente, strassenSegmente);
						return;
					}
				}
			}
		}
	}

	/**
	 * Liefert den Straßenknoten zurück, mit welchem dieser Streckenabschnitt
	 * beginnt.
	 *
	 * @return Startstraßenknoten dieses Streckenabschnitts, kann <code>null</code>
	 *         sein
	 */
	public StrassenKnoten getStartKnoten() {
		if (strassenKnoten.isEmpty()) {
			return null;
		}
		return strassenKnoten.get(0);
	}

	/**
	 * Liefert den Straßenknoten zurück, mit welchem dieser Streckenabschnitt endet.
	 *
	 * @return Endstraßenknoten dieses Streckenabschnitts, kann <code>null</code>
	 *         sein
	 */
	public StrassenKnoten getEndKnoten() {
		if (strassenKnoten.isEmpty()) {
			return null;
		}
		return strassenKnoten.get(strassenKnoten.size() - 1);
	}

	/**
	 * Setzt den Startstraßenknoten dieses Streckenabschnitts. Eine Änderung des
	 * Startstraßenknotens verändert die Liste der auf dem Streckenabschnitt
	 * passierten Straßenknoten und Straßensegmente.
	 *
	 * @param startKnoten der Startstraßenknoten, muss auf dem Streckenabschnitt
	 *                    liegen, darf nicht mit dem Endstraßenknoten identisch
	 *                    sein, nicht <code>null</code>
	 *
	 * @see #setEndKnoten(StrassenKnoten)
	 */
	public void setStartKnoten(final StrassenKnoten startKnoten) {
		final StrassenKnoten alterWert = getStartKnoten();
		if (startKnoten == null) {
			throw new IllegalArgumentException("Übergebenener Startstraßenknoten darf nicht null sein.");
		}
		if (!strassenKnoten.contains(startKnoten)) {
			throw new IllegalArgumentException(
					"Übergebenener Startstraßenknoten muss im Streckenabschnitt enthalten sein.");
		}

		if (startKnoten.equals(getEndKnoten())) {
			setAlleinigenKnoten(startKnoten);

		} else {

			int startIndex = strassenKnoten.indexOf(startKnoten);
			if (startIndex != 0) {

				// (1) ermittel Straßenknoten
				strassenKnoten = new ArrayList<>(strassenKnoten.subList(startIndex, strassenKnoten.size()));

				// (2) Ermittel Straßensegmente
				// (2.1) Finde zunächst das Startsegment
				AeusseresStrassenSegment startSegment = null;
				for (final AeusseresStrassenSegment segment : aeussereStrassenSegmente) {
					final StrassenKnoten vonKnoten = segment.getKdAeusseresStrassenSegment().getDatum().getVonKnoten();
					if (startKnoten.equals(vonKnoten)) {
						startSegment = segment;
						break;
					}
				}

				// (2.2) Initialisiere Liste der Straßensegmente
				if (startSegment == null) {
					startIndex = 0;
				} else {
					startIndex = strassenSegmente.indexOf(startSegment);
				}
				strassenSegmente = new ArrayList<>(strassenSegmente.subList(startIndex, strassenSegmente.size()));

				// (2.3) Initialisiere Liste der äußeren Straßensegmente
				if (startSegment != null) {
					startIndex = aeussereStrassenSegmente.indexOf(startSegment);
				}
				aeussereStrassenSegmente = new ArrayList<>(
						aeussereStrassenSegmente.subList(startIndex, aeussereStrassenSegmente.size()));

				// (3) Initialisiere innere Straßensegmente-Map
				innereStrassenSegmenteMap = new HashMap<>(innereStrassenSegmenteMap);
				final Set<StrassenKnoten> knoten = new HashSet<>(innereStrassenSegmenteMap.keySet());
				for (final StrassenKnoten k : knoten) {
					if (!strassenKnoten.contains(k)) {
						innereStrassenSegmenteMap.remove(k);
					}
				}
				innereStrassenSegmenteMap.remove(getStartKnoten());
			}

			// Die Liste der inneren Straßensegmente muss für den Endknoten neu
			// berechnet werden: Die Map muss hier alle inneren Straßensegmente
			// enthalten, dessen vonSegment dem EndSegment entspricht.
			if ((getEndKnoten() != null) && (getEndASS() != null)) {
				final List<InneresStrassenSegment> innereSegmenteDesEndKnotens = netzCache
						.getInneresStrassenSegmente(getEndKnoten(), getEndASS());
				Collections.sort(innereSegmenteDesEndKnotens, innereSegmenteComparator);
				innereStrassenSegmenteMap.put(getEndKnoten(), innereSegmenteDesEndKnotens);
				for (final InneresStrassenSegment iss : innereSegmenteDesEndKnotens) {
					strassenSegmente.remove(iss);
				}
			}

			// (4) Setze letztes inneres Straßensegment
			final InneresStrassenSegment standardEndInneresSegment = getStandardEndInneresSegment();
			if ((standardEndInneresSegment != null) && !strassenSegmente.contains(standardEndInneresSegment)) {
				strassenSegmente.add(standardEndInneresSegment);
			}
		}

		// (5) Ermittel StartOffset und Anzahl der Fahrstreifen
		setStandardStartBetriebsKilometerBlockNummer();
		setStartBetriebsKilometerBereich();
		setMaxStartOffset();
		setStartOffset(0d);

		setStandardEndBetriebsKilometerBlockNummer();
		setEndBetriebsKilometerBereich();
		setMaxEndOffset();
		setEndOffset(getMaxEndOffset());

		setAnzahlFahrstreifen();

		// (6) feuer PropertyChangeEvent
		firePropertyChange("startKnoten", alterWert, getStartKnoten());
	}

	/**
	 * Setzt den Endstraßenknoten dieses Streckenabschnitts. Eine Änderung des
	 * Endstraßenknoten verändert die Liste der auf dem Streckenabschnitt passierten
	 * Straßenknoten und Straßensegmente.
	 *
	 * @param endKnoten der Endstraßenknoten, muss auf dem Streckenabschnitt liegen,
	 *                  darf nicht mit dem Startstraßenknoten identisch sein, nicht
	 *                  <code>null</code>
	 *
	 * @see #setStartKnoten(StrassenKnoten)
	 */
	public void setEndKnoten(final StrassenKnoten endKnoten) {
		final StrassenKnoten alterWert = getEndKnoten();
		if (endKnoten == null) {
			throw new IllegalArgumentException("Übergebenener Endstraßenknoten darf nicht null sein.");
		}
		if (!strassenKnoten.contains(endKnoten)) {
			throw new IllegalArgumentException(
					"Übergebenener Endstraßenknoten muss im Streckenabschnitt enthalten sein.");
		}

		if (endKnoten.equals(getStartKnoten())) {
			setAlleinigenKnoten(endKnoten);

		} else {

			int endIndex = strassenKnoten.indexOf(endKnoten);
			if (endIndex == 0) {
				return;
			}

			// (1) ermittel Straßenknoten
			strassenKnoten = new ArrayList<>(strassenKnoten.subList(0, endIndex + 1));

			// (2) Ermittel Straßensegmente
			// (2.1) Finde zunächst das Endsegmente
			AeusseresStrassenSegment endSegment = null;
			for (final AeusseresStrassenSegment segment : aeussereStrassenSegmente) {
				final StrassenKnoten nachKnoten = segment.getKdAeusseresStrassenSegment().getDatum().getNachKnoten();
				if (endKnoten.equals(nachKnoten)) {
					endSegment = segment;
					break;
				}
			}

			// (2.2) Initialisiere Liste der Straßensegmente
			if (endSegment == null) {
				endIndex = strassenSegmente.size();
			} else {
				endIndex = strassenSegmente.indexOf(endSegment) + 1;
			}
			strassenSegmente = new ArrayList<>(strassenSegmente.subList(0, endIndex));

			// (2.3) Initialisiere Liste der äußeren Straßensegmente
			if (endSegment == null) {
				endIndex = aeussereStrassenSegmente.size();
			} else {
				endIndex = aeussereStrassenSegmente.indexOf(endSegment) + 1;
			}
			aeussereStrassenSegmente = new ArrayList<>(aeussereStrassenSegmente.subList(0, endIndex));

			// (3) Initialisiere innere Straßensegmente-Map
			innereStrassenSegmenteMap = new HashMap<>(innereStrassenSegmenteMap);
			final Set<StrassenKnoten> knoten = new HashSet<>(innereStrassenSegmenteMap.keySet());
			for (final StrassenKnoten k : knoten) {
				if (!strassenKnoten.contains(k)) {
					innereStrassenSegmenteMap.remove(k);
				}
			}
			// Die Liste der inneren Straßensegmente muss für den Endknoten neu
			// berechnet werden: Die Map muss hier alle inneren Straßensegmente
			// enthalten, dessen vonSegment dem EndSegment entspricht.
			if ((getEndKnoten() != null) && (getEndASS() != null)) {
				final List<InneresStrassenSegment> innereSegmenteDesEndKnotens = netzCache
						.getInneresStrassenSegmente(getEndKnoten(), getEndASS());
				Collections.sort(innereSegmenteDesEndKnotens, innereSegmenteComparator);
				innereStrassenSegmenteMap.put(getEndKnoten(), innereSegmenteDesEndKnotens);
				for (final InneresStrassenSegment iss : innereSegmenteDesEndKnotens) {
					strassenSegmente.remove(iss);
				}
			}

			// (4) Setze letztes inneres Straßensegment
			final InneresStrassenSegment standardEndInneresSegment = getStandardEndInneresSegment();
			if ((standardEndInneresSegment != null) && !strassenSegmente.contains(standardEndInneresSegment)) {
				strassenSegmente.add(standardEndInneresSegment);
			}
		}

		// (5) Ermittel Start- und EndOffset
		setStandardStartBetriebsKilometerBlockNummer();
		setStartBetriebsKilometerBereich();
		setMaxStartOffset();
		setStartOffset(0d);

		setStandardEndBetriebsKilometerBlockNummer();
		setEndBetriebsKilometerBereich();
		setMaxEndOffset();
		setEndOffset(getMaxEndOffset());

		setAnzahlFahrstreifen();

		// (6) feuer PropertyChangeEvent
		firePropertyChange("endKnoten", alterWert, getEndKnoten());
	}

	/**
	 * Hilfsmethode, welche diesen Streckenabschnitt für den Fall definiert, dass
	 * der Startknoten gleich dem Endknoten ist und dieser Streckenabschnitt sich
	 * somit über keinerlei äußere Straßensegmente erstreckt.
	 *
	 * @param knoten der alleinige Straßenknoten dieses Streckenabschnitts
	 */
	private void setAlleinigenKnoten(final StrassenKnoten knoten) {
		strassenKnoten = new ArrayList<>();
		strassenKnoten.add(knoten);
		aeussereStrassenSegmente = new ArrayList<>();
		strassenSegmente = new ArrayList<>();
		final Collection<InneresStrassenSegment> innereStrassenSegmente = knoten.getInnereStrassenSegmente();
		if (innereStrassenSegmente != null) {
			final ArrayList<InneresStrassenSegment> innereSegmente = new ArrayList<>(innereStrassenSegmente);
			for (final Iterator<InneresStrassenSegment> iter = innereSegmente.iterator(); iter.hasNext();) {
				final InneresStrassenSegment iss = iter.next();

				// Prüfe auf Richtung
				if (netzCache.getRichtung(iss) != richtung) {
					iter.remove();
					continue;
				}

				// Prüfe auf Straßennummer
				final String strNr = netzCache
						.getStrassenNummer(iss.getKdStrassenSegment().getDatum().getGehoertZuStrasse());
				if ((strNr == null) || !strNr.equals(strassenNummer)) {
					iter.remove();
				}
			}
			Collections.sort(innereSegmente, innereSegmenteComparator);
			innereStrassenSegmenteMap = new HashMap<>();
			innereStrassenSegmenteMap.put(knoten, innereSegmente);
			if (!innereSegmente.isEmpty()) {
				strassenSegmente.add(innereSegmente.get(0));
			}
		}
	}

	/**
	 * Liefert das erste Straßensegment zurück.
	 *
	 * @return das erste Straßensegment zurück, kann <code>null</code> sein
	 *
	 * @see #getStartASS
	 * @see #getStartISS
	 */
	public StrassenSegment getStartSegment() {
		if (strassenSegmente.isEmpty()) {
			return null;
		}
		return strassenSegmente.get(0);
	}

	/**
	 * Liefert das erste äußere Straßensegment zurück. Das erste äußere
	 * Straßensegment ist das Straßensegment, welches sich in Fahrtrichtung an den
	 * Startstraßenknoten anschließt.
	 *
	 * @return das erste äußere Straßensegment zurück, kann <code>null</code> sein
	 */
	public AeusseresStrassenSegment getStartASS() {
		if (aeussereStrassenSegmente.isEmpty()) {
			return null;
		}
		return aeussereStrassenSegmente.get(0);
	}

	/**
	 * Liefert das erste innere Straßensegment zurück.
	 *
	 * <p>
	 * Das erste innere Straßensegment entspricht entweder dem ersten Straßensegment
	 * (wenn der Startstraßenknoten gleich dem Endstraßenknoten ist) oder dem
	 * inneren Straßensegment, welches dem ersten äußeren Straßensegment folgt (als
	 * auf dem 2. Straßenknoten gestetzt ist).
	 * </p>
	 *
	 * @return das erste innere Straßensegment, kann <code>null</code> sein
	 *
	 * @see #getStartASS
	 * @see #getStartSegment()
	 * @see #getStartKnoten()
	 */
	public InneresStrassenSegment getStartISS() {
		if ((getStartKnoten() != null) && getStartKnoten().equals(getEndKnoten())
				&& (getStartSegment() instanceof InneresStrassenSegment)) {
			return (InneresStrassenSegment) getStartSegment();
		}
		if ((getStrassenKnoten() != null) && (getStrassenKnoten().size() > 1)) {
			final List<InneresStrassenSegment> innereSegmente = innereStrassenSegmenteMap
					.get(getStrassenKnoten().get(1));
			if (innereSegmente != null) {
				for (final InneresStrassenSegment segment : innereSegmente) {
					if (strassenSegmente.contains(segment)) {
						return segment;
					}
				}
			}
		}
		return null;
	}

	/**
	 * Liefert das innere Straßensegment zurück, welches standardmäßig im ersten
	 * Straßenknoten gesetzt ist.
	 *
	 * @return das innere Straßensegment, welches standardmäßig im ersten
	 *         Straßenknoten gesetzt ist
	 */
	private InneresStrassenSegment getStandardStartInneresSegment() {
		if ((getStrassenKnoten() != null) && (getStrassenKnoten().size() > 1)) {
			final List<InneresStrassenSegment> innereSegmente = innereStrassenSegmenteMap
					.get(getStrassenKnoten().get(1));
			if ((innereSegmente != null) && !innereSegmente.isEmpty()) {
				return innereSegmente.get(0);
			}
		}
		return null;
	}

	/**
	 * Liefert das letzte Straßensegment zurück.
	 *
	 * @return das letzte äußere Straßensegment zurück, kann <code>null</code> sein
	 *
	 * @see #getEndASS
	 * @see #getEndISS
	 */
	public StrassenSegment getEndSegment() {
		if (strassenSegmente.isEmpty()) {
			return null;
		}
		return strassenSegmente.get(strassenSegmente.size() - 1);
	}

	/**
	 * Liefert das letzte äußere Straßensegment zurück. Das letzte äußere
	 * Straßensegment ist das Straßensegment, welches sich auf dem Streckenabschnitt
	 * vor dem Endstraßenknoten befindet.
	 *
	 * @return das letzte äußere Straßensegment zurück, kann <code>null</code> sein
	 */
	public AeusseresStrassenSegment getEndASS() {
		if (aeussereStrassenSegmente.isEmpty()) {
			return null;
		}
		return aeussereStrassenSegmente.get(aeussereStrassenSegmente.size() - 1);
	}

	/**
	 * Liefert das letzte innere Straßensegment zurück.
	 *
	 * <p>
	 * Das letzte innere Straßensegment ist das innere Straßensegment, welches auf
	 * dem Endstraßenknoten gesetzt ist. Dieses innere Straßensegment entspricht
	 * somit dem letzten möglichen Straßensegment.
	 * </p>
	 *
	 * @return das innere Straßensegment, welches dem Endstraßenknoten folgt, kann
	 *         <code>null</code> sein
	 *
	 * @see #getEndASS
	 * @see #getEndSegment()
	 * @see #getEndKnoten()
	 */
	public InneresStrassenSegment getEndISS() {
		if (getEndKnoten() != null) {
			final List<InneresStrassenSegment> innereSegmente = innereStrassenSegmenteMap.get(getEndKnoten());
			if (innereSegmente != null) {
				for (final InneresStrassenSegment segment : innereSegmente) {
					if (strassenSegmente.contains(segment)) {
						return segment;
					}
				}
			}
		}
		return null;
	}

	/**
	 * Liefert das innere Straßensegment zurück, welches standardmäßig im letzten
	 * Straßenknoten gesetzt ist.
	 *
	 * @return das innere Straßensegment, welches standardmäßig im letzten
	 *         Straßenknoten gesetzt ist
	 */
	private InneresStrassenSegment getStandardEndInneresSegment() {
		if (getEndKnoten() != null) {
			final List<InneresStrassenSegment> innereSegmente = innereStrassenSegmenteMap.get(getEndKnoten());
			if ((innereSegmente != null) && !innereSegmente.isEmpty()) {
				return innereSegmente.get(0);
			}
		}
		return null;
	}

	/**
	 * Liefert die Länge des gegebenen Straßensegments zurück.
	 *
	 * @param segment das Straßensegment, von welchem die Länge bestimmt werden soll
	 *
	 * @return die Länge des gegebenen Straßensegments
	 */
	public double getLaenge(final StrassenSegment segment) {
		if (segment != null) {
			return segment.getKdStrassenSegment().getDatum().getLaenge().intValue() / 1000d;
		}
		return 0d;
	}

	/**
	 * Liefert den maximalen Wert für den StartOffset zurück (Einheit: Kilometer).
	 * Dieses Wert entspricht der Länge des ersten äußeren Straßensegments.
	 *
	 * @return maximal möglicher Wert für den StartOffset
	 */
	public double getMaxStartOffset() {
		return maxStartOffset;
	}

	/**
	 * Ermittelt den maximalen Wert für den StartOffset.
	 */
	public void setMaxStartOffset() {
		final double alterWert = maxStartOffset;
		final AeusseresStrassenSegment startASS = getStartASS();
		final InneresStrassenSegment startISS = getStartISS();
		double neuerWert = 0d;

		if (getStartKnoten() == null) {
			neuerWert = 0d;
		} else if (getStartKnoten().equals(getEndKnoten())) {
			// Fall 1: StartKnoten == EndKnoten
			neuerWert = startISS != null ? getLaenge(startISS) : 0d;
		} else if (getAeussereStrassenSegmente().size() == 1) {
			// Fall 2: StartKnoten != EndKnoten && ASS.size() == 1
			neuerWert = startASS != null ? getLaenge(startASS) : 0d;
		} else {
			// Fall 3: StartKnoten != EndKnoten && ASS.size() > 1
			if (startASS == null) {
				neuerWert = 0d;
			} else if (startISS == null) {
				neuerWert = getLaenge(startASS);
			} else {
				neuerWert = getLaenge(startASS) + getLaenge(startISS);
			}
		}
		maxStartOffset = Math.round(neuerWert * 1000) / 1000d;
		if (alterWert != maxStartOffset) {
			firePropertyChange("maxStartOffset", alterWert, maxStartOffset);
			if (startOffset > maxStartOffset) {
				setStartOffset(maxStartOffset);
			}
			ermittelStartBetriebsKilometer();
		}
	}

	/**
	 * Liefert den maximalen Wert für den EndOffset zurück (Einheit: Kilometer).
	 * Dieses Wert entspricht der Länge des letzten äußeren Straßensegments.
	 *
	 * @return maximal möglicher Wert für den EndOffset
	 */
	public double getMaxEndOffset() {
		return maxEndOffset;
	}

	/**
	 * Ermittelt den maximalen Wert für den EndOffset.
	 */
	public void setMaxEndOffset() {
		final double alterWert = maxEndOffset;
		final AeusseresStrassenSegment endASS = getEndASS();
		final InneresStrassenSegment endISS = getEndISS();
		double neuerWert = 0d;

		if (getEndKnoten() == null) {
			neuerWert = 0d;
		} else if (getEndKnoten().equals(getStartKnoten())) {
			// Fall 1: StartKnoten == EndKnoten
			neuerWert = endISS != null ? getLaenge(endISS) : 0d;
		} else {
			// Fall 2 & 3: StartKnoten != EndKnoten
			if (endASS == null) {
				neuerWert = 0d;
			} else if (endISS == null) {
				neuerWert = getLaenge(endASS);
			} else {
				neuerWert = getLaenge(endASS) + getLaenge(endISS);
			}
		}
		maxEndOffset = Math.round(neuerWert * 1000) / 1000d;
		if (alterWert != maxEndOffset) {
			firePropertyChange("maxEndOffset", alterWert, maxEndOffset);
			if (endOffset > maxEndOffset) {
				setEndOffset(maxEndOffset);
			}
			ermittelEndBetriebskilometer();
		}
	}

	/**
	 * Liefert die Position des Streckenabschnittanfang im ersten Straßensegment
	 * zurück (Einheit: Kilometer).
	 *
	 * @return die Position des Streckenabschnittanfang im ersten Straßensegment
	 *         (Einheit: Kilometer)
	 */
	public double getStartOffset() {
		return startOffset;
	}

	/**
	 * Setzt die Position des Streckenabschnittanfang im ersten Straßensegment.
	 *
	 * Ist der übergebene Wert größer als der maximale StartOffset, wird der
	 * Maximalwert angenommen.
	 *
	 * @param startOffset die Position des Streckenabschnittanfang im ersten
	 *                    Straßensegment (Einheit: Kilometer)
	 *
	 * @see #getMaxStartOffset()
	 */
	public void setStartOffset(final double startOffset) {
		if (startOffset != this.startOffset) {
			final double alterWert = this.startOffset;
			this.startOffset = Math.round(startOffset * 1000) / 1000d;
			if (this.startOffset < 0) {
				this.startOffset = 0d;
			} else if (this.startOffset > getMaxStartOffset()) {
				this.startOffset = getMaxStartOffset();
			}
			firePropertyChange("startOffset", alterWert, this.startOffset);

			setAnzahlFahrstreifen();
			ermittelStartBetriebsKilometer();
		}
	}

	/**
	 * Ermittelt den Start-Betriebskilometer abhängig vom aktuellen inneren und
	 * äußeren Start-Straßensegment und gesetzten Start-Offset.
	 */
	private void ermittelStartBetriebsKilometer() {
		final double alterWert = startBetriebsKilometer;
		startBetriebsKilometer = 0d;

		if (startBetriebsKilometerBereich != null) {

			double bk = 0d;
			double intervalStart = startBetriebsKilometerBereich.getStart();

			if (getAeussereStrassenSegmente().size() > 1) {

				// Fall 3: Startknoten != Endknoten && ASS.size() > 1
				final AeusseresStrassenSegment startASS = getStartASS();
				final Intervall assBkBereich = teilSegmenteCache.getBetriebsKilometerIntervallInternal(startASS,
						startBetriebsKilometerBlockNummer);

				if (assBkBereich != null) {

					final InneresStrassenSegment startISS = getStartISS();
					final double assLaenge = getLaenge(startASS);

					if ((startISS != null) && (getLaenge(startISS) > 0d) && (assLaenge > 0d)) {
						final double assBkBereichLaenge = assBkBereich.getLaenge();
						if (startOffset < assLaenge) {
							bk = (assBkBereichLaenge * startOffset) / assLaenge;
						} else {
							final double issLaenge = getLaenge(startISS);
							bk = ((startBetriebsKilometerBereich.getLaenge() - assBkBereichLaenge)
									* (startOffset - assLaenge)) / issLaenge;
							intervalStart = assBkBereich.getEnde();
						}
					} else {
						if (maxStartOffset > 0d) {
							bk = (startBetriebsKilometerBereich.getLaenge() * startOffset) / maxStartOffset;
						}
					}
				}

			} else {

				// Fall 1: Startknoten == Endknoten oder
				// Fall 2: Startknoten != Endknoten && ASS.size() == 1
				if (maxStartOffset > 0d) {
					bk = (startBetriebsKilometerBereich.getLaenge() * startOffset) / maxStartOffset;
				}
			}

			if (startBetriebsKilometerBereich.istAufsteigend()) {
				startBetriebsKilometer = intervalStart + bk;
			} else {
				startBetriebsKilometer = intervalStart - bk;
			}
			startBetriebsKilometer = Math.round(startBetriebsKilometer * 1000) / 1000d;
		}

		if (alterWert != startBetriebsKilometer) {
			firePropertyChange("startBetriebsKilometer", alterWert, startBetriebsKilometer);
		}
	}

	/**
	 * Liefert die Position des Streckenabschnittendes im letzten Straßensegment
	 * zurück (Einheit: Kilometer).
	 *
	 * @return die Position des Streckenabschnittendes im letzten Straßensegment
	 *         (Einheit: Kilometer)
	 */
	public double getEndOffset() {
		return endOffset;
	}

	/**
	 * Setzt die Position des Streckenabschittendes im letzten Straßensegment.
	 *
	 * Ist der übergebene Wert größer als der maximale EndOffset, wird der
	 * Maximalwert angenommen.
	 *
	 * @param endOffset die Position des Streckenabschnittendes im letzten
	 *                  Straßensegment (Einheit: Kilometer)
	 *
	 * @see #getMaxEndOffset()
	 */
	public void setEndOffset(final double endOffset) {
		if (endOffset != this.endOffset) {
			final double alterWert = this.endOffset;
			this.endOffset = Math.round(endOffset * 1000) / 1000d;
			if (this.endOffset < 0) {
				this.endOffset = 0d;
			} else if (this.endOffset > getMaxEndOffset()) {
				this.endOffset = getMaxEndOffset();
			}
			firePropertyChange("endOffset", alterWert, this.endOffset);

			// ermittel Betriebskilometer
			ermittelEndBetriebskilometer();
		}
	}

	/**
	 * Ermittelt den End-Betriebskilometer abhängig vom aktuellen inneren und
	 * äußeren End-Straßensegment und gesetzten End-Offset.
	 */
	private void ermittelEndBetriebskilometer() {
		final double alterWert = endBetriebsKilometer;
		endBetriebsKilometer = 0d;

		if ((endBetriebsKilometerBereich != null) && (getEndKnoten() != null)) {

			double bk = 0d;
			double intervalStart = endBetriebsKilometerBereich.getStart();

			if (!getEndKnoten().equals(getStartKnoten())) {

				// Fall 2 & 3: Startknoten != Endknoten
				final AeusseresStrassenSegment endASS = getEndASS();
				final Intervall assBkBereich = teilSegmenteCache.getBetriebsKilometerIntervallInternal(endASS,
						endBetriebsKilometerBlockNummer);

				if (assBkBereich != null) {

					final InneresStrassenSegment endISS = getEndISS();
					final double assLaenge = getLaenge(endASS);

					if ((endISS != null) && (getLaenge(endISS) > 0d) && (assLaenge > 0d)) {
						final double assBkBereichLaenge = assBkBereich.getLaenge();
						if (endOffset <= assLaenge) {
							bk = (assBkBereichLaenge * endOffset) / assLaenge;
						} else {
							final double issLaenge = getLaenge(endISS);
							bk = ((endBetriebsKilometerBereich.getLaenge() - assBkBereichLaenge)
									* (endOffset - assLaenge)) / issLaenge;
							intervalStart = assBkBereich.getEnde();
						}
					} else {
						if (maxEndOffset > 0d) {
							bk = (endBetriebsKilometerBereich.getLaenge() * endOffset) / maxEndOffset;
						}
					}
				}

			} else {

				// Fall 1: Startknoten == Endknoten
				if (maxEndOffset > 0d) {
					bk = (endBetriebsKilometerBereich.getLaenge() * endOffset) / maxEndOffset;
				}
			}

			if (endBetriebsKilometerBereich.istAufsteigend()) {
				endBetriebsKilometer = intervalStart + bk;
			} else {
				endBetriebsKilometer = intervalStart - bk;
			}
			endBetriebsKilometer = Math.round(endBetriebsKilometer * 1000) / 1000d;
		}

		if (alterWert != endBetriebsKilometer) {
			firePropertyChange("endBetriebsKilometer", alterWert, endBetriebsKilometer);
		}
	}

	/**
	 * Liefert die Anzahl der Fahrstreifen am Beginn dieses Streckenabschnitts
	 * zurück. Hierbei wird der StartOffset mitberücksichtigt. Es wird also die
	 * Anzahl der Fahrstreifen des Straßenteilsegments an der entsprechenden
	 * Position des Offsets zurückgeliefert.
	 *
	 * @return die Anzahl der Fahrstreifen am Beginn dieses Streckenabschnitts
	 *         zurück
	 */
	public byte getAnzahlFahrstreifen() {
		return anzahlFahrstreifen;
	}

	/**
	 * Bestimmt die Anzahl der Fahrstreifen. Dieses Wert ist abhängig vom
	 * Startstraßensegment sowie dem StartOffset.
	 */
	public void setAnzahlFahrstreifen() {
		final byte alterWert = anzahlFahrstreifen;
		byte neuerWert = 0;

		final StrassenSegment startSegment = getStartSegment();

		if (startSegment == null) {
			neuerWert = 0;

		} else {

			double relativerStartOffset = getStartOffset();
			List<StrassenTeilSegment> teilSegmente = new ArrayList<>();

			if (startSegment instanceof InneresStrassenSegment) {
				teilSegmente = teilSegmenteCache.getStrassenTeilSegmente(startSegment);

			} else if (startSegment instanceof AeusseresStrassenSegment) {

				final AeusseresStrassenSegment startASS = (AeusseresStrassenSegment) startSegment;
				final InneresStrassenSegment startISS = getStartISS();
				final double assLaenge = getLaenge(startASS);

				if ((startISS != null) && (relativerStartOffset >= assLaenge)) {
					teilSegmente = teilSegmenteCache.getStrassenTeilSegmente(startISS);
					relativerStartOffset -= assLaenge;
				} else {
					teilSegmente = teilSegmenteCache.getStrassenTeilSegmente(startASS);
				}
			}

			double offset = 0;
			for (final StrassenTeilSegment teilSegment : teilSegmente) {
				final StrassenTeilSegmentWrapper wrapper = teilSegmenteCache.getStrassenTeilSegmentWrapper(teilSegment);
				offset += (wrapper.getLaenge() / 1000d);
				if (relativerStartOffset < offset) {
					neuerWert = wrapper.getAnzahlFahrstreifen();
					break;
				}
			}
		}

		if (alterWert != neuerWert) {
			anzahlFahrstreifen = neuerWert;
			firePropertyChange("anzahlFahrstreifen", alterWert, anzahlFahrstreifen);
		}
	}

	/**
	 * Liefert den Betriebskilometer am Beginn dieses Streckenabschnitts zurück
	 * (Einheit: Kilometer).
	 *
	 * @return den Betriebskilometer am Beginn dieses Streckenabschnitts (Einheit:
	 *         Kilometer)
	 */
	public double getStartBetriebsKilometer() {
		return startBetriebsKilometer;
	}

	/**
	 * Setzt den Betriebskilometer am Start dieses Streckenabschnitts.
	 *
	 * @param startBetriebsKilometer die neue Startbetriebskilometerangabe.
	 */
	public void setStartBetriebsKilometer(final double startBetriebsKilometer) {
		if (startBetriebsKilometerBereich == null) {
			return;
		}

		if (startBetriebsKilometer != this.startBetriebsKilometer) {
			final double alterWert = this.startBetriebsKilometer;
			this.startBetriebsKilometer = Math.round(startBetriebsKilometer * 1000) / 1000d;
			if (this.startBetriebsKilometer < startBetriebsKilometerBereich.getMin()) {
				this.startBetriebsKilometer = startBetriebsKilometerBereich.getMin();
			} else if (this.startBetriebsKilometer > startBetriebsKilometerBereich.getMax()) {
				this.startBetriebsKilometer = startBetriebsKilometerBereich.getMax();
			}
			firePropertyChange("startBetriebsKilometer", alterWert, this.startBetriebsKilometer);

			ermittelStartOffset();
			setAnzahlFahrstreifen();
		}
	}

	/**
	 * Ermittelt den Start-Offset abhängig vom aktuellen inneren und äußeren
	 * Start-Straßensegment und gesetzten Start-Betriebskilometer.
	 */
	private void ermittelStartOffset() {
		final double alterWert = startOffset;
		double neuerWert = 0d;

		if ((startBetriebsKilometerBereich != null) && (startBetriebsKilometerBereich.getLaenge() > 0)) {

			// (1) Bestimme zunächste den relativen Wert für den
			// Betriebskilometers
			double bk = 0d;
			if (startBetriebsKilometerBereich.istAufsteigend()) {
				bk = startBetriebsKilometer - startBetriebsKilometerBereich.getMin();
			} else {
				bk = startBetriebsKilometerBereich.getMax() - startBetriebsKilometer;
			}

			// (2) Ermittel Offset
			final AeusseresStrassenSegment startASS = getStartASS();
			final InneresStrassenSegment startISS = getStartISS();
			final Intervall assBkBereich = teilSegmenteCache.getBetriebsKilometerIntervallInternal(startASS,
					startBetriebsKilometerBlockNummer);
			final double assLaenge = getLaenge(startASS);

			if ((getAeussereStrassenSegmente().size() > 1) && (startISS != null) && (assBkBereich != null)
					&& (assBkBereich.getLaenge() > 0)) {

				// Fall 3: Startknoten != Endknoten && ASS.size() > 1
				final double assBkBereichLaenge = assBkBereich.getLaenge();
				if (bk < assBkBereichLaenge) {
					neuerWert = (assLaenge * bk) / assBkBereichLaenge;
				} else {
					neuerWert = assLaenge + ((getLaenge(startISS) * (bk - assBkBereichLaenge))
							/ (startBetriebsKilometerBereich.getLaenge() - assBkBereichLaenge));
				}

			} else {

				// Fall 1: Startknoten == Endknoten oder
				// Fall 2: Startknoten != Endknoten && ASS.size() == 1
				neuerWert = (maxStartOffset * bk) / startBetriebsKilometerBereich.getLaenge();
			}
		}

		startOffset = Math.round(neuerWert * 1000) / 1000d;
		if (alterWert != startOffset) {
			firePropertyChange("startOffset", alterWert, startOffset);
		}
	}

	/**
	 * Liefert den möglichen Bereich für den Betriebskilometer am Beginn dieses
	 * Streckenabschnitts zurück.
	 *
	 * @return den den möglichen Bereich für den Betriebskilometer am Beginn dieses
	 *         Streckenabschnitts
	 */
	public Intervall getStartBetriebsKilometerBereich() {
		return startBetriebsKilometerBereich;
	}

	/**
	 * Setzt den möglichen Bereich für den Betriebskilometer am Beginn dieses
	 * Streckenabschnitts.
	 */
	public void setStartBetriebsKilometerBereich() {
		Intervall alterWert = null;
		if (startBetriebsKilometerBereich != null) {
			alterWert = new Intervall(startBetriebsKilometerBereich);
		}

		final AeusseresStrassenSegment startASS = getStartASS();
		final InneresStrassenSegment startISS = getStartISS();
		final InneresStrassenSegment standardStartISS = getStandardStartInneresSegment();

		if (getStartKnoten() == null) {
			startBetriebsKilometerBereich = null;
		} else if (getStartKnoten().equals(getEndKnoten())) {
			// Fall 1: StartKnoten == EndKnoten
			startBetriebsKilometerBereich = startISS != null
					? teilSegmenteCache.getBetriebsKilometerIntervallInternal(startISS,
							startBetriebsKilometerBlockNummer)
					: null;
		} else if (getAeussereStrassenSegmente().size() == 1) {
			// Fall 2: StartKnoten != EndKnoten && ASS.size() == 1
			startBetriebsKilometerBereich = startASS != null
					? teilSegmenteCache.getBetriebsKilometerIntervallInternal(startASS,
							startBetriebsKilometerBlockNummer)
					: null;
		} else if (startASS != null) {
			// Fall 3: StartKnoten != EndKnoten && ASS.size() > 1

			// Beachte: Hier wird nicht das aktuell gesetzte innere
			// Straßensegment genommen, sondern das innere Straßensegment auf
			// der Hauptfahrbeziehung bzw. das Kürzeste
			if (standardStartISS != null) {
				startBetriebsKilometerBereich = teilSegmenteCache.getBetriebsKilometerIntervallInternal(startASS,
						standardStartISS, startBetriebsKilometerBlockNummer);
				if (startBetriebsKilometerBereich == null) {
					startBetriebsKilometerBereich = teilSegmenteCache.getBetriebsKilometerIntervallInternal(startASS,
							startBetriebsKilometerBlockNummer);
				}
			} else {
				startBetriebsKilometerBereich = teilSegmenteCache.getBetriebsKilometerIntervallInternal(startASS,
						startBetriebsKilometerBlockNummer);
			}
		} else {
			startBetriebsKilometerBereich = null;
		}

		firePropertyChange("startBetriebsKilometerBereich", alterWert, startBetriebsKilometerBereich);
		ermittelStartBetriebsKilometer();
	}

	/**
	 * Liefert die Blocknummer des Bereichs für den Betriebskilometer am Beginn des
	 * Streckenabschnitts zurück.
	 *
	 * @return die Blocknummer des Bereichs für den Betriebskilometer am Beginn des
	 *         Streckenabschnitts
	 */
	public String getStartBetriebsKilometerBlockNummer() {
		return startBetriebsKilometerBlockNummer;
	}

	/**
	 * Setzt den Standard-Wert für die Blocknummer des Bereichs für den
	 * Betriebskilometer am Beginn des Streckenabschnitts.
	 */
	public void setStandardStartBetriebsKilometerBlockNummer() {
		List<String> blockNummern = new ArrayList<>();
		if (getStartKnoten() == null) {
			blockNummern = new ArrayList<>();
		} else if (getStartKnoten().equals(getEndKnoten())) {
			// Fall 1: StartKnoten == EndKnoten
			blockNummern = getStartISS() != null ? teilSegmenteCache.getBlockNummern(getStartISS())
					: new ArrayList<>();
		} else if (getStartASS() != null) {
			// Fall 2 & 3: StartKnoten != EndKnoten
			blockNummern = new ArrayList<>(teilSegmenteCache.getBlockNummern(getStartASS()));
		}
		startBetriebsKilometerBlockNummer = blockNummern.isEmpty() ? null : blockNummern.get(0);
	}

	/**
	 * Setzt die Blocknummer des Bereichs für den Betriebskilometer am Beginn des
	 * Streckenabschnitts.
	 *
	 * @param neuerWert die neue Blocknummer
	 */
	public void setStartBetriebsKilometerBlockNummer(final String neuerWert) {
		final String alterWert = startBetriebsKilometerBlockNummer;
		startBetriebsKilometerBlockNummer = neuerWert;
		firePropertyChange("startBetriebsKilometerBlockNummer", alterWert, neuerWert);
		setStartBetriebsKilometerBereich();
	}

	/**
	 * Liefert den Betriebskilometer am Ende dieses Streckenabschnitts zurück
	 * (Einheit: Kilometer).
	 *
	 * @return den Betriebskilometer am Ende dieses Streckenabschnitts (Einheit:
	 *         Kilometer)
	 */
	public double getEndBetriebsKilometer() {
		return endBetriebsKilometer;
	}

	/**
	 * Setzt den Betriebskilometer am Ende dieses Streckenabschnitts.
	 *
	 * @param endBetriebsKilometer die neue Endbetriebskilometerangabe
	 */
	public void setEndBetriebsKilometer(final double endBetriebsKilometer) {
		if (endBetriebsKilometerBereich == null) {
			return;
		}

		if (endBetriebsKilometer != this.endBetriebsKilometer) {
			final double alterWert = this.endBetriebsKilometer;
			this.endBetriebsKilometer = Math.round(endBetriebsKilometer * 1000) / 1000d;
			if (this.endBetriebsKilometer < endBetriebsKilometerBereich.getMin()) {
				this.endBetriebsKilometer = endBetriebsKilometerBereich.getMin();
			} else if (this.endBetriebsKilometer > endBetriebsKilometerBereich.getMax()) {
				this.endBetriebsKilometer = endBetriebsKilometerBereich.getMax();
			}
			firePropertyChange("endBetriebsKilometer", alterWert, this.endBetriebsKilometer);

			ermittelEndOffset();
		}
	}

	/**
	 * Ermittelt den End-Offset abhängig vom aktuellen inneren und äußeren
	 * End-Straßensegment und gesetzten End-Betriebskilometer.
	 */
	private void ermittelEndOffset() {
		final double alterWert = endOffset;
		double neuerWert = getMaxEndOffset();

		if ((endBetriebsKilometerBereich != null) && (endBetriebsKilometerBereich.getLaenge() > 0)) {

			// (1) Bestimme zunächste den relativen Wert für den
			// Betriebskilometers
			double bk = 0d;
			if (endBetriebsKilometerBereich.istAufsteigend()) {
				bk = endBetriebsKilometer - endBetriebsKilometerBereich.getMin();
			} else {
				bk = endBetriebsKilometerBereich.getMax() - endBetriebsKilometer;
			}

			// (2) Ermittel Offset
			final AeusseresStrassenSegment endASS = getEndASS();
			final InneresStrassenSegment endISS = getEndISS();
			final Intervall assBkBereich = teilSegmenteCache.getBetriebsKilometerIntervallInternal(endASS,
					endBetriebsKilometerBlockNummer);
			final double assLaenge = getLaenge(endASS);

			if ((getEndKnoten() != null) && !getEndKnoten().equals(getStartKnoten()) && (endISS != null)
					&& (assBkBereich != null) && (assBkBereich.getLaenge() > 0)) {

				// Fall 2 & 3: Startknoten != Endknoten
				final double assBkBereichLaenge = assBkBereich.getLaenge();
				if (bk < assBkBereichLaenge) {
					neuerWert = (assLaenge * bk) / assBkBereichLaenge;
				} else {
					neuerWert = assLaenge + ((getLaenge(endISS) * (bk - assBkBereichLaenge))
							/ (endBetriebsKilometerBereich.getLaenge() - assBkBereichLaenge));
				}

			} else {

				// Fall 1: Startknoten == Endknoten
				neuerWert = (maxEndOffset * bk) / endBetriebsKilometerBereich.getLaenge();
			}
		}

		endOffset = Math.round(neuerWert * 1000) / 1000d;
		if (alterWert != endOffset) {
			firePropertyChange("endOffset", alterWert, endOffset);
		}
	}

	/**
	 * Liefert den möglichen Bereich für den Betriebskilometer am Ende dieses
	 * Streckenabschnitts zurück.
	 *
	 * @return den den möglichen Bereich für den Betriebskilometer am Ende dieses
	 *         Streckenabschnitts
	 */
	public Intervall getEndBetriebsKilometerBereich() {
		return endBetriebsKilometerBereich;
	}

	/**
	 * Setzt den möglichen Bereich für den Betriebskilometer am Ende dieses
	 * Streckenabschnitts.
	 */
	public void setEndBetriebsKilometerBereich() {
		Intervall alterWert = null;
		if (endBetriebsKilometerBereich != null) {
			alterWert = new Intervall(endBetriebsKilometerBereich);
		}

		final AeusseresStrassenSegment endASS = getEndASS();
		final InneresStrassenSegment endISS = getEndISS();
		final InneresStrassenSegment standardEndISS = getStandardEndInneresSegment();

		if (getEndKnoten() == null) {
			endBetriebsKilometerBereich = null;
		} else if (getEndKnoten().equals(getStartKnoten())) {
			// Fall 1: StartKnoten == EndKnoten
			endBetriebsKilometerBereich = endISS != null
					? teilSegmenteCache.getBetriebsKilometerIntervallInternal(endISS, endBetriebsKilometerBlockNummer)
					: null;
		} else if (endASS != null) {
			// Fall 2 & 3: StartKnoten != EndKnoten

			// Beachte: Hier wird nicht das aktuell gesetzte innere
			// Straßensegment genommen, sondern das innere Straßensegment auf
			// der Hauptfahrbeziehung bzw. das Kürzeste
			if (standardEndISS != null) {
				endBetriebsKilometerBereich = teilSegmenteCache.getBetriebsKilometerIntervallInternal(endASS,
						standardEndISS, endBetriebsKilometerBlockNummer);
				if (endBetriebsKilometerBereich == null) {
					endBetriebsKilometerBereich = teilSegmenteCache.getBetriebsKilometerIntervallInternal(endASS,
							endBetriebsKilometerBlockNummer);
				}
			} else {
				endBetriebsKilometerBereich = teilSegmenteCache.getBetriebsKilometerIntervallInternal(endASS,
						endBetriebsKilometerBlockNummer);
			}
		} else {
			endBetriebsKilometerBereich = null;
		}

		firePropertyChange("endBetriebsKilometerBereich", alterWert, endBetriebsKilometerBereich);
		ermittelEndBetriebskilometer();
	}

	/**
	 * Liefert die Blocknummer des Bereichs für den Betriebskilometer am Ende des
	 * Streckenabschnitts zurück.
	 *
	 * @return die Blocknummer des Bereichs für den Betriebskilometer am Ende des
	 *         Streckenabschnitts
	 */
	public String getEndBetriebsKilometerBlockNummer() {
		return endBetriebsKilometerBlockNummer;
	}

	/**
	 * Setzt den Standard-Wert für die Blocknummer des Bereichs für den
	 * Betriebskilometer am Beginn des Streckenabschnitts.
	 */
	public void setStandardEndBetriebsKilometerBlockNummer() {
		List<String> blockNummern = new ArrayList<>();
		if (getEndKnoten() == null) {
			blockNummern = new ArrayList<>();
		} else if (getEndKnoten().equals(getStartKnoten())) {
			// Fall 1: StartKnoten == EndKnoten
			blockNummern = getEndISS() != null ? teilSegmenteCache.getBlockNummern(getEndISS())
					: new ArrayList<>();
		} else if (getEndASS() != null) {
			// Fall 2 & 3: StartKnoten != EndKnoten
			blockNummern = new ArrayList<>(teilSegmenteCache.getBlockNummern(getEndASS()));
		}
		endBetriebsKilometerBlockNummer = blockNummern.isEmpty() ? null : blockNummern.get(0);
	}

	/**
	 * Setzt die Blocknummer des Bereichs für den Betriebskilometer am Ende des
	 * Streckenabschnitts.
	 *
	 * @param neuerWert die neue Blocknummer
	 */
	public void setEndBetriebsKilometerBlockNummer(final String neuerWert) {
		final String alterWert = endBetriebsKilometerBlockNummer;
		endBetriebsKilometerBlockNummer = neuerWert;
		firePropertyChange("endBetriebsKilometerBlockNummer", alterWert, neuerWert);
		setEndBetriebsKilometerBereich();
	}

	/**
	 * Liefert die Straßenrichtung dieses Streckenabschnitts zurück.
	 *
	 * @return die Straßenrichtung dieses Streckenabschnitts, niemals
	 *         <code>null</code>
	 */
	public AttTmcRichtung getRichtung() {
		return richtung;
	}

	/**
	 * Liefert die Straßennummer der logischen Straße zurück, auf welcher sich
	 * dieser Streckenabschnitt befindet.
	 *
	 * @return die Straße auf welcher sich dieser Streckenabschnitt befindet,
	 *         niemals <code>null</code>
	 */
	public String getStrassenNummer() {
		return strassenNummer;
	}

	/**
	 * Liefert die sortierte Liste der auf dem Streckenabschnitt liegenden
	 * Straßenknoten zurück.
	 *
	 * @return die sortierte Liste der auf dem Streckenabschnitt liegenden
	 *         Straßenknoten, niemals <code>null</code>
	 */
	public List<StrassenKnoten> getStrassenKnoten() {
		return strassenKnoten;
	}

	/**
	 * Liefert die sortierte Liste der Straßensegmente zurück. Dabei wechseln sich
	 * äußere und innere Straßensegmente ab.
	 *
	 * Achtung: Es kann vorkommen, dass innere Straßensegmente fehlen.
	 *
	 * @see #hatFehlendeInnereSegmente()
	 *
	 * @return die sortierte Liste der Straßensegmente, niemals <code>null</code>
	 */
	public List<StrassenSegment> getStrassenSegmente() {
		return strassenSegmente;
	}

	/**
	 * Liefert die sortierte Liste der äußeren Straßensegmente zurück. Dabei
	 * wechseln sich äußere und innere Straßensegmente ab.
	 *
	 * @return die sortierte Liste der äußeren Straßensegmente, niemals
	 *         <code>null</code>
	 */
	public List<AeusseresStrassenSegment> getAeussereStrassenSegmente() {
		return aeussereStrassenSegmente;
	}

	/**
	 * Liefert die Strassenknoten zurück, welche auf dem Streckenabschnitt nach dem
	 * gegebenen Straßenknoten liegen.
	 *
	 * @param knoten der Straßenknoten, nicht <code>null</code>
	 *
	 * @return die Strassenknoten, welche auf dem Streckenabschnitt nach dem
	 *         gegebenen Straßenknoten liegen, niemals <code>null</code>
	 */
	public List<StrassenKnoten> getNachfolgerStrassenKnoten(final StrassenKnoten knoten) {

		if (knoten == null) {
			throw new IllegalArgumentException("Übergebenener Straßenknoten darf nicht null sein.");
		}

		final int index = strassenKnoten.indexOf(knoten);
		if ((index >= 0) && ((index + 1) < strassenKnoten.size())) {
			return new ArrayList<>(strassenKnoten.subList(index + 1, strassenKnoten.size()));
		}
		return new ArrayList<>();
	}

	/**
	 * Liefert die Strassenknoten zurück, welche auf dem Streckenabschnitt vor dem
	 * gegebenen Straßenknoten liegen.
	 *
	 * @param knoten der Straßenknoten, nicht <code>null</code>
	 *
	 * @return die Strassenknoten, welche auf dem Streckenabschnitt vor dem
	 *         gegebenen Straßenknoten liegen, niemals <code>null</code>
	 */
	public List<StrassenKnoten> getVorgaengerStrassenKnoten(final StrassenKnoten knoten) {

		if (knoten == null) {
			throw new IllegalArgumentException("Übergebenener Straßenknoten darf nicht null sein.");
		}

		final int index = strassenKnoten.indexOf(knoten);
		if ((index >= 0) && (index < strassenKnoten.size())) {
			return new ArrayList<>(strassenKnoten.subList(0, index));
		}
		return new ArrayList<>();
	}

	/**
	 * Liefert wahr zurück, wenn in der Folge der Straßensegmente in diesem
	 * Streckenabschnitt innere Straßensegmente fehlen.
	 *
	 * @return wahr, wenn in der Folge der Straßensegmente in diesem
	 *         Streckenabschnitt innere Straßensegmente fehlen
	 */
	public boolean hatFehlendeInnereSegmente() {
		for (final Entry<StrassenKnoten, List<InneresStrassenSegment>> entry : innereStrassenSegmenteMap.entrySet()) {
			if (entry.getValue().isEmpty()) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Liefert <code>true</code> zurück, wenn es sich bei allen äußeren
	 * Straßensegmenten, welche diesen Streckenabschnitt definieren, um äußere
	 * Straßensegmente handelt, dessen PID auf '0' endet.
	 *
	 * @return <code>true</code>, wenn dieser Streckenabschnitt den bevorzugten Weg
	 *         zwischen den definierten Start- und Endstraßenknoten definiert.
	 */
	public boolean istStandardStreckenAbschnitt() {
		for (final AeusseresStrassenSegment segment : getAeussereStrassenSegmente()) {
			if (!segment.getPid().endsWith("0")) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Liefert die inneren Straßensegmente für den gegebenen Straßenknoten zurück.
	 *
	 * @param knoten der Straßenknoten, nicht <code>null</code>
	 *
	 * @return die inneren Straßensegmente für den gegebenen Straßenknoten, niemals
	 *         <code>null</code>
	 */
	public List<InneresStrassenSegment> getInnereStrassenSegmente(final StrassenKnoten knoten) {
		List<InneresStrassenSegment> ergebnis = innereStrassenSegmenteMap.get(knoten);
		if (ergebnis == null) {
			ergebnis = new ArrayList<>();
		}
		return ergebnis;
	}

	/**
	 * Liefert <code>true</code> zurück, wenn für mindestens einen Straßenknoten
	 * mehrere innere Straßensegmente existieren.
	 *
	 * @return <code>true</code>, wenn für mindestens einen Straßenknoten mehrere
	 *         innere Straßensegmente existieren
	 */
	public boolean existierenAlternativeInnereStrassenSegmente() {
		for (final Entry<StrassenKnoten, List<InneresStrassenSegment>> entry : innereStrassenSegmenteMap.entrySet()) {
			if (entry.getValue().size() > 1) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Gibt die Straßenknoten zurück, welche alternative innere Straßensegmente
	 * besitzen.
	 *
	 * @return die Straßenknoten, welche alternative innere Straßensegmente
	 *         besitzen, niemals <code>null</code>
	 */
	public List<StrassenKnoten> getStrassenKnotenMitAlternativenInnerenStrassenSegmenten() {
		final List<StrassenKnoten> ergebnis = new ArrayList<>();
		for (final Entry<StrassenKnoten, List<InneresStrassenSegment>> entry : innereStrassenSegmenteMap.entrySet()) {
			if (entry.getValue().size() > 1) {
				ergebnis.add(entry.getKey());
			}
		}
		return ergebnis;
	}

	/**
	 * Liefert <code>true</code> zurück, wenn dieser Streckenabschnitt lediglich auf
	 * einem Straßenknoten definiert ist.
	 *
	 * @return <code>true</code> zurück, wenn dieser Streckenabschnitt lediglich auf
	 *         einem Straßenknoten definiert ist, ansonsten <code>false</code>
	 */
	public boolean isStartGleichEndKnoten() {
		return (getStartKnoten() != null) && getStartKnoten().equals(getEndKnoten());
	}

	@Override
	public String toString() {
		final StringBuffer buf = new StringBuffer();
		buf.append(getStartKnoten().getName());
		buf.append(" --> ");
		buf.append(getEndKnoten().getName());
		buf.append(" (");
		buf.append(aeussereStrassenSegmente.size());
		buf.append(" Segmente)");
		return buf.toString();
	}

	/**
	 * Liefert einen Info-String zurück.
	 *
	 * @return einen Info-String
	 */
	public String getInfo() {
		for (final StrassenSegment segment : strassenSegmente) {
			if (segment instanceof AeusseresStrassenSegment) {
				final AeusseresStrassenSegment a = (AeusseresStrassenSegment) segment;
				return a.getName() + ": " + a.getKdAeusseresStrassenSegment().getDatum().getVonKnoten().getName()
						+ " --> " + a.getKdAeusseresStrassenSegment().getDatum().getNachKnoten().getName();
			} else if (segment instanceof InneresStrassenSegment) {
				final InneresStrassenSegment i = (InneresStrassenSegment) segment;
				return i.getName() + ": " + i.getKdInneresStrassenSegment().getDatum().getVonStrassenSegment().getName()
						+ " --> " + i.getKdInneresStrassenSegment().getDatum().getNachStrassenSegment().getName();
			}
		}
		return "Leerer Streckenabschnitt";
	}

	@Override
	public boolean equals(final Object obj) {

		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}

		final StreckenAbschnitt other = (StreckenAbschnitt) obj;

		// Strasse
		if (!Objects.equals(strassenNummer, other.strassenNummer)) {
			return false;
		}

		// Richtung
		if (richtung == null) {
			if (other.richtung != null) {
				return false;
			}
		} else if (richtung != other.richtung) {
			return false;
		}

		// Offsets
		if (Double.doubleToLongBits(startOffset) != Double.doubleToLongBits(other.startOffset)) {
			return false;
		}
		if (Double.doubleToLongBits(endOffset) != Double.doubleToLongBits(other.endOffset)) {
			return false;
		}

		// Segmente
		if (strassenSegmente == null) {
			if (other.strassenSegmente != null) {
				return false;
			}
		} else if (other.strassenSegmente != null) {
			if ((strassenSegmente.size() != other.strassenSegmente.size())
					|| !strassenSegmente.containsAll(other.strassenSegmente)) {
				return false;
			}
		}

		return true;
	}

	@Override
	public int hashCode() {
		return Objects.hash(endOffset, richtung, startOffset, strassenNummer, strassenSegmente);
	}

	/**
	 * Fügt diesem Streckenabschnitt einen neuen PropertyChangeListener hinzu.
	 *
	 * @param propertyName der Name der Property
	 * @param listener     der Listener
	 *
	 * @see PropertyChangeSupport#addPropertyChangeListener(String,
	 *      PropertyChangeListener)
	 */
	public void addPropertyChangeListener(final String propertyName, final PropertyChangeListener listener) {
		if (!Arrays.asList(changeSupport.getPropertyChangeListeners(propertyName)).contains(listener)) {
			changeSupport.addPropertyChangeListener(propertyName, listener);
		}
	}

	/**
	 * Entfernt den gegebenen PropertyChangeListener von diesem Streckenabschnit.
	 *
	 * @param propertyName der Name der Property
	 * @param listener     der Listener
	 *
	 * @see PropertyChangeSupport#removePropertyChangeListener(String,
	 *      PropertyChangeListener)
	 */
	public void removePropertyChangeListener(final String propertyName, final PropertyChangeListener listener) {
		changeSupport.removePropertyChangeListener(propertyName, listener);
	}

	/**
	 * Informiert alle registrierten PropertyChangeListener über eine Änderung.
	 *
	 * @param propertyName der Name der Property
	 * @param oldValue     der alter Wert
	 * @param newValue     der neue Wert
	 *
	 * @see PropertyChangeSupport#firePropertyChange(String, Object, Object)
	 */
	protected void firePropertyChange(final String propertyName, final Object oldValue, final Object newValue) {
		changeSupport.firePropertyChange(propertyName, oldValue, newValue);
	}
}
