/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.bitctrl.util.monitor.IMonitor;
import com.bitctrl.util.monitor.SubMonitor;

import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsLocation;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsTMCRichtung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.objekte.RdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.parameter.PdRdsMeldung;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcLocationCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.konfigurationsdaten.KdTmcLocationCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcLinie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.objekte.TmcPunkt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdAeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrassenKnotenLclInfo;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.konfigurationsdaten.KdStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.AeusseresStrassenSegment;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.Strasse;
import de.bsvrz.sys.funclib.bitctrl.modell.tmverkehrglobal.objekte.StrassenKnoten;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Diese Hilfsklasse cacht interessante Beziehungen zwischen Objekten des
 * Straßennetzes und TMC-Lokationen.
 *
 * @author BitCtrl Systems GmbH, thierfelder
 */
public class TmcCache extends AbstractCache {

	private static final int LOCATION_CODE_UNBEKANNT = -1;

	// TMC-Location-Code <-> äußeres Straßensegment
	private Map<Integer, Set<AeusseresStrassenSegment>> tmcLocationCode2aess;
	private Map<AeusseresStrassenSegment, Integer> aess2TmcLocationCode;
	private Map<AeusseresStrassenSegment, TmcPunkt> aess2TmcPunkt;
	private Map<TmcPunkt, AeusseresStrassenSegment> tmcPunkt2aess;
	private Map<Strasse, TmcLinie> strasse2TmcLinie;
	private Map<Strasse, Integer> strasse2TmcLocationCode;
	private Map<TmcLinie, Strasse> tmcLinie2strasse;

	// TMC-Location-Code <-> Straßenknoten
	private Map<StrassenKnoten, TmcPunkt> sk2TmcPunkt;
	private Map<TmcPunkt, StrassenKnoten> tmcPunkt2sk;

	/**
	 * Der Netz-Cache von dem dieser Cache indirekt abhängig ist.
	 */
	private NetzCache netzCache;

	/**
	 * Der Verbindungs-Cache von dem dieser Cache indirekt abhängig ist.
	 */
	private AbstractDavVerbindungsCache verbindungsCache;

	/**
	 * Der Konstruktor.
	 *
	 * @param netzCache        der Netz-Cache, von welchem dieser Cache abhängig ist
	 * @param verbindungsCache der Verbindungs-Cache, von welchem dieser Cache
	 *                         abhängig ist
	 */
	public TmcCache(final NetzCache netzCache, final AbstractDavVerbindungsCache verbindungsCache) {
		super(netzCache, verbindungsCache);
	}

	/**
	 * Initialisiert diesen Cache.
	 *
	 * @param netzC        der Netz-Cache, von welchem dieser Cache abhängig ist
	 * @param verbindungsC der Verbindungs-Cache, von welchem dieser Cache abhängig
	 *                     ist
	 */
	protected void preInit(final NetzCache netzC, final AbstractDavVerbindungsCache verbindungsC) {
		netzCache = netzC;
		verbindungsCache = verbindungsC;
	}

	@SuppressWarnings("unchecked")
	@Override
	protected boolean doInit(final IMonitor monitor) {
		setName(getNetzCache().getName());

		final SubMonitor subMonitor = SubMonitor.convert(monitor);
		subMonitor.beginTask("Initialisiere TMC-Cache", 100000);

		if (!analysiereStrassensegmente(subMonitor)) {
			return false;
		}
		final Map<Integer, StrassenKnoten> lc2Sk = analysiereStrassenknoten(subMonitor);
		analysiereTmcPunkte(subMonitor, lc2Sk);

		return true;
	}

	private boolean analysiereStrassensegmente(final SubMonitor subMonitor) {
		tmcLocationCode2aess = new HashMap<>();
		aess2TmcLocationCode = new HashMap<>();
		tmcPunkt2aess = new HashMap<>();
		aess2TmcPunkt = new HashMap<>();
		strasse2TmcLinie = new HashMap<>();
		strasse2TmcLocationCode = new HashMap<>();
		tmcLinie2strasse = new HashMap<>();

		final List<AeusseresStrassenSegment> segmente = netzCache.getAeussereStrassenSegmente();
		final int anzahlSegmente = segmente.size();
		final SubMonitor subMonitor2 = subMonitor.newChild(30000);
		subMonitor2.beginTask("Analysiere Straßensegmente", anzahlSegmente);

		int i = 1;
		for (final AeusseresStrassenSegment segment : segmente) {
			subMonitor2.subTask(new StringBuffer("Analysiere äußeres Straßensegment ").append(i).append(" von ")
					.append(anzahlSegmente).append(": ").append(segment.getName()).toString());

			cacheLocationCode(segment);
			subMonitor2.worked(1);
			if (subMonitor.isCanceled()) {
				subMonitor.done();
				return false;
			}
			i++;
		}
		return true;
	}

	private Map<Integer, StrassenKnoten> analysiereStrassenknoten(final SubMonitor subMonitor) {
		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();

		final List<StrassenKnoten> strassenKnoten = (List<StrassenKnoten>) objektFactory
				.bestimmeModellobjekte(StrassenKnoten.PID);
		final int anzahlKnoten = strassenKnoten.size();
		final SubMonitor subMonitor3 = subMonitor.newChild(30000);
		subMonitor3.beginTask("Analysiere Straßenknoten", anzahlKnoten);

		int i = 0;
		final Map<Integer, StrassenKnoten> lc2Sk = new HashMap<>();
		for (final StrassenKnoten sk : strassenKnoten) {
			i++;
			subMonitor3.subTask(new StringBuffer("Analysiere Straßenknoten ").append(i).append(" von ")
					.append(anzahlKnoten).append(": ").append(sk.getName()).toString());

			int tmcLocationCode = bestimmeLocationCodePerLclInfo(sk);
			if (tmcLocationCode == LOCATION_CODE_UNBEKANNT) {
				tmcLocationCode = bestimmeLocationCodePerPid(sk);
			}
			if (tmcLocationCode != LOCATION_CODE_UNBEKANNT) {
				lc2Sk.put(tmcLocationCode, sk);
			}
		}

		return lc2Sk;
	}

	private int bestimmeLocationCodePerLclInfo(final StrassenKnoten sk) {
		final KdStrassenKnotenLclInfo.Daten lclInfo = sk.getKdStrassenKnotenLclInfo().getDatum();
		if (lclInfo != null) {
			for (final TmcPunkt p : lclInfo.getTmcPunkt()) {
				if (p != null) {
					final KdTmcLocationCode.Daten lc = p.getKdTmcLocationCode().getDatum();
					if (lc != null) {
						return lc.getTmcLocationCode().getValue();
					}
				}
			}
		}

		return LOCATION_CODE_UNBEKANNT;
	}

	private int bestimmeLocationCodePerPid(final StrassenKnoten sk) {
		final String pid = sk.getPid();
		final String[] pidKomponenten = pid.split("\\.");

		if (pidKomponenten.length < 2) {
			return LOCATION_CODE_UNBEKANNT;
		}

		final String lcl = pidKomponenten[1];
		try {
			return Integer.parseInt(lcl);
		} catch (final NumberFormatException ex) {
			return LOCATION_CODE_UNBEKANNT;
		}
	}

	private void analysiereTmcPunkte(final SubMonitor subMonitor, final Map<Integer, StrassenKnoten> lc2Sk) {
		final ObjektFactory objektFactory = verbindungsCache.getObjektFactory();

		sk2TmcPunkt = new HashMap<>();
		tmcPunkt2sk = new HashMap<>();

		final List<TmcPunkt> tmcPunkte = (List<TmcPunkt>) objektFactory.bestimmeModellobjekte(TmcPunkt.PID);
		final Collection<SystemObject> tmcPunkteSO = new ArrayList<>(tmcPunkte.size());
		for (final SystemObjekt objekt : tmcPunkte) {
			tmcPunkteSO.add(objekt.getSystemObject());
		}
		final DataModel dataModel = objektFactory.getDav().getDataModel();
		final AttributeGroup tmcCodeAtg = dataModel.getAttributeGroup(KdTmcLocationCode.PID);

		blockAbruf(dataModel, dataModel.getType(TmcPunkt.PID), tmcCodeAtg);

		// if (isBenutzeBlockAbruf()) {
		// dataModel.getConfigurationData(tmcPunkteSO, tmcCodeAtg);
		// } else {
		// for (final SystemObject so : tmcPunkteSO) {
		// so.getConfigurationData(tmcCodeAtg);
		// }
		// }

		final int anzahlTmcPunkte = tmcPunkte.size();
		final SubMonitor subMonitor4 = subMonitor.newChild(40000);
		subMonitor4.beginTask("Analysiere TMC-Punkte", anzahlTmcPunkte);

		int i = 1;
		for (final TmcPunkt p : tmcPunkte) {
			subMonitor4.subTask(new StringBuffer("Analysiere TMC-Punkt ").append(i).append(" von ")
					.append(anzahlTmcPunkte).append(": ").append(p.getName()).toString());

			final KdTmcLocationCode.Daten tmcLocationCodeDatum = p.getKdTmcLocationCode().getDatum();
			if (tmcLocationCodeDatum != null) {
				final StrassenKnoten sk = lc2Sk.get(tmcLocationCodeDatum.getTmcLocationCode().getValue());
				if (sk != null) {
					sk2TmcPunkt.put(sk, p);
					tmcPunkt2sk.put(p, sk);
				}
			}
			i++;
		}
	}

	/**
	 * Liefert den Netz-Cache zurück, von welchem dieser Cache abhängig ist.
	 *
	 * @return den Netz-Cache, von welchem dieser Cache abhängig ist
	 */
	protected final NetzCache getNetzCache() {
		return netzCache;
	}

	private void cacheLocationCode(final AeusseresStrassenSegment aess) {
		final KdAeusseresStrassenSegment.Daten assKd = aess.getKdAeusseresStrassenSegment().getDatum();

		if ((assKd != null) && (assKd.getTmcPrimaerOrt() != null)) {
			final TmcPunkt tmcPunkt = assKd.getTmcPrimaerOrt();
			aess2TmcPunkt.put(aess, tmcPunkt);
			tmcPunkt2aess.put(tmcPunkt, aess);

			// AeussereStrassensegmente-Map
			if ((tmcPunkt.getKdTmcLocationCode() != null) && (tmcPunkt.getKdTmcLocationCode().getDatum() != null)) {
				final KdTmcLocationCode.Daten locationCodeDaten = tmcPunkt.getKdTmcLocationCode().getDatum();
				final AttTmcLocationCode locationCode = locationCodeDaten.getTmcLocationCode();
				if (locationCode != null) {
					Set<AeusseresStrassenSegment> aessSet = tmcLocationCode2aess.get(locationCode.getValue());
					if (aessSet == null) {
						aessSet = new HashSet<>();
						tmcLocationCode2aess.put(locationCode.getValue(), aessSet);
					}
					aessSet.add(aess);
					aess2TmcLocationCode.put(aess, locationCode.getValue());
				}
			}

			// Strassen-Map
			final KdStrassenSegment.Daten ssKd = aess.getKdStrassenSegment().getDatum();
			if ((ssKd != null) && (ssKd.getGehoertZuStrasse() != null)) {
				final Strasse strasse = ssKd.getGehoertZuStrasse();
				if (strasse2TmcLocationCode.get(strasse) == null) {
					if ((tmcPunkt.getKdTmcPunkt() != null) && (tmcPunkt.getKdTmcPunkt().getDatum() != null)
							&& (tmcPunkt.getKdTmcPunkt().getDatum().getIstTeilvonTmcLinie() != null)) {
						final TmcLinie tmcLinie = tmcPunkt.getKdTmcPunkt().getDatum().getIstTeilvonTmcLinie();
						strasse2TmcLinie.put(strasse, tmcLinie);
						tmcLinie2strasse.put(tmcLinie, strasse);
						if ((tmcLinie.getKdTmcLocationCode() != null)
								&& (tmcLinie.getKdTmcLocationCode().getDatum() != null)
								&& (tmcLinie.getKdTmcLocationCode().getDatum().getTmcLocationCode() != null)) {
							final AttTmcLocationCode locationCode = tmcLinie.getKdTmcLocationCode().getDatum()
									.getTmcLocationCode();
							if (locationCode != null) {
								strasse2TmcLocationCode.put(strasse, locationCode.getValue());
							}
						}
					}
				}
			}
		}
	}

	/**
	 * Liefert die zu einer gegebenen Straße dazugehörige TMC-Linie zurück.
	 *
	 * @param strasse die Straße
	 *
	 * @return die zu einer gegebenen Straße dazugehörige TMC-Linie
	 */
	public TmcLinie getTmcLinie(final Strasse strasse) {
		ensureInit();
		return strasse2TmcLinie.get(strasse);
	}

	/**
	 * Liefert den zu einer gegeben Straße dazugehörigen TMC-Location-Code zurück.
	 *
	 * @param strasse die Straße
	 *
	 * @return den zu einer gegeben Straße dazugehörigen TMC-Location-Code
	 */
	public Integer getTmcLocationCode(final Strasse strasse) {
		ensureInit();
		return strasse2TmcLocationCode.get(strasse);
	}

	/**
	 * Liefert die zu einer gegebenen TMC-Linie dazugehörige Straße zurück.
	 *
	 * @param tmcLinie die TMC-Linie
	 *
	 * @return die zu einer gegebenen TMC-Linie dazugehörige Straße
	 */
	public Strasse getStrasse(final TmcLinie tmcLinie) {
		ensureInit();
		return tmcLinie2strasse.get(tmcLinie);
	}

	/**
	 * Liefert den zu einem gegeben äußeren Straßensegment dazugehörigen
	 * TMC-Location-Code zurück.
	 *
	 * @param aess das äußere Straßensegment
	 *
	 * @return den zu einem gegeben äußeren Straßensegment dazugehörigen
	 *         TMC-Location-Code
	 */
	public int getTmcLocationCode(final AeusseresStrassenSegment aess) {
		ensureInit();
		return aess2TmcLocationCode.get(aess);
	}

	/**
	 * Liefert das zu einem gegeben TMC-Location-Code dazugehörigen äußere
	 * Straßensegment zurück.
	 *
	 * @param tmcLocationCode der TMC-Location-Code
	 *
	 * @return das zu einem gegeben TMC-Location-Code dazugehörigen äußere
	 *         Straßensegment
	 */
	public Set<AeusseresStrassenSegment> getAeusseresStrassenSegment(final int tmcLocationCode) {
		ensureInit();
		return tmcLocationCode2aess.get(tmcLocationCode);
	}

	/**
	 * Liefert den zu einem gegeben Straßenknoten dazugehörigen TMC-Punkt zurück.
	 *
	 * @param knoten der Straßenknoten
	 *
	 * @return den zu einem gegeben Straßenknoten dazugehörigen TMC-Punkt
	 */
	public TmcPunkt getTmcLocationCode(final StrassenKnoten knoten) {
		ensureInit();
		return sk2TmcPunkt.get(knoten);
	}

	/**
	 * Liefert den zu einem gegeben TMC-Punkt dazugehörigen Straßenknoten zurück.
	 *
	 * @param tmcPunkt der TMC-Punkt
	 *
	 * @return den zu einem gegeben TMC-Punkt dazugehörigen Straßenknoten
	 */
	public StrassenKnoten getStrassenKnoten(final TmcPunkt tmcPunkt) {
		ensureInit();
		return tmcPunkt2sk.get(tmcPunkt);
	}

	/**
	 * Liefert den zu einem gegeben äußeren Straßensegment dazugehörigen TMC-Punkt
	 * zurück.
	 *
	 * @param segment das äußere Straßensegment
	 *
	 * @return den zu einem gegeben Straßenknoten dazugehörigen TMC-Punkt
	 */
	public TmcPunkt getTmcPunkt(final AeusseresStrassenSegment segment) {
		ensureInit();
		return aess2TmcPunkt.get(segment);
	}

	/**
	 * Liefert das zu einem gegeben TMC-Punkt dazugehörigen äußere Straßensegment
	 * zurück.
	 *
	 * @param tmcPunkt der TMC-Punkt
	 *
	 * @return den zu einem gegeben TMC-Punkt dazugehörigen Straßenknoten
	 */
	public AeusseresStrassenSegment getAeussereStrassenSegment(final TmcPunkt tmcPunkt) {
		ensureInit();
		return tmcPunkt2aess.get(tmcPunkt);
	}

	public int compareStrasse(final Strasse strasse1, final Strasse strasse2) {
		return compareTmcLinien(getTmcLinie(strasse1), getTmcLinie(strasse2));
	}

	public static final int compareTmcLinien(final TmcLinie tmcLinie1, final TmcLinie tmcLinie2) {
		if ((tmcLinie1 != null) && (tmcLinie2 != null)) {

			// Teste Nachfolger
			TmcLinie nachfolger = tmcLinie1;
			final Set<String> loopPids = new LinkedHashSet<>();
			while ((nachfolger != null) && !nachfolger.getPid().equals(tmcLinie2.getPid())) {
				loopPids.add(nachfolger.getPid());
				if ((nachfolger.getKdTmcLinie() != null) && (nachfolger.getKdTmcLinie().getDatum() != null)) {
					nachfolger = nachfolger.getKdTmcLinie().getDatum().getNachfolger();
					if ((nachfolger != null) && loopPids.contains(nachfolger.getPid())) {
						nachfolger = null;
					}
				}
			}
			if (nachfolger != null) {
				return -1;
			}

			// Teste Vorgänger
			TmcLinie vorgaenger = tmcLinie1;
			loopPids.clear();
			while ((vorgaenger != null) && !vorgaenger.getPid().equals(tmcLinie2.getPid())) {
				loopPids.add(vorgaenger.getPid());
				if ((vorgaenger.getKdTmcLinie() != null) && (vorgaenger.getKdTmcLinie().getDatum() != null)) {
					vorgaenger = vorgaenger.getKdTmcLinie().getDatum().getVorgaenger();
					if ((vorgaenger != null) && loopPids.contains(vorgaenger.getPid())) {
						vorgaenger = null;
					}
				}
			}
			if (vorgaenger != null) {
				return +1;
			}
		}
		return 0;
	}

	/**
	 * Erfragt anhand der aktuellen Parameter einer RDS-Meldung die
	 * richtungsgeordnete Liste der Mengen der Strassensegmente, auf die sie sich
	 * beziehen (koennte).
	 *
	 * @param meldung die RDS-Meldung selbst.
	 * @return ggf. leere die Liste von Mengen von Strassensegmenten. Fuer jeden
	 *         Location-Code innerhalb der RDS-Meldung wird eine Menge von
	 *         Strassensegmenten zurueckgeliefert, auf die sich der Location-Code
	 *         beziehen koennte.
	 */
	private List<Set<AeusseresStrassenSegment>> getSegmentMengenListOf(final RdsMeldung meldung) {
		final List<Set<AeusseresStrassenSegment>> segmentMengen = new ArrayList<>();
		ensureInit();

		PdRdsMeldung.Daten rdsMeldung = meldung.getPdRdsMeldung().getDatum(PdRdsMeldung.Aspekte.RdsEmpfangen);

		/**
		 * Fallback:
		 */
		if (!rdsMeldung.dContainsDaten()) {
			rdsMeldung = meldung.getPdRdsMeldung().getDatum();
		}

		if (!rdsMeldung.dContainsDaten()) {
			return Collections.emptyList(); // Keine Daten im Parameter
		}

		final Feld<AtlRdsLocation> locationFeld = rdsMeldung.getVersion().getVerkehrsInformationen().getLocationDaten()
				.getRDSLocation();

		AttRdsTMCRichtung strassenRichtungAnhandDerAnschlussStellenOrdnungPositiv = AttRdsTMCRichtung.ZUSTAND_0_POSITIV;
		String alteAusfahrt = null;
		for (final AtlRdsLocation location : locationFeld) {
			final AttTmcLocationCode tmcLocationCode = location.getLocationCode();
			if (tmcLocationCode != null) {
				final Set<AeusseresStrassenSegment> segmenteFound = tmcLocationCode2aess
						.get(tmcLocationCode.getValue());
				if ((segmenteFound != null) && !segmenteFound.isEmpty()) {
					segmentMengen.add(segmenteFound);
				}
			}
			if (alteAusfahrt != null) {
				/**
				 * d.h. wir sind mindestens bei der 2. Iteration
				 */
				try {
					final int alteNr = Integer.parseInt(alteAusfahrt);
					final int neueNr = Integer.parseInt(location.getAusfahrtNummer());
					strassenRichtungAnhandDerAnschlussStellenOrdnungPositiv = new Integer(alteNr).compareTo(neueNr) < 0
							? AttRdsTMCRichtung.ZUSTAND_0_POSITIV
							: AttRdsTMCRichtung.ZUSTAND_1_NEGATIV;
				} catch (final NumberFormatException ex) {
					Debug.getLogger().warning("Konnte Richtung von " + meldung + " nicht bestimmen", ex);
					continue;
				}
			}
			alteAusfahrt = location.getAusfahrtNummer();
		}

		final AttRdsTMCRichtung richtung = rdsMeldung.getVersion().getVerkehrsInformationen().getLocationDaten()
				.getTMCRichtung();
		if ((segmentMengen.size() > 1) && (richtung != null)) {
			if (!richtung.equals(strassenRichtungAnhandDerAnschlussStellenOrdnungPositiv)) {
				Collections.reverse(segmentMengen);
			}
		}

		return segmentMengen;
	}

	public final List<AeusseresStrassenSegment> getSegmenteOf(final RdsMeldung meldung) {

		final List<Set<AeusseresStrassenSegment>> segmentMengenList = getSegmentMengenListOf(meldung);

		if (segmentMengenList.isEmpty()) {
			return Collections.emptyList();
		}

		final List<AeusseresStrassenSegment> assList = new ArrayList<>();
		for (int i = 0; i < segmentMengenList.size(); i++) {
			if (segmentMengenList.size() > (i + 1)) {
				final Set<AeusseresStrassenSegment> mengeMoeglicheVorganger;
				if (!assList.isEmpty()) {
					mengeMoeglicheVorganger = new HashSet<>();
					mengeMoeglicheVorganger.add(assList.get(assList.size() - 1));
				} else {
					mengeMoeglicheVorganger = segmentMengenList.get(i);
				}
				final Set<AeusseresStrassenSegment> mengeMoeglicheNachfolger = segmentMengenList.get(i + 1);
				verkette(assList, mengeMoeglicheVorganger, mengeMoeglicheNachfolger);
			}
		}

		if (assList.size() > 1) {
			assList.remove(0);
		}

		return assList;
	}

	/**
	 * Erfragt, ob zwei aeussere Strassensegmente aufeinander folgen.
	 *
	 * @param moeglicherVorgaenger Aeusseres Strassensegment 1.
	 * @param moeglicherNachfolger Aeusseres Strassensegment 2.
	 * @return ob die uebergebenen aeussere Strassensegmente aufeinander folgen.
	 */
	private static final boolean match(final AeusseresStrassenSegment moeglicherVorgaenger,
			final AeusseresStrassenSegment moeglicherNachfolger) {
		boolean result = false;
		if ((moeglicherVorgaenger != null) && (moeglicherNachfolger != null)) {
			final KdAeusseresStrassenSegment kdAssVorgaenger = moeglicherVorgaenger.getKdAeusseresStrassenSegment();
			final KdAeusseresStrassenSegment kdAssNachfolger = moeglicherNachfolger.getKdAeusseresStrassenSegment();
			if ((kdAssVorgaenger != null) && (kdAssNachfolger != null) && (kdAssVorgaenger.getDatum() != null)
					&& (kdAssNachfolger.getDatum() != null) && (kdAssVorgaenger.getDatum().getNachKnoten() != null)
					&& (kdAssNachfolger.getDatum().getVonKnoten() != null)) {
				result = kdAssVorgaenger.getDatum().getNachKnoten().equals(kdAssNachfolger.getDatum().getVonKnoten())
						&& !kdAssVorgaenger.getDatum().getVonKnoten()
								.equals(kdAssNachfolger.getDatum().getNachKnoten());
			}
		}
		return result;
	}

	/**
	 * Erfragt die ggf. leere Liste zweier aufeinanderfolgender aesserer
	 * Strassensegmente.
	 *
	 * @param assList                  die Liste der beteiligten Straßensegmente
	 * @param mengeMoeglicheVorganger  Menge moeglicher Vorgaenger.
	 * @param mengeMoeglicheNachfolger Menge moeglicher Nachfolger.
	 */
	private static final void verkette(final List<AeusseresStrassenSegment> assList,
			final Set<AeusseresStrassenSegment> mengeMoeglicheVorganger,
			final Set<AeusseresStrassenSegment> mengeMoeglicheNachfolger) {
		for (final AeusseresStrassenSegment assVor : mengeMoeglicheVorganger) {
			for (final AeusseresStrassenSegment assNach : mengeMoeglicheNachfolger) {
				if (match(assVor, assNach)) {
					if (!assList.isEmpty()) {
						if (!assList.get(assList.size() - 1).equals(assVor)) {
							assList.add(assVor);
						}
					} else {
						assList.add(assVor);
					}
					assList.add(assNach);
					return;
				}
			}
		}
	}

}
