/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.cache;

import java.util.ArrayList;
import java.util.List;

import de.bsvrz.sys.funclib.bitctrl.modell.tmtmcglobal.attribute.AttTmcOrtsTyp;
import de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc.TmcLinieWrapper;

/**
 * Filter f&uuml;r TMC-Linienlokationen.
 *
 * @author BitCtrl Systems GmbH, gysi
 */
public class TmcLinienFilter {

	/** Liste der Verwaltungsbereiche, welche diesen Filter passieren. */
	private List<String> verwaltungsBereiche;

	/** Liste der Ortstypen, welche diesen Filter passieren. */
	private List<AttTmcOrtsTyp> ortsTypen;

	/**
	 * Der Default-Konstruktor.
	 *
	 * <p>
	 * Standardmäßig wird kein Verwaltungsbereich gesetzt, was bedeutet, dass
	 * sämtliche TMC-Linien (unabhängig vom Verwaltungsbereich) diesen Filter
	 * passieren.
	 * </p>
	 *
	 * <p>
	 * Bei den Ortstypen werden standardmäßig alle Linien-Ortstypen gesetzt außer
	 * dem Ortstyp 'Straße:(Straße 1.Ordnung) Bundesautobahn' (Value: 201001).
	 * </p>
	 */
	public TmcLinienFilter() {
		verwaltungsBereiche = new ArrayList<>();
		ortsTypen = new ArrayList<>();
		for (final AttTmcOrtsTyp typ : AttTmcOrtsTyp.getZustaende()) {
			if (((typ.getValue() / 100000) == 2) && (typ.getValue() != 201001)) {
				ortsTypen.add(typ);
			}
		}
	}

	/**
	 * Der Konstruktor.
	 *
	 * <p>
	 * Bei den Ortstypen werden standardmäßig alle Linien-Ortstypen gesetzt außer
	 * dem Ortstyp 'Straße:(Straße 1.Ordnung) Bundesautobahn' (Value: 201001).
	 * </p>
	 *
	 * @param verwaltungsBereich der alleinige Verwaltungsbereich
	 */
	public TmcLinienFilter(final String verwaltungsBereich) {
		this();
		setVerwaltungsBereich(verwaltungsBereich);
	}

	/**
	 * Der Konstruktor.
	 *
	 * @param verwaltungsBereiche die Liste der Verwaltungsbereiche, welche diesen
	 *                            Filter passieren sollen
	 * @param ortsTypen           die Liste der Ortstypen, welchen diesen Filter
	 *                            passieren sollen
	 */
	public TmcLinienFilter(final List<String> verwaltungsBereiche, final List<AttTmcOrtsTyp> ortsTypen) {
		this();
		if (verwaltungsBereiche != null) {
			this.verwaltungsBereiche = new ArrayList<>(verwaltungsBereiche);
		}
		if (ortsTypen != null) {
			this.ortsTypen = new ArrayList<>(ortsTypen);
		}
	}

	/**
	 * Liefert <code>true</code> zurück, wenn der Verwaltungsbereich der gegebene
	 * TMC-Linie diesen Filter passieren darf.
	 *
	 * @param wrapper die TMC-Linie
	 *
	 * @return <code>true</code>, wenn der Verwaltungsbereich der gegebene TMC-Linie
	 *         diesen Filter passieren darf, ansonsten <code>false</code>
	 */
	private boolean isInVerwaltungsBereich(final TmcLinieWrapper wrapper) {
		return (verwaltungsBereiche.size() == 0) || verwaltungsBereiche.contains(wrapper.getVerwaltungsBereich());
	}

	/**
	 * Liefert <code>true</code> zurück, wenn der Ortstyp der gegebenen TMC-Linie
	 * diesen Filter passieren darf.
	 *
	 * @param wrapper die TMC-Linie
	 *
	 * @return <code>true</code>, wenn der Ortstyp der gegebenen TMC-Linie diesen
	 *         Filter passieren darf, ansonsten <code>false</code>
	 */
	private boolean isInOrtstTypen(final TmcLinieWrapper wrapper) {
		return (ortsTypen.size() == 0) || ortsTypen.contains(wrapper.getTmcOrtsTyp());
	}

	/**
	 * Liefert <code>true</code> zurück, wenn die gegebene TMC-Linie diesen Filter
	 * passiert.
	 *
	 * @param wrapper die TMC-Linie
	 *
	 * @return <code>true</code>, wenn die gegebene TMC-Linie diesen Filter
	 *         passiert, ansonsten <code>false</code>
	 */
	public boolean passiertFilter(final TmcLinieWrapper wrapper) {
		if (wrapper != null) {
			return isInVerwaltungsBereich(wrapper) && isInOrtstTypen(wrapper);
		}
		return false;
	}

	/**
	 * Setzt den alleinigen Verwaltungsbereich dieses Filters.
	 *
	 * @param verwaltungsBereich der alleinige Verwaltungsbereich
	 */
	public void setVerwaltungsBereich(final String verwaltungsBereich) {
		verwaltungsBereiche = new ArrayList<>();
		if ((verwaltungsBereich != null) && !"".equals(verwaltungsBereich)) {
			verwaltungsBereiche.add(verwaltungsBereich);
		}
	}

	/**
	 * Fügt den gegebenen Verwaltungsbereich diesem Filter hinzu.
	 *
	 * @param verwaltungsBereich der neue Verwaltungsbereich
	 */
	public void addVerwaltungsBereich(final String verwaltungsBereich) {
		if (!verwaltungsBereiche.contains(verwaltungsBereich)) {
			verwaltungsBereiche.add(verwaltungsBereich);
		}
	}

	/**
	 * Entfernt den gegebenen Verwaltungsbereich von diesem Filter.
	 *
	 * @param verwaltungsBereich der zu entfernende Verwaltungsbereich
	 */
	public void removeVerwaltungsBereich(final String verwaltungsBereich) {
		verwaltungsBereiche.remove(verwaltungsBereich);
	}
}
