/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.rdstmc;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import de.bsvrz.sys.funclib.bitctrl.modell.att.Feld;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Zahl;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsEreignisDaten;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AtlRdsEreignisQuantitaet;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsEreignisDauer;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmstglobal.attribute.AttRdsEreignisKategorie;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.attribute.AttRdsEmpfehlungsCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.attribute.AttRdsEreignisCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkexlmsttmccodes.attribute.AttRdsVorhersageCode;
import de.bsvrz.sys.funclib.bitctrl.modell.util.cache.RdsCache;

/**
 * Wrapper, welcher für ein EreignisDaten-SystemObjekt sämtliche Daten vorhält.
 *
 * @author BitCtrl Systems GmbH, Andreas Meissner
 *
 * @see AtlRdsEreignisDaten
 */
public class EreignisWrapper extends AbstractWrapper {

	/** Platzhalter für einen Quantitäts-Wert. */
	private static final String PLATZHALTER_QUANTITAET = "(Q)";

	/** Platzhalter für die betroffenen Länge. */
	private static final String PLATZHALTER_BETROFFENE_LAENGE = "(L)";

	/** Die in diesem Wrapper gekapselte Ereignisdaten. */
	private final AtlRdsEreignisDaten rdsEreignisDaten;

	/**
	 * Der RdsMeldungsWrapper, welcher diesen EreignisWrapper enthält.
	 */
	private final RdsMeldungWrapper rdsMeldungWrapper;

	/**
	 * Numerischer Wert der Ereignis-Kategorie.
	 *
	 * <p>
	 * Kategorien 1-31 stehen für Ereignisse, Kategorien 32-39 für Vorhersagen,
	 * Kategorien 40-49 sind unbenutzt und Kategorie 50 steht für Empfehlungen.
	 * </p>
	 */
	private AttRdsEreignisKategorie kategorie;

	/**
	 * Dauer des Ereignisses.
	 *
	 * <p>
	 * Wertebereich:
	 * </p>
	 * <ul>
	 * <li>0: Unbekannt</li>
	 * <li>1: Dynamisch</li>
	 * <li>2: Länger</li>
	 * </ul>
	 */
	private AttRdsEreignisDauer dauer;

	/**
	 * Die empfohlene Dauer des Ereignisses.
	 */
	private AttRdsEreignisDauer empfohleneDauer;

	/**
	 * Die betroffene Länge dieses Ereignisses.
	 *
	 * <p>
	 * Dieser Wert entspricht der Quantität LEN: betroffene Länge [km].
	 * </p>
	 */
	private BetroffeneLaenge betroffeneLaenge;

	/**
	 * Quantitätsangabe zum Ereignis.
	 *
	 * <p>
	 * Die Quantität ist definiert durch eine Kennung, eine Beschreibung, den Wert
	 * und die Einheit.
	 * </p>
	 *
	 * Zulässige Quantitäten sind:
	 * <ul>
	 * <li>Q00: Anzahl [1, 2, ..., 30, 32, 34, 36]</li>
	 * <li>Q01: Anzahl [1, 2, ..., 10, 20, ..., 100, 150, ..., 1000]</li>
	 * <li>VIS: Sichtweite [m]</li>
	 * <li>POC: Parkplatzbelegung [%]</li>
	 * <li>AVV: Durchschnittsgeschwindigkeit [km/h]</li>
	 * <li>WDS: durchschnittliche Windgeschwindigkeit [km/h]</li>
	 * <li>Q05: Verzögerung [Minuten/Stunde(n)]</li>
	 * <li>ATE: Lufttemperatur [°C]</li>
	 * <li>Q07: Zeit</li>
	 * <li>WEI: Fahrzeuggewicht [t]</li>
	 * <li>Q09: Begrenzung [m]</li>
	 * <li>PDD: Niederschlagshöhe [mm]</li>
	 * <li>FMH: FM Frequenz [MHz]</li>
	 * <li>AMH: AM Frequenz [kHz]</li>
	 * </ul>
	 */
	private RdsQuantitaetWrapper quantitaet;

	/**
	 * Der String-Wert der aktuell gesetzten Quantität.
	 */
	private String quantitaetStringWert;

	/**
	 * Der Integer-Wert der aktuell gesetzten Quantität.
	 *
	 * <p>
	 * Beachte: Double-Werte werden ebenfalls als Integer hier abgespeichert.
	 * </p>
	 */
	private Integer quantitaetIntWert;

	/**
	 * Der abhängig von der Kategorie gesetzte Code. Kann ein Ereignis-, Vorhersage-
	 * oder Empfehlungs-Code sein.
	 */
	private Zahl<Integer> code;

	/**
	 * Der Ausgabe-Text für dieses Ereignis. Der Text ist abhängig von dem gesetzten
	 * Ereignis-, Vorhersage- oder Empfehlungs-Code sowie möglicherweise gesetzten
	 * Quantitäten.
	 */
	private String text;

	/**
	 * Der Default-Konstruktor.
	 *
	 * @param rdsMeldungWrapper der RdsMeldungWrapper, der diesen EreignisWrapper
	 *                          enthält, darf nicht <code>null</code> sein
	 */
	public EreignisWrapper(final RdsMeldungWrapper rdsMeldungWrapper) {
		assert null != rdsMeldungWrapper;
		this.rdsMeldungWrapper = rdsMeldungWrapper;
		rdsEreignisDaten = null;
		kategorie = AttRdsEreignisKategorie.ZUSTAND_1_VERKEHRSLAGE;
		dauer = AttRdsEreignisDauer.ZUSTAND_0_UNBEKANNT;
		empfohleneDauer = AttRdsEreignisDauer.ZUSTAND_0_UNBEKANNT;
		code = AttRdsEreignisCode.ZUSTAND_1_VERKEHRSBEHINDERUNG;
		quantitaet = null;
		quantitaetIntWert = null;
		quantitaetStringWert = "";
		betroffeneLaenge = null;
		setText(null, true);
	}

	/**
	 * Der Konstruktor.
	 *
	 * @param rdsMeldungWrapper der RdsMeldungWrapper, der diesen EreignisWrapper
	 *                          enthält, darf nicht <code>null</code> sein
	 * @param rdsEreignisDaten  die RDS-Ereignisdaten, darf nicht <code>null</code>
	 *                          sein
	 */
	public EreignisWrapper(final RdsMeldungWrapper rdsMeldungWrapper, final AtlRdsEreignisDaten rdsEreignisDaten) {
		assert null != rdsEreignisDaten;
		assert null != rdsMeldungWrapper;
		this.rdsMeldungWrapper = rdsMeldungWrapper;
		this.rdsEreignisDaten = rdsEreignisDaten;

		// Codes
		final RdsCache rdsCache = rdsMeldungWrapper.getRdsCache();
		if ((rdsEreignisDaten.getEreignisCode() != null)
				&& (rdsEreignisDaten.getEreignisCode() != AttRdsEreignisCode.ZUSTAND_0_NICHT_DEFINIERT)) {
			code = rdsEreignisDaten.getEreignisCode();
		} else if ((rdsEreignisDaten.getVorhersageCode() != null)
				&& (rdsEreignisDaten.getVorhersageCode() != AttRdsVorhersageCode.ZUSTAND_0_NICHT_DEFINIERT)) {
			code = rdsEreignisDaten.getVorhersageCode();
		} else if ((rdsEreignisDaten.getEmpfehlungsCode() != null)
				&& (rdsEreignisDaten.getEmpfehlungsCode() != AttRdsEmpfehlungsCode.ZUSTAND_0_NICHT_DEFINIERT)) {
			code = rdsEreignisDaten.getEmpfehlungsCode();
		}
		if (code == null) {
			code = AttRdsEreignisCode.ZUSTAND_1_VERKEHRSBEHINDERUNG;
			rdsMeldungWrapper.setzeUngueltig("Ereignis-Code");
		}
		final RdsTmcCodeWrapper rds = rdsCache.getRdsTmcCodeWrapper(code);

		// Dauer
		dauer = rdsEreignisDaten.getEreignisDauer();
		empfohleneDauer = AttRdsEreignisDauer.ZUSTAND_0_UNBEKANNT;
		if (rds != null) {
			empfohleneDauer = rds.getEreignisdauer();
		}
		if (dauer == null) {
			dauer = empfohleneDauer;
			rdsMeldungWrapper.setzeUngueltig("Ereignis-Dauer");
		}

		// Kategorie
		kategorie = rdsEreignisDaten.getEreignisKategorie();
		if (kategorie == null) {
			kategorie = AttRdsEreignisKategorie.ZUSTAND_1_VERKEHRSLAGE;
		}
		if ((rds != null) && !rdsCache.getCodes(kategorie).contains(code)) {
			kategorie = rds.getKategorie();
			rdsMeldungWrapper.setzeUngueltig("Ereignis-Kategorie");
		}

		// Betroffene Länge und Quantität
		quantitaet = null;
		quantitaetIntWert = null;
		quantitaetStringWert = "";
		betroffeneLaenge = null;
		final List<String> gueltigeKennungen = Arrays.asList(QuantitaetenUtil.GUELTIGE_QUANTITAETEN_KENNUNGEN);
		final Feld<AtlRdsEreignisQuantitaet> ereignisQuantitaeten = rdsEreignisDaten.getEreignisQuantitaet();
		if (ereignisQuantitaeten != null) {
			for (final AtlRdsEreignisQuantitaet q : ereignisQuantitaeten) {
				final RdsQuantitaetWrapper quantitaetWrapper = rdsCache
						.getRdsQuantitaetWrapper(q.getQuantitaetsKennung());
				final String kennung = quantitaetWrapper.getKennung();
				if ((betroffeneLaenge == null) && QuantitaetenUtil.KENNUNG_BETROFFENE_LAENGE.equals(kennung)) {
					betroffeneLaenge = BetroffeneLaenge.getValue(q.getQuantitaetsWert());
				} else if ((quantitaet == null) && gueltigeKennungen.contains(kennung)) {

					// ermittel Quantitaet
					if (QuantitaetenUtil.KENNUNG_ANZAHL_GROSS.equals(kennung)) {
						if ((rds != null) && (rds.getQuantitaetReferenz() != null)) {
							quantitaet = rdsCache.getRdsQuantitaetWrapper(rds.getQuantitaetReferenz());
						}
					} else {
						quantitaet = rdsCache.getRdsQuantitaetWrapper(kennung);
					}

					// ermittel Wert
					if (QuantitaetenUtil.KENNUNG_VERZOEGERUNG.equals(kennung)) {
						setQuantitaetIntWert(QuantitaetenUtil.getVerzoegerungInt(
								q.getQuantitaetsWert() + " " + q.getQuantitaetsEinheit()), true, false);
					} else {
						setQuantitaetStringWert(q.getQuantitaetsWert(), true, false);
					}
				}
			}
		} else {
			rdsMeldungWrapper.setzeUngueltig("Ereignis-Quantitäten");
		}

		setText(null, false);
	}

	/**
	 * Der Konstruktor.
	 *
	 * @param rdsMeldungWrapper der RdsMeldungWrapper, der diesen EreignisWrapper
	 *                          enthält, darf nicht <code>null</code> sein
	 * @param wrapper           ein anderer EreignisWrapper, darf nicht
	 *                          <code>null</code> sein
	 */
	public EreignisWrapper(final RdsMeldungWrapper rdsMeldungWrapper, final EreignisWrapper wrapper) {
		assert null != wrapper;
		this.rdsMeldungWrapper = rdsMeldungWrapper;
		rdsEreignisDaten = wrapper.rdsEreignisDaten;
		kategorie = AttRdsEreignisKategorie.getZustand(wrapper.kategorie.getValue());
		dauer = AttRdsEreignisDauer.getZustand(wrapper.dauer.getValue());
		empfohleneDauer = AttRdsEreignisDauer.getZustand(wrapper.empfohleneDauer.getValue());
		code = wrapper.code;
		quantitaet = null;
		if (wrapper.quantitaet != null) {
			quantitaet = wrapper.quantitaet;
		}
		quantitaetIntWert = null;
		if (wrapper.quantitaetIntWert != null) {
			quantitaetIntWert = new Integer(wrapper.quantitaetIntWert);
		}
		quantitaetStringWert = "";
		if (wrapper.quantitaetStringWert != null) {
			quantitaetStringWert = new String(wrapper.quantitaetStringWert);
		}
		betroffeneLaenge = wrapper.betroffeneLaenge;
		text = new String(wrapper.text);
	}

	/**
	 * Liefert das in diesem Objekt gekapselte RdsEreignisDaten-Systemobjekt zurück.
	 *
	 * @return das in diesem Objekt gekapselte RdsEreignisDaten-Systemobjekt
	 */
	public AtlRdsEreignisDaten getRdsEreignisDaten() {
		return rdsEreignisDaten;
	}

	/**
	 * Liefert den RdsMeldungWrapper zurück, welcher diesen EreignisWrapper enthält.
	 *
	 * @return den RdsMeldungWrapper, welcher diesen EreignisWrapper enthält,
	 *         niemals <code>null</code>
	 */
	public RdsMeldungWrapper getParent() {
		return rdsMeldungWrapper;
	}

	/**
	 * Setzt die Dauer dieses Ereignisses.
	 *
	 * @param neuerWert die neue Dauer, darf nicht <code>null</code> sein
	 */
	public void setDauer(final AttRdsEreignisDauer neuerWert) {
		assert null != neuerWert;
		if (neuerWert != dauer) {
			final AttRdsEreignisDauer alterWert = dauer;
			dauer = neuerWert;
			firePropertyChange("dauer", alterWert, neuerWert);
			rdsMeldungWrapper.setBits();
		}
	}

	/**
	 * Liefert die Dauer für dieses Ereignis zurück.
	 *
	 * @return Dauer dieses Ereignisses, niemals <code>null</code>
	 *
	 */
	public AttRdsEreignisDauer getDauer() {
		return dauer;
	}

	/**
	 * Liefert die empfohlene Dauer für dieses Ereignis zurück.
	 *
	 * @return die empfohlene Dauer dieses Ereignisses, niemals <code>null</code>
	 *
	 */
	public AttRdsEreignisDauer getEmpfohleneDauer() {
		return empfohleneDauer;
	}

	/**
	 * Setzt die empfohlene Dauer für dieses Ereignis.
	 *
	 * @param neuerWert der neue Wert für die empfohlene Dauer
	 */
	private void setEmpfohleneDauer(final AttRdsEreignisDauer neuerWert) {
		assert null != neuerWert;
		if (neuerWert != empfohleneDauer) {
			final AttRdsEreignisDauer alterWert = empfohleneDauer;
			empfohleneDauer = neuerWert;
			firePropertyChange("empfohleneDauer", alterWert, neuerWert);
			rdsMeldungWrapper.setBits();
		}
	}

	/**
	 * Setzt die Kategorie dieses Ereignisses.
	 *
	 * @param neuerWert die neue Kategorie, darf nicht <code>null</code> sein
	 */
	public void setKategorie(final AttRdsEreignisKategorie neuerWert) {
		if (neuerWert == null) {
			return;
		}
		if (neuerWert != kategorie) {
			final AttRdsEreignisKategorie alterWert = neuerWert;
			kategorie = neuerWert;
			firePropertyChange("kategorie", alterWert, neuerWert);

			// setze ggf. einen neuen Code
			final List<Zahl<Integer>> rdsTmcCodeWrapper = rdsMeldungWrapper.getRdsCache().getCodes(neuerWert);
			if (!rdsTmcCodeWrapper.isEmpty()) {
				setCode(rdsTmcCodeWrapper.get(0));
			}
		}
	}

	/**
	 * Liefert die Kategorie dieses Ereignisses zurück.
	 *
	 * <p>
	 * Kategorien 1-31 stehen für Ereignisse, Kategorien 32-39 für Vorhersagen,
	 * Kategorien 40-49 sind unbenutzt und Kategorie 50 steht für Empfehlungen.
	 * </p>
	 *
	 * @return Kategorie dieses Ereignisses
	 */
	public AttRdsEreignisKategorie getKategorie() {
		return kategorie;
	}

	/**
	 * Setzt die Quantität dieses Ereignisses.
	 *
	 * @param neuerWert die neue Quantität
	 */
	private void setQuantitaet(final RdsQuantitaetWrapper neuerWert) {
		if (neuerWert != quantitaet) {
			final RdsQuantitaetWrapper alterWert = quantitaet;
			quantitaet = neuerWert;
			firePropertyChange("quantitaet", alterWert, neuerWert);
			rdsMeldungWrapper.setBits();
			setText(null, true);
		}
		if (neuerWert == null) {
			setQuantitaetIntWert(null);
		} else {
			setQuantitaetStringWert(QuantitaetenUtil.getDefaultWert(neuerWert.getKennung()));
		}
	}

	/**
	 * Liefert die mit diesem Ereignis assoziierte Quantität zurück, oder
	 * <code>null</code>, wenn eine solche nicht gibt.
	 *
	 * @return die mit diesem Ereignis assoziierte Quantität zurück, oder
	 *         <code>null</code>
	 */
	public RdsQuantitaetWrapper getQuantitaet() {
		return quantitaet;
	}

	/**
	 * Setzt den String-Wert der assozierten Quantität dieses Ereignisses.
	 *
	 * @param neuerWert der neue Wert, darf nicht <code>null</code> sein
	 */
	public void setQuantitaetStringWert(final String neuerWert) {
		setQuantitaetStringWert(neuerWert, true, true);
	}

	/**
	 * Setzt den String-Wert der assozierten Quantität dieses Ereignisses.
	 *
	 * @param neuerWert               der neue Wert, darf nicht <code>null</code>
	 *                                sein
	 * @param aktualisiereIntWert     <code>true</code>, wenn der dazugehörige
	 *                                Integer-Wert mit aktualisiert werden soll,
	 *                                ansonsten <code>false</code>
	 * @param firePropertyChangeEvent <code>true</code>, wenn ein
	 *                                PropertyChange-Event gefeuert werden soll,
	 *                                ansonsten <code>false</code>
	 */
	private void setQuantitaetStringWert(final String neuerWert, final boolean aktualisiereIntWert,
			final boolean firePropertyChangeEvent) {
		assert null != neuerWert;
		if (!neuerWert.equals(quantitaetStringWert)) {
			final String alterWert = quantitaetStringWert != null ? new String(quantitaetStringWert) : null;
			quantitaetStringWert = neuerWert;
			if (firePropertyChangeEvent) {
				firePropertyChange("quantitaetStringWert", alterWert, neuerWert);
			}
			if (aktualisiereIntWert) {
				if (quantitaet != null) {
					setQuantitaetIntWert(QuantitaetenUtil.getIntWert(quantitaet.getKennung(), neuerWert), false,
							firePropertyChangeEvent);
				} else {
					setQuantitaetIntWert(null, false, firePropertyChangeEvent);
				}
			}
			setText(null, true);
		}
	}

	/**
	 * Liefert den String-Wert der assozierten Quantität dieses Ereignisses zurück.
	 *
	 * @return den String-Wert der assozierten Quantität dieses Ereignisses
	 */
	public String getQuantitaetStringWert() {
		return quantitaetStringWert;
	}

	/**
	 * Setzt den Integer-Wert der assozierten Quantität dieses Ereignisses.
	 *
	 * @param neuerWert der neue Wert
	 */
	public void setQuantitaetIntWert(final Integer neuerWert) {
		setQuantitaetIntWert(neuerWert, true, true);
	}

	/**
	 * Setzt den Integer-Wert der assozierten Quantität dieses Ereignisses.
	 *
	 * @param neuerWert               der neue Wert
	 * @param aktualisiereStringWert  <code>true</code>, wenn der dazugehörige
	 *                                String-Wert mit aktualisiert werden soll,
	 *                                ansonsten <code>false</code>
	 * @param firePropertyChangeEvent <code>true</code>, wenn ein
	 *                                PropertyChange-Event gefeuert werden soll,
	 *                                ansonsten <code>false</code>
	 */
	private void setQuantitaetIntWert(final Integer neuerWert, final boolean aktualisiereStringWert,
			final boolean firePropertyChangeEvent) {
		if (!Objects.equals(neuerWert, quantitaetIntWert)) {
			final Integer alterWert = quantitaetIntWert;
			quantitaetIntWert = neuerWert;
			if (firePropertyChangeEvent) {
				firePropertyChange("quantitaetIntWert", alterWert, neuerWert);
			}
			if (aktualisiereStringWert) {
				if (quantitaet != null) {
					setQuantitaetStringWert(QuantitaetenUtil.getStringWert(quantitaet.getKennung(), neuerWert), false,
							firePropertyChangeEvent);
				} else {
					setQuantitaetStringWert("", false, firePropertyChangeEvent);
				}
			}
			setText(null, true);
		}
	}

	/**
	 * Liefert den Integer-Wert der assozierten Quantität dieses Ereignisses.
	 *
	 * @return den Integer-Wert der assozierten Quantität dieses Ereignisses
	 */
	public Integer getQuantitaetIntWert() {
		return quantitaetIntWert;
	}

	/**
	 * Setzt den neuen Wert für die betroffene Länge dieses Ereignisses.
	 *
	 * @param neuerWert der neue Wert, darf <code>null</code> sein
	 */
	public void setBetroffeneLaenge(final BetroffeneLaenge neuerWert) {
		setBetroffeneLaenge(neuerWert, true);
	}

	/**
	 * Setzt den neuen Wert für die betroffene Länge dieses Ereignisses.
	 *
	 * @param neuerWert        der neue Wert, darf <code>null</code> sein
	 * @param aktualisiereText <code>true</code>, wenn der Ausgabe-Text aktualisiert
	 *                         werden soll, ansonsten <code>false</code>
	 */
	private void setBetroffeneLaenge(final BetroffeneLaenge neuerWert, final boolean aktualisiereText) {
		if (neuerWert != betroffeneLaenge) {
			final BetroffeneLaenge alterWert = betroffeneLaenge;
			betroffeneLaenge = neuerWert;
			firePropertyChange("betroffeneLaenge", alterWert, neuerWert);
			if (aktualisiereText) {
				setText(null, true);
			}
			rdsMeldungWrapper.setBits();
		}
	}

	/**
	 * Liefert die betroffene Länge dieses Ereignisses zurück.
	 *
	 * @return die betroffene Länge dieses Ereignisses, kann <code>null</code> sein
	 */
	public BetroffeneLaenge getBetroffeneLaenge() {
		return betroffeneLaenge;
	}

	/**
	 * Liefert den aktuell gesetzten Code zurück. Kann ein Ereignis-, Vorhersage-
	 * oder Empfehlungs-Code darstellen.
	 *
	 * @return den aktuell gesetzten Code
	 */
	public Zahl<Integer> getCode() {
		return code;
	}

	/**
	 * Liefert den Ereignis-Code zurück.
	 *
	 * @return den Ereignis-Code, niemals <code>null</code>
	 */
	public AttRdsEreignisCode getEreignisCode() {
		if ((getKategorie().getValue() > 0) && (getKategorie().getValue() < 32)) {
			return (AttRdsEreignisCode) code;
		}
		return AttRdsEreignisCode.ZUSTAND_0_NICHT_DEFINIERT;
	}

	/**
	 * Liefert den Vorhersage-Code zurück.
	 *
	 * @return den Vorhersage-Code, niemals <code>null</code>
	 */
	public AttRdsVorhersageCode getVorhersageCode() {
		if ((getKategorie().getValue() > 31) && (getKategorie().getValue() < 40)) {
			return (AttRdsVorhersageCode) code;
		}
		return AttRdsVorhersageCode.ZUSTAND_0_NICHT_DEFINIERT;
	}

	/**
	 * Liefert den Empfehlungs-Code zurück.
	 *
	 * @return den Empfehlungs-Code, niemals <code>null</code>
	 */
	public AttRdsEmpfehlungsCode getEmpfehlungsCode() {
		if (getKategorie().getValue() == 50) {
			return (AttRdsEmpfehlungsCode) code;
		}
		return AttRdsEmpfehlungsCode.ZUSTAND_0_NICHT_DEFINIERT;
	}

	/**
	 * Setzt den Code. Muss ein Ereignis-, Vorhersage- oder Empfehlungs-Code
	 * darstellen.
	 *
	 * @param neuerWert der neue Code, muss eine Instanz der Klasse
	 *                  {@link AttRdsEreignisCode}, {@link AttRdsVorhersageCode}
	 *                  oder {@link AttRdsEmpfehlungsCode} sein, darf nicht
	 *                  <code>null</code> sein
	 */
	public void setCode(final Zahl<Integer> neuerWert) {
		if (neuerWert == null) {
			return;
		}

		assert (neuerWert instanceof AttRdsEreignisCode) || (neuerWert instanceof AttRdsVorhersageCode)
				|| (neuerWert instanceof AttRdsEmpfehlungsCode);

		if (neuerWert != code) {
			final Zahl<Integer> alterWert = code;
			code = neuerWert;
			firePropertyChange("code", alterWert, neuerWert);

			// aktualisiere ggf. die Kategorie und Dauerdieses Ereignisses
			final RdsCache rdsCache = rdsMeldungWrapper.getRdsCache();
			final RdsTmcCodeWrapper rdsTmcCodeWrapper = rdsCache.getRdsTmcCodeWrapper(code);
			if (rdsTmcCodeWrapper != null) {
				setDauer(rdsTmcCodeWrapper.getEreignisdauer());
				setEmpfohleneDauer(rdsTmcCodeWrapper.getEreignisdauer());
				setKategorie(rdsTmcCodeWrapper.getKategorie());
				setQuantitaet(rdsCache.getRdsQuantitaetWrapper(rdsTmcCodeWrapper.getQuantitaetReferenz()));
				setText(rdsTmcCodeWrapper, true);
			}
			// aktualisiere sämtliche Daten, welche von der gesamten
			// Ereignis-Liste abhängen
			rdsMeldungWrapper.aktualisiereEreignisAbhaengigeDaten(true, true);

			// aktualisiere die Bits
			rdsMeldungWrapper.setBits();

			// aktualisiere den Meldungstext
			rdsMeldungWrapper.setMeldungsText();
		}
	}

	/**
	 * Liefert den aktuellen Ausgabe-Text zurück. Der Text ist abhängig von dem
	 * gesetzten Ereignis-, Vorhersage- bzw. Empfehlungs-Code sowie möglicherweise
	 * gesetzten Quantitäten.
	 *
	 * @return den aktuellen Ausgabe-Text
	 */
	public String getText() {
		return text;
	}

	/**
	 * Setzt den aktuellen Ausgabe-Text.
	 *
	 * @param rdsTmcCodeWrapper            der {@link RdsTmcCodeWrapper}
	 * @param aktualisiereBetroffeneLaenge <code>true</code>, wenn die betroffene
	 *                                     Länge aktualisiert werden soll, ansonsten
	 *                                     <code>false</code>
	 */
	private void setText(final RdsTmcCodeWrapper rdsTmcCodeWrapper, final boolean aktualisiereBetroffeneLaenge) {

		// Speicher zunächste den alten Wert
		String alterWert = null;
		if (text != null) {
			alterWert = new String(text);
		}

		// Ermittel die assoziierte Quantitäten und Texte
		final RdsCache rdsCache = rdsMeldungWrapper.getRdsCache();
		RdsTmcCodeWrapper rds = rdsTmcCodeWrapper;
		if (rdsTmcCodeWrapper == null) {
			rds = rdsCache.getRdsTmcCodeWrapper(getCode());
		}

		// Ermittel den Default-Text
		String neuerWert = "";
		if (rds != null) {

			final RdsQuantitaetWrapper quantitaetWrapper = rdsCache
					.getRdsQuantitaetWrapper(rds.getQuantitaetReferenz());
			final String rdsText = rds.getText();
			final String rdsTextEinzahl = rds.getTextQuantitaetEinzahl();
			final String rdsTextMehrzahl = rds.getTextQuantitaetMehrzahl();

			if ((quantitaetWrapper != null) && (quantitaetStringWert != null) && !"".equals(quantitaetStringWert)) {
				if ("1".equals(quantitaetStringWert) && (rdsTextEinzahl != null) && !"".equals(rdsTextEinzahl)) {
					neuerWert = rdsTextEinzahl;
				} else if ((rdsTextMehrzahl != null) && !"".equals(rdsTextMehrzahl)) {
					neuerWert = rdsTextMehrzahl;
				}
			}
			if ("".equals(neuerWert)) {
				neuerWert = rdsText;
			}
		} else if (getCode() != null) {
			neuerWert = getCode().toString();
		}

		// Setze den Wert für die betroffene Länge ein
		if (neuerWert.contains(PLATZHALTER_BETROFFENE_LAENGE)) {
			if ((betroffeneLaenge == null) && aktualisiereBetroffeneLaenge) {
				setBetroffeneLaenge(BetroffeneLaenge.KM_1, false);
			}
			if (betroffeneLaenge != null) {
				neuerWert = neuerWert.replace(PLATZHALTER_BETROFFENE_LAENGE, betroffeneLaenge.toString() + " km");
			}
		} else {
			setBetroffeneLaenge(null, false);
		}

		// Setze den Wert für die Quantität ein
		if (neuerWert.contains(PLATZHALTER_QUANTITAET) && (quantitaet != null)) {

			if (QuantitaetenUtil.KENNUNG_VERZOEGERUNG.equals(quantitaet.getKennung())) {

				neuerWert = neuerWert.replace("bis zu ", PLATZHALTER_QUANTITAET);
				neuerWert = neuerWert.replace(PLATZHALTER_QUANTITAET,
						"bis zu " + QuantitaetenUtil.getVerzoegerungString(quantitaetIntWert));
			} else {

				String einheit = quantitaet.getEinheit();
				if (!"".equals(einheit)) {
					einheit = " " + einheit;
				}
				neuerWert = neuerWert.replace(PLATZHALTER_QUANTITAET,
						quantitaetStringWert != null ? quantitaetStringWert + einheit : PLATZHALTER_QUANTITAET);
			}
		}

		// Setze den Meldungstext
		if (!Objects.equals(neuerWert,alterWert)) {
			text = neuerWert;
			firePropertyChange("text", alterWert, neuerWert);
			rdsMeldungWrapper.setMeldungsText();
		}
	}

	@Override
	protected void firePropertyChange(final String propertyName, final Object oldValue, final Object newValue) {
		super.firePropertyChange(propertyName, oldValue, newValue);
		rdsMeldungWrapper.setDirty(true);
		rdsMeldungWrapper.setStatus(propertyName);
	}

	@Override
	public String toString() {
		if (code != null) {
			return code.toString();
		}
		return "unbekanntes Ereignis";
	}
}
