/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.util.test;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.bitctrl.modell.AnmeldeException;
import de.bsvrz.sys.funclib.bitctrl.modell.DefaultObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Zeitstempel;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.attribute.AttTlsDEFehlerStatus;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.attribute.AttTlsDEKanalStatus;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.attribute.AttTlsDEProjektierungsStatus;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.attribute.AttTlsHersteller;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.attribute.AttTlsHerstellerDefinierterCode;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.konfigurationsdaten.KdDe;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.objekte.DeUfd;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.objekte.DeVlt;
import de.bsvrz.sys.funclib.bitctrl.modell.tmkextlsglobal.onlinedaten.OdTlsGloDeFehler;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Testdatensender für diejenigen Attributgruppen, die durch KzdTestSender und
 * UfdTestSender nicht bedient werden, die aber für eine sinnvolle Simulation
 * erforderlich oder zumindest vorteilhaft sind. Eine aktive Simulation von
 * DE-Fehlern ist möglich und kann über Kommandozeilenoptionen gesteuert werden.
 * Dann wird in einem einstellbaren Intervall (in Sekunden) für alle DE UFD und
 * DE VLT ein zufälliger DE-Fehlerstatus gesendet. Ansonsten werden nur initial
 * einmalig "Ok" DE-Fehlerdaten gesendet.
 *
 * Im Einzelnen sind dies:
 * <ul>
 * <li>Globale DE-Fehler FG2,3,6 (kommandozeilenoption
 * -gloDeFehler=[2|3|6|keine])</li>
 * <li>Simulation von DE-Fehlern (Optionen -simulationAktiv=[ja|nein]
 * -simulationIntervall=&lt;Sekunden&gt;)</li>
 * </ul>
 * 
 * 
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class RestDatenSender implements StandardApplication {

	/**
	 * Liste aller DE UFD (=FG3), die keine Sammelkanäle sind.
	 */
	private final List<DeUfd> deUfdList = new ArrayList<>();

	/**
	 * Liste aller DE VLT (=FG6), die keine Sammelkanäle sind.
	 */
	private final List<DeVlt> deVltList = new ArrayList<>();

	/**
	 * Liste aller Funktionsgruppen, die bedinet werden. Gültige WErte: 2,3,6.
	 */
	private final List<Integer> funktionsGruppen = new ArrayList<>();

	/**
	 * Zufallsgenerator für die Simulation von DE-Fehlern.
	 */
	private final Random random = new Random();

	/**
	 * Flag, ob DE-Fehler simuliert werden sollen.
	 */
	private boolean simulationAktiv = false;

	/**
	 * Intervall in Sekunden für die Simulation von DE-Fehlern.
	 */
	private long simulationIntervallSekunden = 60;

	private final ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor();

	/**
	 * Objekt für Debug-Ausgaben.
	 */
	private Debug debug;

	@Override
	public void parseArguments(final ArgumentList argumentList) throws Exception {
		debug = Debug.getLogger();
		final String fgListStr = argumentList.fetchArgument("-gloDeFehler=keine").asNonEmptyString();
		if ("keine".equalsIgnoreCase(fgListStr)) {
			debug.warning("Versende keine globalen DE-Fehler");
		} else {
			final String[] fgStr = fgListStr.split(",");
			for (final String s : fgStr) {
				funktionsGruppen.add(Integer.parseInt(s));
			}
			debug.info("Versende globale DE-Fehler für folgende FG", funktionsGruppen);
		}

		// Simulation Parameter auslesen
		simulationAktiv = argumentList.fetchArgument("-simulationAktiv=nein").booleanValue();
		if (simulationAktiv) {
			simulationIntervallSekunden = argumentList.fetchArgument("-simulationIntervall=60").longValue();
			debug.info("DE-Fehler-Simulation aktiv mit Intervall von " + simulationIntervallSekunden + " Sekunden");
		} else {
			debug.info("DE-Fehler-Simulation nicht aktiv");
		}
	}

	@Override
	public void initialize(final ClientDavInterface connection) throws Exception {
		DefaultObjektFactory.getInstanz().setDav(connection);
		doInitDeListen();
		doInitGlobaleDeFehler();

		// Simulation starten, falls aktiviert
		if (simulationAktiv) {
			startSimulation();
		}
	}

	/**
	 * Befüllt die Listen mit den DE-Modellobjekten, wobei die Sammelkanäle
	 * ignoriert werden.
	 */
	private void doInitDeListen() {
		for (final SystemObjekt o : DefaultObjektFactory.getInstanz().bestimmeModellobjekte("typ.deUfd")) {
			final DeUfd deUfd = (DeUfd) o;
			final KdDe.Daten d = deUfd.getKdDe().getDatum();
			if (d.getDEKanal().getValue() == 255) {
				debug.config("Ignoriere FG3 Samelkanal ", deUfd);
				continue;
			}
			deUfdList.add(deUfd);
		}
		for (final SystemObjekt o : DefaultObjektFactory.getInstanz().bestimmeModellobjekte("typ.deVlt")) {
			final DeVlt deVlt = (DeVlt) o;
			final KdDe.Daten d = deVlt.getKdDe().getDatum();
			if (d.getDEKanal().getValue() == 255) {
				debug.config("Ignoriere FG6 Samelkanal ", deVlt);
				continue;
			}
			deVltList.add(deVlt);
		}
	}

	/**
	 * Anmeldung für globale DE-Fehler an allen DE-Listen, die per Kommandozeile
	 * gefordert wurden. Gleichzeitig Versand der Daten.
	 */
	private void doInitGlobaleDeFehler() {
		final long z = DefaultObjektFactory.getInstanz().getDav().getTime();
		for (final Integer i : funktionsGruppen) {
			if (i == 3) {
				doAnmeldenUndSendenGloDeFehlerUfd(z);
			} else if (i == 6) {
				doAnmeldenUndSendenGloDeFehlerVlt(z);
			}
		}
	}

	/**
	 * Anmeldung und Versand eines "gut" Datensatezs für alle DE UFD.
	 *
	 * @param z Zeitstempel der Datensätze
	 */
	private void doAnmeldenUndSendenGloDeFehlerUfd(final long z) {
		for (final DeUfd deUfd : deUfdList) {
			final OdTlsGloDeFehler.Daten gloDeFehlerDaten = deUfd.getOdTlsGloDeFehler().createDatum();
			fillGloDeFehlerDatensatzGutmeldung(z, gloDeFehlerDaten);
			try {
				deUfd.getOdTlsGloDeFehler().anmeldenQuelle(OdTlsGloDeFehler.Aspekte.TlsAntwort, gloDeFehlerDaten);
			} catch (final AnmeldeException ex) {
				debug.error("Fehler bei Quellanmeldung an globale DEFehler " + deUfd, ex);
			}
		}
	}

	/**
	 * Anmeldung und Versand eines "gut" Datensatezs für alle DE VLT.
	 *
	 * @param z Zeitstempel der Datensätze
	 */
	private void doAnmeldenUndSendenGloDeFehlerVlt(final long z) {
		for (final DeVlt deUfd : deVltList) {
			final OdTlsGloDeFehler.Daten gloDeFehlerDaten = deUfd.getOdTlsGloDeFehler().createDatum();
			fillGloDeFehlerDatensatzGutmeldung(z, gloDeFehlerDaten);
			try {
				deUfd.getOdTlsGloDeFehler().anmeldenQuelle(OdTlsGloDeFehler.Aspekte.TlsAntwort, gloDeFehlerDaten);
			} catch (final AnmeldeException ex) {
				debug.error("Fehler bei Quellanmeldung an globale DEFehler " + deUfd, ex);
			}
		}
	}

	/**
	 * Befüllt einen Datensatz mit Werten für eine Gutmeldung.
	 *
	 * @param z                Zeitstempel des Datensatzes
	 * @param gloDeFehlerDaten zu befüllender Datensatz
	 */
	private void fillGloDeFehlerDatensatzGutmeldung(final long z, final OdTlsGloDeFehler.Daten gloDeFehlerDaten) {
		gloDeFehlerDaten.dSetZeitstempel(new Zeitstempel(z));
		gloDeFehlerDaten.setDEFehlerStatus(AttTlsDEFehlerStatus.ZUSTAND_0_OK);
		gloDeFehlerDaten.setDEKanalStatus(AttTlsDEKanalStatus.ZUSTAND_0_AKTIV);
		gloDeFehlerDaten.setDEProjektierungsStatus(AttTlsDEProjektierungsStatus.ZUSTAND_0_PROJEKTIERUNG_OK);
		gloDeFehlerDaten.setHersteller(AttTlsHersteller.ZUSTAND_0_HERSTELLER_UNBEKANNT);
		gloDeFehlerDaten.setHerstellerDefinierterCode(new AttTlsHerstellerDefinierterCode((byte) 0));
	}

	/**
	 * Befüllt einen Datensatz mit zufälligen Werten für die Simulation.
	 *
	 * @param z                Zeitstempel des Datensatzes
	 * @param gloDeFehlerDaten zu befüllender Datensatz
	 */
	private void fillGloDeFehlerDatensatzZufaellig(final long z, final OdTlsGloDeFehler.Daten gloDeFehlerDaten) {
		gloDeFehlerDaten.dSetZeitstempel(new Zeitstempel(z));
		AttTlsDEFehlerStatus deFehlerStatus = getRandomDEFehlerStatus();
		gloDeFehlerDaten.setDEFehlerStatus(deFehlerStatus);
		gloDeFehlerDaten.setDEKanalStatus(AttTlsDEKanalStatus.ZUSTAND_0_AKTIV);
		gloDeFehlerDaten.setDEProjektierungsStatus(AttTlsDEProjektierungsStatus.ZUSTAND_0_PROJEKTIERUNG_OK);
		gloDeFehlerDaten.setHersteller(AttTlsHersteller.ZUSTAND_0_HERSTELLER_UNBEKANNT);
		if (Objects.equals(deFehlerStatus,AttTlsDEFehlerStatus.ZUSTAND_0_OK)) {
			gloDeFehlerDaten.setHerstellerDefinierterCode(new AttTlsHerstellerDefinierterCode((byte) 0));
		} else {
			gloDeFehlerDaten
					.setHerstellerDefinierterCode(new AttTlsHerstellerDefinierterCode((byte) random.nextInt(15)));
		}
	}

	/**
	 * Gibt einen zufälligen DE-Fehlerstatus zurück.
	 *
	 * @return zufälliger DE-Fehlerstatus
	 */
	private AttTlsDEFehlerStatus getRandomDEFehlerStatus() {
		final List<AttTlsDEFehlerStatus> zustaende = AttTlsDEFehlerStatus.getZustaende();
		List<AttTlsDEFehlerStatus> gewichtet = new ArrayList<>(zustaende);
		// Erhöhe die Wahrscheinlichkeit für "OK" auf mind. 75%
		for (int i = 0; i < 3 * zustaende.size(); i++) {
			gewichtet.add(AttTlsDEFehlerStatus.ZUSTAND_0_OK);
		}
		return gewichtet.get(random.nextInt(gewichtet.size()));
	}

	/**
	 * Startet die periodische Simulation von DE-Fehlern.
	 */
	private void startSimulation() {
		executorService.scheduleWithFixedDelay(this::simuliereDeFehler, simulationIntervallSekunden,
				simulationIntervallSekunden, TimeUnit.SECONDS);

		debug.info("DE-Fehler-Simulation gestartet mit Intervall: " + simulationIntervallSekunden + " Sekunden");
	}

	/**
	 * Sendet zufällige DE-Fehler für alle registrierten DE-Objekte.
	 */
	private void simuliereDeFehler() {
		final long z = DefaultObjektFactory.getInstanz().getDav().getTime();
		debug.fine("Simuliere DE-Fehler für Zeitstempel: " + z);

		for (final Integer i : funktionsGruppen) {
			if (i == 3) {
				simuliereDeFehlerUfd(z);
			} else if (i == 6) {
				simuliereDeFehlerVlt(z);
			}
		}
	}

	/**
	 * Simuliert DE-Fehler für alle DE UFD.
	 *
	 * @param z Zeitstempel der Datensätze
	 */
	private void simuliereDeFehlerUfd(final long z) {
		for (final DeUfd deUfd : deUfdList) {
			try {
				final OdTlsGloDeFehler.Daten gloDeFehlerDaten = deUfd.getOdTlsGloDeFehler().createDatum();
				fillGloDeFehlerDatensatzZufaellig(z, gloDeFehlerDaten);
				deUfd.getOdTlsGloDeFehler().sendeDatum(OdTlsGloDeFehler.Aspekte.TlsAntwort, gloDeFehlerDaten);
			} catch (final Exception ex) {
				debug.error("Fehler beim Senden von simulierten DEFehler-Daten für " + deUfd, ex);
			}
		}
	}

	/**
	 * Simuliert DE-Fehler für alle DE VLT.
	 *
	 * @param z Zeitstempel der Datensätze
	 */
	private void simuliereDeFehlerVlt(final long z) {
		for (final DeVlt deVlt : deVltList) {
			try {
				final OdTlsGloDeFehler.Daten gloDeFehlerDaten = deVlt.getOdTlsGloDeFehler().createDatum();
				fillGloDeFehlerDatensatzZufaellig(z, gloDeFehlerDaten);

				deVlt.getOdTlsGloDeFehler().sendeDatum(OdTlsGloDeFehler.Aspekte.TlsAntwort, gloDeFehlerDaten);
			} catch (final Exception ex) {
				debug.error("Fehler beim Senden von simulierten DEFehler-Daten für " + deVlt, ex);
			}
		}
	}

	/**
	 * Die Hauptfunktion des Restdatensenders.
	 *
	 * @param args Programm-Argumente
	 */
	public static void main(final String[] args) {
		StandardApplicationRunner.run(new RestDatenSender(), args);
	}
}
