/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2011 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
/* Copyright by BitCtrl Systems Leipzig */
/* BitCtrl Systems Leipzig */
/* Weisenfelser Str. 67 */
/* 04229 Leipzig */
/* Tel.: +49 341 49067 - 0 */
/* Fax.: +49 341 49067 - 15 */
/* mailto:info@bitctrl.de */
/* http://www.bitctrl.de */
/*---------------------------------------------------------------*/
package de.bsvrz.sys.funclib.bitctrl.modell.util.wzg;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.bitctrl.Constants;

import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.DefaultObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.tmbitctrlanzeigen.objekte.BitctrlWzgInhaltGrafikFrei;
import de.bsvrz.sys.funclib.bitctrl.modell.tmbitctrlanzeigen.objekte.GrafikSymbol;
import de.bsvrz.sys.funclib.bitctrl.modell.tmbitctrlanzeigen.parameter.PdInhaltGrafikFreiBilddaten;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Hilfsklasse zum Zwischenpuffern der Parameter-Ist- und Parameter-Soll-Daten
 * der Grafikinhalte.
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public final class BitctrlWzgInhaltGrafikFreiUtil implements DatensatzUpdateListener {

	/** Das Singleton. */
	private static BitctrlWzgInhaltGrafikFreiUtil instance;

	/** Für Logausgaben zur Beobachtung des Cache. */
	private final Debug debug = Debug.getLogger();

	/** Der Zwischenpuffer für Parameter-Ist. */
	private final Map<BitctrlWzgInhaltGrafikFrei, PdInhaltGrafikFreiBilddaten.Daten> freiBildDatenIst = new LinkedHashMap<>();

	/** Der Zwischenpuffer für Parameter-Soll. */
	private final Map<BitctrlWzgInhaltGrafikFrei, PdInhaltGrafikFreiBilddaten.Daten> freiBildDatenSoll = new LinkedHashMap<>();

	/** Die Anzahl Anmeldungen, die wir halten. */
	private int numAnmeldungen;

	/** Die zu verwendende {@link ObjektFactory}. **/
	private final ObjektFactory factory;

	/**
	 * Privater Konstruktor, führt die Anmeldungen am DaV aus.
	 *
	 * @param factory die zu verwendende {@link ObjektFactory}.
	 */
	private BitctrlWzgInhaltGrafikFreiUtil(final ObjektFactory factory) {
		if (factory == null) {
			this.factory = DefaultObjektFactory.getInstanz();
		} else {
			this.factory = factory;
		}
		anmelden();
	}

	/**
	 * Liefert ein Objekt zum On-Demand-Zugriff auf die Parameter-Ist- und
	 * Soll-Daten eines Grafikinhaltes.
	 *
	 * @param factory die zu verwendende {@link ObjektFactory}.
	 *
	 * @return das Objekt.
	 */
	public static BitctrlWzgInhaltGrafikFreiUtil getInstance(final ObjektFactory factory) {
		if (null == instance) {
			instance = new BitctrlWzgInhaltGrafikFreiUtil(factory);
		}
		return instance;
	}

	@Override
	public void datensatzAktualisiert(final DatensatzUpdateEvent event) {
		if ((event.getObjekt() instanceof BitctrlWzgInhaltGrafikFrei)
				&& (event.getDatum() instanceof PdInhaltGrafikFreiBilddaten.Daten)) {
			final BitctrlWzgInhaltGrafikFrei grafik = (BitctrlWzgInhaltGrafikFrei) event.getObjekt();
			if (PdInhaltGrafikFreiBilddaten.Aspekte.ParameterIst.equals(event.getAspekt())) {
				synchronized (freiBildDatenIst) {
					freiBildDatenIst.put(grafik, (PdInhaltGrafikFreiBilddaten.Daten) event.getDatum());
				}
			} else {
				synchronized (freiBildDatenSoll) {
					freiBildDatenSoll.put(grafik, (PdInhaltGrafikFreiBilddaten.Daten) event.getDatum());
				}
			}
		}
	}

	/**
	 * Liefert die Grafikdaten eines bestimmten Wvz-Inhalts Grafik, unter dem Aspekt
	 * "Parameter-Ist". Die Daten werden nicht abgerufen, sondern aus einer lokalen
	 * Map geholt.
	 *
	 * @param wzgInhalt    die fragliche Grafik
	 * @param useIstAspekt Bestimmt, ob der Ist- oder der Soll-Aspket der Bilddaten
	 *                     benutzt wird.
	 * @return die Bilddaten. Kann null sein.
	 */
	public PdInhaltGrafikFreiBilddaten.Daten getBildDaten(final BitctrlWzgInhaltGrafikFrei wzgInhalt,
			final boolean useIstAspekt) {
		PdInhaltGrafikFreiBilddaten.Daten result = null;
		if (useIstAspekt) {
			synchronized (freiBildDatenIst) {
				result = freiBildDatenIst.get(wzgInhalt);
			}
		} else {
			synchronized (freiBildDatenSoll) {
				result = freiBildDatenSoll.get(wzgInhalt);
			}
		}
		return result;
	}

	/**
	 * Start der BMV, Vorladen aller Grafiksymbole, dann Anmeldung auf Soll- und
	 * Ist-Aspekt der WzgInhaltGrafiken, potentiell in eigenem Thread.
	 */
	private void anmelden() {
		debug.info("Pre-Abfrage aller Grafiksymbole");
		final DataModel model = factory.getDav().getDataModel();
		final SystemObjectType grafikSymbolType = model.getType(GrafikSymbol.PID);
		for (final SystemObject o : grafikSymbolType.getElements()) {
			debug.config("Erfrage Grafiksymbol", o);
		}
		debug.info("Anmelden an PdInhaltGrafikFreiBilddaten (Ist und Soll)");
		final SystemObjectType freiTyp = model.getType(BitctrlWzgInhaltGrafikFrei.PID);
		final List<SystemObject> grafikInhalte = freiTyp.getElements();
		numAnmeldungen = 2 * grafikInhalte.size();
		anmeldenGrafikInhalte(grafikInhalte);
	}

	/**
	 * Anmeldung auf Soll- und Ist-Aspekt der WzgInhaltGrafiken, potentiell in
	 * eigenem Thread.
	 *
	 * @param grafikInhalte die Liste der anzumeldenden Grafikinhalte.
	 */
	private void anmeldenGrafikInhalte(final List<SystemObject> grafikInhalte) {
		for (final SystemObject o : grafikInhalte) {
			final BitctrlWzgInhaltGrafikFrei grafikFrei = (BitctrlWzgInhaltGrafikFrei) factory.getModellobjekt(o);
			grafikFrei.getPdInhaltGrafikFreiBilddaten()
					.addUpdateListener(PdInhaltGrafikFreiBilddaten.Aspekte.ParameterIst, this);
			grafikFrei.getPdInhaltGrafikFreiBilddaten()
					.addUpdateListener(PdInhaltGrafikFreiBilddaten.Aspekte.ParameterSoll, this);
		}
		final long maxWarteZeit = 2 * Constants.MILLIS_PER_MINUTE;
		long remainingWarteZeit = Constants.MILLIS_PER_MINUTE;
		int updates = 0;
		try {
			while ((remainingWarteZeit > 0L) && (updates < numAnmeldungen)) {
				Thread.sleep(Constants.MILLIS_PER_SECOND);
				updates = 0;
				synchronized (freiBildDatenSoll) {
					updates += freiBildDatenSoll.size();
				}
				synchronized (freiBildDatenIst) {
					updates += freiBildDatenIst.size();
				}
				remainingWarteZeit -= Constants.MILLIS_PER_SECOND;
			}
		} catch (final InterruptedException ex) {
			// leer
		}
		debug.info(
				"Anmeldungen abgeschlossen, nach " + ((maxWarteZeit - remainingWarteZeit) / Constants.MILLIS_PER_SECOND)
						+ "s, gekommen: " + updates + ", erwartet: " + numAnmeldungen);
	}
}
