/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.internal.modell;

import java.util.Arrays;
import java.util.List;

import de.bsvrz.dav.daf.main.config.SystemObject;

/**
 * Allgemeine Hilfsmethoden für den Umgang mit dem Modell.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public final class ModellUtil {

	/** Liste aller derzeit reservierten Schlüsselworte von Java. */
	public static final List<String> JAVA_KEYWORDS = Arrays.asList("abstract", "continue", "for", "new", "switch",
			"assert", "default", "if", "package", "synchronized", "boolean", "do", "goto", "private", "this", "break",
			"double", "implements", "protected", "throw", "byte", "else", "import", "public", "throws", "case", "enum",
			"instanceof", "return", "transient", "catch", "extends", "int", "short", "try", "char", "final",
			"interface", "static", "void", "class", "finally", "long", "strictfp", "volatile", "const", "float",
			"native", "super", "while");

	/**
	 * Bestimmt einen Java-kompatiblen Class-Namen für ein Systemobjekt.
	 *
	 * @param so ein Systemobjekt.
	 * @return der dazugehörige Class-Name.
	 */
	public static String determineJavaName(final SystemObject so) {
		String result;
		if (so.getPid().length() > 0) {
			final String pid = so.getPid();

			// Name ist der Teil der PID nach dem Punkt "."
			result = pid.substring(pid.indexOf('.') + 1);
		} else if (so.getName().length() > 0) {
			result = so.getName();
		} else {
			return null;
		}

		result = determineJavaName(result);
		return result;
	}

	/**
	 * Bestimmt einen Java-kompatiblen Class-Namen für einen beliebigen Namen.
	 *
	 * @param name ein Name.
	 * @return der dazugehörige Class-Name.
	 */
	public static String determineJavaName(final String name) {
		String result = name;

		// Erstes Zeichen als Großbuchstabe und Sonderzeichen ersetzen
		result = result.substring(0, 1).toUpperCase() + result.substring(1);
		result = replaceSpecialCharacters(result);
		result = removeUnderscore(result);

		if (JAVA_KEYWORDS.contains(result.toLowerCase())) {
			result += "_JavaKeyword";
		}
		return result;
	}

	/**
	 * Bestimmt einen Java-kompatiblen Package-Namen für ein Systemobjekt.
	 *
	 * @param so ein Systemobjekt.
	 * @return das dazugehörige Java-Package.
	 */
	public static String determineJavaPackage(final SystemObject so) {
		final String pid = so.getConfigurationArea().getPid();

		// Name ist der Teil der PID nach dem Punkt "."
		String result = pid.substring(pid.indexOf('.') + 1);
		result = result.substring(0, 1).toUpperCase() + result.substring(1);

		result = replaceSpecialCharacters(result);
		result = removeUnderscore(result);
		result = "de.bsvrz.sys.funclib.bitctrl.modell." + result.toLowerCase();

		return result;
	}

	/**
	 * Entfernt alle Sonderzeichen (wie Umlaute) um einen gültigen Java-Bezeichner
	 * zu erzeugen. Alle Zeichen die als Whitespace angesehen werden (auch Punkt,
	 * Bindestrrich usw.) werden in einen Unterstrich umgewandelt.
	 *
	 * @param name ein beliebiger String.
	 * @return ein gültiger Java-Bezeichner.
	 */
	public static String determineZustands(final String name) {
		return replaceSpecialCharacters(name);
	}

	/**
	 * Entfernt alle Sonderzeichen (wie Umlaute), einen gültigen Java-Bezeichner zu
	 * erzeugen. Alle Zeichen die als Whitespace angesehen werden (auch Punkt,
	 * Bindestrrich usw.) werden in einen Unterstrich umgewandelt.
	 *
	 * @param name ein belieber String.
	 * @return ein gültiger Java-Bezeichner.
	 */
	private static String replaceSpecialCharacters(final String name) {
		String result = name;

		// Umlaute
		result = result.replace("ä", "ae");
		result = result.replace("Ä", "Ae");
		result = result.replace("ö", "oe");
		result = result.replace("Ö", "Oe");
		result = result.replace("ü", "ue");
		result = result.replace("Ü", "Ue");
		result = result.replace("ß", "ss");

		// "Whitespaces" in Unterstriche umwandeln
		// result = result.replace("=", "_");
		// result = result.replace("-", "_");
		// result = result.replace(" ", "_");
		// result = result.replace(".", "_");
		// result = result.replace("/", "_");
		// result = result.replace(":", "_");
		// result = result.replace("+", "_");
		// result = result.replace(",", "_");

		// Sonstige Sonderzeichen entfernen
		for (int i = 0; i < result.length(); ++i) {
			if (i == 0) {
				if (!Character.isJavaIdentifierStart(result.charAt(i))) {
					result = "_" + result.substring(i);
				}
			} else {
				if (!Character.isJavaIdentifierPart(result.charAt(i))) {
					result = result.substring(0, i) + "_" + result.substring(i + 1);
				}
			}
		}

		// Mehrfache Unterstriche zusammenfassen
		while (result.contains("__")) {
			result = result.replace("__", "_");
		}

		return result;
	}

	/**
	 * Entfernt alle Unterstriche und macht den Java-typischen Camel-Case draus.
	 *
	 * @param name der zu korrigierende Text
	 * @return der korrigierte Text
	 */
	private static String removeUnderscore(final String name) {
		String result = name;
		if (result.startsWith("_")) {
			result = result.substring(1);
		}
		final String[] split = result.split("_");
		if (split.length > 1) {
			result = "";
			// den ersten Bustaben zum Großbustaben machen
			for (int i = 0; i < split.length; i++) {
				final String upperCase = split[i].substring(0, 1).toUpperCase();
				split[i] = upperCase + split[i].substring(1);
			}

			for (final String element : split) {
				result += element;
			}
		}

		return result;
	}

	/**
	 * Korrigiert Kommentar Texte. Ausgewählte Sonderzeichen werden durch
	 * HTLM-Entities ersetzt.
	 *
	 * @param text der zu korrigierende Text
	 * @return der korrigierte Text
	 */
	public static String correctComment(final String text) {
		String result = text.replaceAll("/", "&#47;");
		result = result.replaceAll("<", "&lt;");
		result = result.replaceAll(">", "&gt;");

		return result;
	}

	private ModellUtil() {
		// Keine Instanz von Utility-Klasse erlaubt.
	}

}
