/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell;

import java.util.AbstractCollection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;

import javax.swing.event.EventListenerList;

import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.MutableSet;
import de.bsvrz.dav.daf.main.config.MutableSetChangeListener;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.dynobj.DynObjektException;
import de.bsvrz.sys.funclib.dynobj.DynamischeObjekte;

/**
 * Wrapper für online änderbare Mengen im Datenverteiler.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @param <E> der Typ der Objekte in der Menge.
 */
public class AenderbareMenge<E extends SystemObjekt> extends AbstractCollection<E> implements SystemObjekt {

	private class Listener implements MutableSetChangeListener {

		@Override
		public void update(final MutableSet set, final SystemObject[] addedObjects,
				final SystemObject[] removedObjects) {
			final Collection<E> hinzugefuegt = new ArrayList<>();
			for (final SystemObject so : addedObjects) {
				hinzugefuegt.add((E) objektFactory.getModellobjekt(so));
			}

			final Collection<E> entfernt = new ArrayList<>();
			for (final SystemObject so : removedObjects) {
				entfernt.add((E) objektFactory.getModellobjekt(so));
			}

			fireMengeAktualisiert(Collections.unmodifiableCollection(hinzugefuegt),
					Collections.unmodifiableCollection(entfernt));
		}

	}

	private final EventListenerList listeners = new EventListenerList();
	private final Debug log = Debug.getLogger();
	private final Listener listener = new Listener();

	private final ObjektFactory objektFactory;
	private final MutableSet mutableSet;
	private final int minAnzahl;
	private final int maxAnzahl;
	private final SystemObjekt systemObjekt;

	/**
	 * Initialisiert den Wrapper.
	 *
	 * <p>
	 * <em>Hinweis:</em> Der Konstruktor gehört nicht zur public API.
	 *
	 * @param objektFactory die Objektfabrik, die von der Menge verwendet werden
	 *                      soll.
	 * @param mutableSet    die zu kapselnde Menge,
	 * @param minAnzahl     die minimale Anzahl von Objekten in der Menge.
	 * @param maxAnzahl     die maximale Anzahl von Objekten in der Menge.
	 * @param systemObjekt  das Systemobjekt an dem die Menge hängt.
	 */
	public AenderbareMenge(final ObjektFactory objektFactory, final MutableSet mutableSet, final int minAnzahl,
			final int maxAnzahl, final SystemObjekt systemObjekt) {
		if (objektFactory == null) {
			throw new NullPointerException("Die Objektfabrik darf nicht null sein.");
		}
		if (mutableSet == null) {
			throw new NullPointerException("Die online änderbare Menge darf nicht null sein.");
		}
		if (systemObjekt == null) {
			throw new NullPointerException("Das Systemobjekt darf nicht null sein.");
		}

		this.objektFactory = objektFactory;
		this.mutableSet = mutableSet;
		this.minAnzahl = minAnzahl;
		this.maxAnzahl = maxAnzahl == 0 ? Integer.MAX_VALUE : maxAnzahl;
		this.systemObjekt = systemObjekt;

		// Aktualisiere sich selbst, bei Änderungen
		addMengenListener(new MengenListener() {

			@Override
			public void mengeAktualisiert(final MengenEvent e) {
				removeAll(e.getEntfernt());
				addAll((Collection<? extends E>) e.getHinzugefuegt());
			}
		});
	}

	/**
	 * Gibt die gekapselte Menge zurück.
	 *
	 * <p>
	 * <em>Hinweis:</em> Diese Methode gehört nicht zur public API.
	 *
	 * @return die gekapselte Menge.
	 */
	public MutableSet getMutableSet() {
		return mutableSet;
	}

	/**
	 * Gibt die minimal erlaubte Anzahl von Objekten in der Menge zurück.
	 *
	 * @return die minimale Objektanzahl.
	 */
	public int getMinAnzahl() {
		return minAnzahl;
	}

	/**
	 * Gibt die maximal erlaubte Anzahl von Objekten in der Menge zurück.
	 *
	 * @return die maximale Objektanzahl.
	 */
	public int getMaxAnzahl() {
		return maxAnzahl;
	}

	/**
	 * Gibt das Systemobjekt zurück, an dem die Menge hängt.
	 *
	 * @return das Systemobjekt, an dem die Menge hängt.
	 */
	public SystemObjekt getSystemObjekt() {
		return systemObjekt;
	}

	/**
	 * @throws IllegalStateException wenn die maximal erlaubte Anzahl von Objekten
	 *                               in dieser Menge bereits erreicht ist.
	 * @see #getMaxAnzahl()
	 */
	@Override
	public boolean add(final E e) {
		if (size() == maxAnzahl) {
			throw new IllegalStateException("Die maximale Anzahl Objekt in der Menge " + this + " ist erreicht.");
		}
		try {
			mutableSet.add(e.getSystemObject());
			return true;
		} catch (final ConfigurationChangeException ex) {
			log.error("Das Objekt " + e + " konnte nicht in die Menge " + this + " eingetragen werden.", ex);
			return false;
		}
	}

	/**
	 * @throws IllegalStateException wenn die minimal erlaubte Anzahl von Objekten
	 *                               in dieser Menge bereits erreicht ist.
	 * @see #getMinAnzahl()
	 */
	@Override
	public boolean remove(final Object o) {
		if (size() == minAnzahl) {
			throw new IllegalStateException("Die minimale Anzahl Objekt in der Menge " + this + " ist erreicht.");
		}
		try {
			mutableSet.remove(((SystemObjekt) o).getSystemObject());
			return true;
		} catch (final ConfigurationChangeException ex) {
			log.error("Das Objekt " + o + " konnte nicht aus der Menge " + this + " entfernt werden.", ex);
			return false;
		}
	}

	/**
	 * Entfernt ein Objekt aus der Menge und invalidert das Objekt anschließend.
	 * Kann das Objekt aus der Menge nicht entfernt werden, wird es auch nicht
	 * invalidiert.
	 *
	 * @param o das zu entfernende und zu invalidierende Objekt.
	 * @return <code>true</code>, wenn das Objekt aus der Menge entfernt wurde und
	 *         invalidiert ist. Sollte eines von beiden schiefgehen, wird
	 *         <code>false</code> zurückgegeben.
	 */
	public boolean removeAndInvalidate(final E o) {
		if (remove(o) && o instanceof DynamischesObjekt) {
			try {
				objektFactory.invalidateDynamischesObjekt((DynamischesObjekt) o);
				return true;
			} catch (final DynObjektException ex) {
				log.error("Das Objekt " + o + " konnte nicht invalidiert werden.", ex);
			}
		}
		return false;
	}

	@Override
	public Iterator<E> iterator() {
		final Iterator<SystemObject> iterator = mutableSet.getElements().iterator();
		return new Iterator<E>() {

			private E current;

			@Override
			public boolean hasNext() {
				return iterator.hasNext();
			}

			@Override
			public E next() {
				current = (E) objektFactory.getModellobjekt(iterator.next());
				return current;
			}

			@Override
			public void remove() {
				AenderbareMenge.this.remove(current);
			}

		};
	}

	@Override
	public int size() {
		return mutableSet.getElements().size();
	}

	/**
	 * Entfernt alle invalidierten Objekte aus der Menge.
	 *
	 * @throws DynObjektException wenn nicht alle invalidierten Objekte aus der
	 *                            Menge erfolgreich entfernt werden konnten.
	 */
	public void cleanUp() throws DynObjektException {
		DynamischeObjekte.getInstanz(objektFactory.getDav()).bereinigeMenge(mutableSet);
	}

	/**
	 * Registriert einen Listener für Änderungen an dieser Menge.
	 *
	 * @param l ein Listener.
	 */
	public synchronized void addMengenListener(final MengenListener l) {
		listeners.add(MengenListener.class, l);
		if (listeners.getListenerCount(MengenListener.class) == 1) {
			mutableSet.addChangeListener(listener);
		}
	}

	/**
	 * Deregistriert einen Listener für Änderungen an dieser Menge.
	 *
	 * @param l ein Listener.
	 */
	public synchronized void removeMengenListener(final MengenListener l) {
		listeners.remove(MengenListener.class, l);
		if (listeners.getListenerCount(MengenListener.class) == 0) {
			mutableSet.removeChangeListener(listener);
		}
	}

	/**
	 * Informiert die angemeldeten Listener über eine Änderung an dieser Menge.
	 *
	 * @param hinzugefuegt die Menge der hinzugefügten Objekte.
	 * @param entfernt     die Menge der entfernten Objekte.
	 */
	protected synchronized void fireMengeAktualisiert(final Collection<E> hinzugefuegt, final Collection<E> entfernt) {
		final MengenEvent e = new MengenEvent(this, hinzugefuegt, entfernt);
		for (final MengenListener l : listeners.getListeners(MengenListener.class)) {
			l.mengeAktualisiert(e);
		}

	}

	@Override
	public String toString() {
		return mutableSet.toString();
	}

	@Override
	public long getId() {
		return mutableSet.getId();
	}

	@Override
	public String getName() {
		return mutableSet.getNameOrPidOrId();
	}

	@Override
	public String getPid() {
		return mutableSet.getPid();
	}

	@Override
	public SystemObject getSystemObject() {
		return mutableSet;
	}

	/**
	 * Wird nicht unterstützt und wirft immer eine
	 * {@link UnsupportedOperationException}.
	 */
	@Override
	public SystemObjektTyp getTyp() {
		// TODO Bestimmung des Systemobjekttyps für Mengen unterstützen
		throw new UnsupportedOperationException();
	}

	/**
	 * Wird nicht unterstützt und wirft immer eine
	 * {@link UnsupportedOperationException}.
	 */
	@Override
	public SystemObjektBereich getKonfigurationsBereich() {
		// TODO Bestimmung des Konfigurationsbereichs für Mengen unterstützen
		throw new UnsupportedOperationException();
	}

}
