/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataAndATGUsageInformation;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.DynamicObject;
import de.bsvrz.dav.daf.main.config.DynamicObjectType;
import de.bsvrz.dav.daf.main.config.ObjectSet;
import de.bsvrz.dav.daf.main.config.ObjectTimeSpecification;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.daf.DefaultDavProvider;
import de.bsvrz.sys.funclib.bitctrl.modell.impl.AbstractDatum;
import de.bsvrz.sys.funclib.bitctrl.modell.impl.AbstractSystemObjekt;
import de.bsvrz.sys.funclib.dynobj.DynObjektException;
import de.bsvrz.sys.funclib.dynobj.DynamischeObjekte;

/**
 * Defaultimplmentierung der Schnittstelle {@link ObjektFactory}. Kann als
 * Grundlage für eigene Implementierungen verwendet werden oder wenn nur eine
 * Instanz (Singleton) benötigt wird.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public class DefaultObjektFactory extends DefaultDavProvider implements ObjektFactory {

	private static DefaultObjektFactory singleton;

	/**
	 * Gibt eine DefaultFabrik als Singleton zurück. Nützlich für Applikationen, die
	 * nur mit einer Verbindung umgehen müssen.
	 *
	 * @return eine Singletonverbindung.
	 */
	public static ObjektFactory getInstanz() {
		if (singleton == null) {
			singleton = new DefaultObjektFactory("default", null);
		}
		return singleton;
	}

	private final Map<SystemObject, SystemObjekt> cache = new HashMap<>();

	/**
	 * Klasse darf nicht direkt instanziiert werden, wegen dem
	 * Singleton-Entwurfsmuster.
	 *
	 * @param name der Name der Verbindung.
	 * @param dav  die Verbindung.
	 */
	protected DefaultObjektFactory(final String name, final ClientDavInterface dav) {
		super(name, dav);
	}

	@Override
	public void setDav(final ClientDavInterface dav) {
		super.setDav(dav);
	}

	@Override
	protected void setVerbunden(final boolean verbunden) {
		if (!verbunden) {
			synchronized (cache) {
				cache.clear();
			}
		}
		super.setVerbunden(verbunden);
	}

	@Override
	public SystemObjekt getModellobjekt(final String pid) {
		final DataModel modell = getDav().getDataModel();
		final SystemObject so = modell.getObject(pid);
		if (so != null) {
			return getModellobjekt(so);
		}
		return null;
	}

	@Override
	public SystemObjekt getModellobjekt(final long id) {
		final DataModel modell = getDav().getDataModel();
		final SystemObject so = modell.getObject(id);
		if (so != null) {
			return getModellobjekt(so);
		}
		return null;
	}

	@Override
	public synchronized SystemObjekt getModellobjekt(final SystemObject obj) {
		if (obj == null) {
			throw new IllegalArgumentException("Argument darf nicht null sein.");
		}

		// Liegt Objekt bereits im Cache?
		synchronized (cache) {
			if (cache.containsKey(obj)) {
				return cache.get(obj);
			}
		}

		ModellObjektFactory factory = ModellObjektFactoryRegistry.INSTANZ.getFactory(obj.getType());
		if (factory == null) {
			factory = ModellObjektFactoryRegistry.DEFAULT_FACTORY;
		}
		final SystemObjekt so = factory.createModellobjekt(obj, this);

		// Objekt im Cache ablegen
		synchronized (cache) {
			if (so instanceof DynamicObject) {
				final DynamicObject dynObj = (DynamicObject) so;
				if (dynObj.isValid()) {
					// Dynamisches Objekt wird nur gecacht, wenn es noch nicht
					// gelöscht wurde. Wird es gelöscht, wird es automatisch aus
					// dem Cache entfernt.

					cache.put(obj, so);
					dynObj.addListenerForInvalidation(dynamicObject -> cache.remove(dynamicObject));
				}
			} else {
				cache.put(obj, so);
			}
		}

		return so;
	}

	@Override
	public List<SystemObjekt> bestimmeModellobjekte(final String... pids) {
		final List<SystemObjekt> objekte = new ArrayList<>();

		final ClientDavInterface dav = getDav();
		if (dav != null) {
			final DataModel model = dav.getDataModel();

			for (final String pid : pids) {
				final SystemObject obj = model.getObject(pid);

				if (obj != null) {
					if (obj instanceof ConfigurationArea) {
						// Alle Objekte des Konfigurationsbereich suchen
						final ConfigurationArea kb = (ConfigurationArea) obj;
						for (final SystemObject objekt : kb.getObjects(null, ObjectTimeSpecification.valid())) {
							if (!(objekt instanceof ObjectSet)) {
								// Mengen können nicht richtig aufgelöst werden und sind normalerweise auch
								// nicht gefragt, wenn man alles Konfigurationsobjekte eines KB haben möchte.
								objekte.add(getModellobjekt(objekt));
							}
						}
					} else if (obj instanceof SystemObjectType) {
						// Alle Objekte des Typs suchen
						final SystemObjectType typ = (SystemObjectType) obj;
						for (final SystemObject so : typ.getElements()) {
							objekte.add(getModellobjekt(so));
						}
					} else {
						// Das Objekt selber suchen
						objekte.add(getModellobjekt(obj));
					}
				}
			}
		}
		return objekte;
	}

	@Override
	public Collection<SystemObjektTyp> getTypen() {
		return ModellObjektFactoryRegistry.INSTANZ.getSystemObjektTypen();
	}

	@Override
	public <T extends DynamischesObjekt> T createDynamischesObjekt(final Class<T> clazz, final String name,
			final String pid, final KonfigurationsDatum... daten) throws DynObjektException {
		SystemObjektTyp typ = null;

		try {
			final String className = clazz.getPackage().getName() + ".impl." + clazz.getSimpleName() + "Impl";
			final Class<?> clazzImpl = Class.forName(className);
			final AbstractSystemObjekt modell = (AbstractSystemObjekt) clazzImpl.newInstance();
			modell.init(null, this);
			typ = modell.getTyp();
		} catch (final ClassNotFoundException | InstantiationException | IllegalAccessException ex) {
			// TODO Auto-generated catch block
			ex.printStackTrace();
		}

		if (typ == null) {
			for (final SystemObjektTyp e : getTypen()) {
				if (clazz.equals(e.getClass())) {
					typ = e;
					break;
				}
			}
		}

		if (typ == null) {
			throw new IllegalArgumentException("Der Objekttyp " + clazz + " ist unbekannt.");
		}

		// Objekttyp
		final DynamicObjectType type = (DynamicObjectType) getDav().getDataModel().getObject(typ.getPid());

		// Initiale Konfigurationsdaten
		final List<DataAndATGUsageInformation> konfigurationsDaten = new ArrayList<>();
		for (final Datum datum : daten) {
			final Aspect asp = getDav().getDataModel().getAspect(datum.dGetAspekt().getPid());
			final AttributeGroup atg = getDav().getDataModel().getAttributeGroup(datum.dGetDatensatz().getPid());
			final AttributeGroupUsage atgUsage = atg.getAttributeGroupUsage(asp);
			final Data data = datum.dGetDatensatz().konvertiere(datum);
			final DataAndATGUsageInformation info = new DataAndATGUsageInformation(atgUsage, data);
			konfigurationsDaten.add(info);
		}

		// Objekt anlegen
		final DynamicObject obj = DynamischeObjekte.getInstanz(getDav()).erzeugeObjekt(type, name, pid,
				konfigurationsDaten);

		return (T) getModellobjekt(obj);
	}

	@Override
	public void invalidateDynamischesObjekt(final DynamischesObjekt objekt) throws DynObjektException {
		DynamischeObjekte.getInstanz(getDav()).loescheObjekt((DynamicObject) objekt.getSystemObject());
	}

	@Override
	public <T extends KonfigurationsDatum> List<T> getDaten(final List<? extends SystemObjekt> objekte,
			final Class<? extends KonfigurationsDatensatz<T>> datensatzTyp) {
		if (objekte.isEmpty()) {
			return Collections.emptyList();
		}

		// Standardaspekt für Konfigurationsdaten
		final Aspekt asp = (Aspekt) getModellobjekt("asp.eigenschaften");

		final List<SystemObject> davObjekte = new ArrayList<>();
		final List<T> result = new ArrayList<>();
		for (final SystemObjekt so : objekte) {
			// Systemobjektliste für Sammelanfrage bestimmen
			davObjekte.add(so.getSystemObject());

			// Ergebnisliste initialisieren
			final KonfigurationsDatensatz<T> datensatz = ((AbstractSystemObjekt) so).getDatensatz(datensatzTyp);
			final T datum = datensatz.createDatum();
			((AbstractDatum) datum).dSetAspekt(asp);
			result.add(datum);
		}

		// Attributgruppe für Sammelanfrage bestimmen
		final AttributeGroup atg = result.get(0).dGetDatensatz().getSystemObject();

		// Sammelanfrage durchführen
		final DataModel dataModel = getDav().getDataModel();
		final Data[] datas = dataModel.getConfigurationData(davObjekte, atg);

		// Ergebnisliste mit den Daten der Sammelanfrage füllen
		for (int i = 0; i < result.size(); ++i) {
			if (datas[i] != null) {
				result.get(i).atl2Bean(datas[i], this);
			} else {
				result.set(i, null);
			}
		}

		return result;
	}

}
