/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell;

import java.util.Arrays;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.sys.funclib.bitctrl.daf.DavProvider;

/**
 * Überwacht alle {@link ObjektFactory} die als OSGi-Service registriert wurden.
 * 
 * <p>
 * Wird eine neue Factory angelegt , wird diese als neuer Service registriert.
 * Wird eine Factory entfernt, wird der entsprechende Service deregistriert. Mit
 * Hilfe eines {@link ServiceTrackerCustomizer} kann auf diese Änderungen
 * reagiert werden.
 * 
 * <p>
 * Ein oder mehrere Services mit dem beschriebenen Verhalten müssen selbst am
 * BundleContext registriert werden. Als Vorlage für einen solchen Service kann
 * {@link DefaultObjektFactory} dienen.
 * 
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @version $Id: ObjektFactoryServiceTracker.java 21438 2010-02-05 12:52:29Z
 *          schumann $
 * @see BundleContext#registerService(String, Object, java.util.Dictionary)
 */
public class ObjektFactoryServiceTracker extends ServiceTracker {

	/**
	 * Erzeugt einen Service Tracker für einen bestimmten Bundlekontext.
	 * 
	 * @param context der Bundlekontext für den Service Tracker angelegt wird.
	 */
	public ObjektFactoryServiceTracker(final BundleContext context) {
		super(context, ObjektFactory.class.getName(), null);
	}

	/**
	 * Erzeugt einen Service Tracker für einen bestimmten Bundlekontext.
	 * 
	 * @param context    der Bundlekontext für den Service Tracker angelegt wird.
	 * @param customizer Customizer der auf Hinzufügen und Entfernen von Factories
	 *                   reagiert.
	 */
	public ObjektFactoryServiceTracker(final BundleContext context, final ServiceTrackerCustomizer customizer) {
		super(context, ObjektFactory.class.getName(), customizer);
	}

	/**
	 * Gibt die Objekt Factory der Nutzerverbindung zurück.
	 * 
	 * @return die Objekt Factory oder <code>null</code>, wenn keine verfügbar ist.
	 */
	public ObjektFactory getNutzerverbindung() {
		final ServiceReference[] serviceReferences = getServiceReferences();
		if (serviceReferences != null) {
			for (final ServiceReference reference : serviceReferences) {
				final Object name = reference.getProperty(DavProvider.PROP_NAME);
				if (DavProvider.NUTZVERVERBINDUNG.equals(name)) {
					return (ObjektFactory) getService(reference);
				}
			}
		}

		return null;
	}

	/**
	 * Gibt die Objekt Factory der Urlasserverbindung zurück.
	 * 
	 * @return die Objekt Factory oder <code>null</code>, wenn keine verfügbar ist.
	 */
	public ObjektFactory getUrlasserverbindung() {
		final ServiceReference[] serviceReferences = getServiceReferences();
		if (serviceReferences != null) {
			for (final ServiceReference reference : serviceReferences) {
				final Object name = reference.getProperty(DavProvider.PROP_NAME);
				if (DavProvider.URLASSERVERBINDUNG.equals(name)) {
					return (ObjektFactory) getService(reference);
				}
			}
		}

		return null;
	}

	/**
	 * Gibt eine Objektfabrik zurück.
	 * 
	 * @return eine Objektfabrik oder <code>null</code>, wenn im Moment keine zur
	 *         Verfügung steht.
	 * @see #getService()
	 */
	public ObjektFactory getObjektFactory() {
		return (ObjektFactory) getService();
	}

	/**
	 * Gibt alle verfügbaren Objektfabrik zurück.
	 * 
	 * @return alle im Moment vorhandenen Objektfabriken oder <code>null</code>,
	 *         wenn im keine zur Verfügung stehen.
	 * @see #getServices()
	 */
	public ObjektFactory[] getObjektFactories() {
		final Object[] services = getServices();
		return Arrays.asList(services).toArray(new ObjektFactory[services.length]);
	}

	/**
	 * Gibt eine bestimmte Objektfabrik zurück.
	 * 
	 * @param dav die Datenverteilerverbindung zu der eine Fabrik gesucht wird.
	 * @return die gesuchte Objektfabrik oder <code>null</code>, wenn diese im
	 *         Moment nicht zur Verfügung steht.
	 */
	public ObjektFactory getObjektFactory(final ClientDavInterface dav) {
		final ObjektFactory[] factories = getObjektFactories();
		if (factories != null) {
			for (final ObjektFactory f : factories) {
				if (dav.equals(f.getDav())) {
					return f;
				}
			}
		}
		return null;
	}

}
