/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.att;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Locale;

/**
 * Kapselt ein Zahlattribut eines Datensatzes. Es wird nicht zwischen Ganzzahl
 * und Kommazahl unterschieden, da eine Ganzzahl mit Skalierung auch eine
 * Kommazahl sein kann.
 * <p>
 * Der Typ als Java-Ganzzahl oder Java-Gleitkommazahl ergibt sich aus dem
 * Wertebereich des Datenverteilerattributs. Der Typ sollte als kleinster
 * Datentyp gewählt werden, der den geforderten Wertebereich abbilden kann.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @param <T> Der Zahlentyp, der gekapselt wird.
 */
public class Zahl<T extends Number> extends Number implements Comparable<Zahl<T>> {

	private static class ScaledNumberFormatter {

		public String format(final double doubleValue, final double skalierung) {
			final DecimalFormat skaleFormatter = (DecimalFormat) NumberFormat.getInstance(Locale.GERMAN);
			skaleFormatter.setGroupingUsed(false);
			skaleFormatter.getDecimalFormatSymbols().setDecimalSeparator(',');

			final NumberFormat resultFormatter = NumberFormat.getInstance();
			final String skaleFormat = skaleFormatter.format(skalierung);

			final int index = skaleFormat.indexOf(',');
			if (index < 0) {
				resultFormatter.setMinimumFractionDigits(0);
				resultFormatter.setMaximumFractionDigits(999);
			} else {
				final int precision = skaleFormat.length() - (index + 1);
				resultFormatter.setMinimumFractionDigits(precision);
				resultFormatter.setMaximumFractionDigits(precision);
			}

			return resultFormatter.format(doubleValue);
		}
	}

	private static final long serialVersionUID = 1L;

	private final T value;
	private final String zustand;

	private WerteBereich werteBereich;

	private static final ScaledNumberFormatter skaledFormatter = new ScaledNumberFormatter();

	/**
	 * Der Konstruktor darf nur intern von der Bibliothek verwendet werden.
	 *
	 * @param value der neue Zahlenwert.
	 * @throws IllegalArgumentException der übergebene Wert kann nicht für die
	 *                                  Initialisierung einer Instanz verwendet
	 *                                  werden
	 */
	public Zahl(final T value) {
		this(null, value);
	}

	/**
	 * Nicht öffentlicher Konstruktor zum Anlegen eines Zustands.
	 *
	 * @param zustand der Name des Zustands.
	 * @param value   der Zahlenwert des Zustands.
	 */
	protected Zahl(final String zustand, final T value) {
		if (value == null) {
			throw new NullPointerException("Der Wert darf nicht null sein.");
		}

		this.zustand = zustand;
		this.value = value;

		werteBereich = WerteBereich.getWerteBereich(getClass());
		if ((werteBereich != null) && !isZustand()) {
			if (value.doubleValue() < getWerteBereich().getMinimum()) {
				throw new IllegalArgumentException(
						"Der Wert muss größer oder gleich " + getWerteBereich().getMinimum() + " sein: " + value);
			}
			if (value.doubleValue() > getWerteBereich().getMaximum()) {
				throw new IllegalArgumentException(
						"Der Wert muss kleiner oder gleich " + getWerteBereich().getMaximum() + " sein: " + value);
			}
		}
	}

	@Override
	public double doubleValue() {
		return value.doubleValue();
	}

	@Override
	public float floatValue() {
		return value.floatValue();
	}

	@Override
	public int intValue() {
		return value.intValue();
	}

	@Override
	public long longValue() {
		return value.longValue();
	}

	/**
	 * Gibt den Wertebereich der Zahl zurück. Der Wertebereich ist nur
	 * Ganzzahlattribute definiert, Kommazahlattribute haben keinen Wertebereich.
	 *
	 * @return der Wertebereich der Zahl oder <code>null</code>, wenn keiner
	 *         definiert ist.
	 */
	public WerteBereich getWerteBereich() {
		return werteBereich;
	}

	/**
	 * Gibt den Wert der Zahl zurück.
	 *
	 * @return der Zahlenwert.
	 */
	public T getValue() {
		return value;
	}

	/**
	 * Flag, ob diese Zahl ein Zustand darstellt.
	 *
	 * @return {@code true}, wenn diese Zahl ein Zustand ist.
	 */
	public boolean isZustand() {
		return zustand != null;
	}

	@Override
	public boolean equals(final Object obj) {
		if (obj == this) {
			return true;
		}
		if (getClass().isInstance(obj)) {
			return getValue().equals(((Zahl<?>) obj).getValue());
		}
		return false;
	}

	@Override
	public int hashCode() {
		return getValue().hashCode();
	}

	@Override
	public int compareTo(final Zahl<T> arg0) {
		return Double.compare(doubleValue(), arg0.doubleValue());
	}

	/**
	 * Gibt den Wert mit der Einheit zurück oder den Namen des Zustandes, falls die
	 * Zahl einen Zustand repräsentiert.
	 */
	@Override
	public String toString() {
		if (isZustand()) {
			return zustand;
		}

		String s;
		if ((werteBereich != null) && (werteBereich.getSkalierung() != 1.0)) {
			s = skaledFormatter.format(doubleValue(), werteBereich.getSkalierung());
		} else {
			s = getValue().toString();
		}

		if ((werteBereich != null) && (werteBereich.getEinheit() != null)) {
			s += " " + werteBereich.getEinheit();
		}

		return s;
	}

}
