/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.att;

import java.text.DateFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Date;

/**
 * Kapselt ein absolutes Zeitstempelattribut eines Datensatzes.
 *
 * @author BitCtrl Systems GmbH, Falko Schuman
 */
public class Zeitstempel implements Comparable<Zeitstempel> {

	private final long time;

	/**
	 * Initialisiert den Zeitstempel mit der aktuellen Zeit.
	 */
	public Zeitstempel() {
		this(System.currentTimeMillis());
	}

	/**
	 * Initialisiert den Zeitstempel.
	 *
	 * @param time der Wert des Zeitstempels.
	 */
	public Zeitstempel(final long time) {
		this.time = time;
	}

	/**
	 * Der Zeitstempel in Millisekunden.
	 *
	 * @return der Zeitstempel.
	 */
	public long getTime() {
		return time;
	}

	@Override
	public boolean equals(final Object obj) {
		if (obj == this) {
			return true;
		} else if (obj instanceof Zeitstempel) {
			return compareTo((Zeitstempel) obj) == 0;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return Long.valueOf(getTime()).hashCode();
	}

	@Override
	public int compareTo(final Zeitstempel o) {
		if (getTime() < o.getTime()) {
			return -1;
		} else if (getTime() > o.getTime()) {
			return 1;
		}
		return 0;
	}

	@Override
	public String toString() {
		return DateFormat.getDateTimeInstance().format(new Date(getTime()));
	}

	/**
	 * Konvertiert den Zeitstempel zu einem Instant.
	 * 
	 * @return der Zeitstempel als Instant
	 */
	public Instant toInstant() {
		return Instant.ofEpochMilli(time);
	}

	/**
	 * Konvertiert den Zeitstempel zu einem ZonedDateTime in der angegebenen Zeitzone.
	 * 
	 * @param zone die Zeitzone
	 * @return der Zeitstempel als ZonedDateTime
	 */
	public ZonedDateTime toZonedDateTime(ZoneId zone) {
		return ZonedDateTime.ofInstant(toInstant(), zone);
	}

	/**
	 * Konvertiert den Zeitstempel zu einem ZonedDateTime in der System-Zeitzone.
	 * 
	 * @return der Zeitstempel als ZonedDateTime in der System-Zeitzone
	 */
	public ZonedDateTime toZonedDateTime() {
		return toZonedDateTime(ZoneId.systemDefault());
	}

	/**
	 * Konvertiert den Zeitstempel zu einem LocalDateTime in der angegebenen Zeitzone.
	 * 
	 * @param zone die Zeitzone
	 * @return der Zeitstempel als LocalDateTime
	 */
	public LocalDateTime toLocalDateTime(ZoneId zone) {
		return LocalDateTime.ofInstant(toInstant(), zone);
	}

	/**
	 * Konvertiert den Zeitstempel zu einem LocalDateTime in der System-Zeitzone.
	 * 
	 * @return der Zeitstempel als LocalDateTime in der System-Zeitzone
	 */
	public LocalDateTime toLocalDateTime() {
		return toLocalDateTime(ZoneId.systemDefault());
	}

	/**
	 * Erstellt einen Zeitstempel aus einem Instant.
	 * 
	 * @param instant das Instant
	 * @return neuer Zeitstempel
	 */
	public static Zeitstempel from(Instant instant) {
		return new Zeitstempel(instant.toEpochMilli());
	}

	/**
	 * Erstellt einen Zeitstempel aus einem ZonedDateTime.
	 * 
	 * @param zonedDateTime das ZonedDateTime
	 * @return neuer Zeitstempel
	 */
	public static Zeitstempel from(ZonedDateTime zonedDateTime) {
		return from(zonedDateTime.toInstant());
	}

	/**
	 * Erstellt einen Zeitstempel aus einem LocalDateTime in der angegebenen Zeitzone.
	 * 
	 * @param localDateTime das LocalDateTime
	 * @param zone die Zeitzone
	 * @return neuer Zeitstempel
	 */
	public static Zeitstempel from(LocalDateTime localDateTime, ZoneId zone) {
		return from(localDateTime.atZone(zone));
	}

	/**
	 * Erstellt einen Zeitstempel aus einem LocalDateTime in der System-Zeitzone.
	 * 
	 * @param localDateTime das LocalDateTime
	 * @return neuer Zeitstempel
	 */
	public static Zeitstempel from(LocalDateTime localDateTime) {
		return from(localDateTime, ZoneId.systemDefault());
	}

}