/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.impl;

import static com.bitctrl.Constants.MILLIS_PER_SECOND;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import javax.swing.event.EventListenerList;

import com.bitctrl.Constants;
import com.bitctrl.util.Interval;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataNotSubscribedException;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.archiv.ArchivIterator;
import de.bsvrz.sys.funclib.bitctrl.archiv.ArchivUtilities;
import de.bsvrz.sys.funclib.bitctrl.daf.OperationTimedOutException;
import de.bsvrz.sys.funclib.bitctrl.daf.SendRegistrationStore;
import de.bsvrz.sys.funclib.bitctrl.daf.SendRegistrationStoreDataRequestListener;
import de.bsvrz.sys.funclib.bitctrl.modell.AnmeldeException;
import de.bsvrz.sys.funclib.bitctrl.modell.Aspekt;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensendeException;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.OnlineDatensatz;
import de.bsvrz.sys.funclib.bitctrl.modell.OnlineDatum;
import de.bsvrz.sys.funclib.bitctrl.modell.SendeSteuerungEvent;
import de.bsvrz.sys.funclib.bitctrl.modell.SendeSteuerungListener;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.att.Zeitstempel;

/**
 * Implementiert gemeinsame Funktionen von Onlinedatens&auml;tzen.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @param <T> Der Typ des Datums den der Datensatz sichert.
 */
public abstract class AbstractOnlineDatensatz<T extends OnlineDatum> extends AbstractDatensatz<T>
        implements OnlineDatensatz<T>, SendRegistrationStoreDataRequestListener {

    /**
     * {@link Executors} zum asynchronen Abarbeiten der Daten aus dem Dav - Updater
     * Thread.
     */
    private static final Executor UPDATE_EXECUTOR = Executors.newSingleThreadExecutor();

    /**
     * Der Empf&auml;nger wird in einer internen Klasse vor dem Anwender versteckt.
     */
    private class AsynchronerReceiver implements ClientReceiverInterface {

        /**
         * {@link Runnable} zur asyncronen Abarbeitung der Daten aus dem Dav-Updater
         * Thread. Damit können jetzt auch gefahrlos Konfigurationsabfragen hier
         * passieren.
         *
         * @author BitCtrl Systems GmbH, Christian Hösel
         */
        private final class UpdateAbarbeiter implements Runnable {
            private final ResultData[] results;

            private UpdateAbarbeiter(final ResultData[] results) {
                this.results = results;
            }

            @Override
            public void run() {
                for (final ResultData result : results) {
                    boolean aspektGefunden = false;
                    for (final Aspekt asp : getAspekte()) {
                        if (getAspect(asp).equals(result.getDataDescription().getAspect())) {
                            aspektGefunden = true;
                            fireDatensatzAktualisiert(result.getDataDescription(), konvertiere(result));
                        }
                    }
                    if (!aspektGefunden) {
                        throw new IllegalStateException(
                                "Unbekannter Aspekt " + result.getDataDescription().getAspect() + " im Datensatz "
                                        + getClass() + ". Das Modell ist möglicherweise nicht mehr aktuell.");
                    }
                }
            }
        }

        private final Set<DataDescription> angemeldet = Collections.synchronizedSet(new HashSet<DataDescription>());

        /**
         * Meldet eine vorhandene Sendeanmeldung wieder ab. Existiert keine Anmeldung,
         * passiert nichts.
         *
         * @param dbs die {@link DataDescription}, für welche die Abmeldung erfolgt.
         * @see ClientDavInterface#unsubscribeReceiver(ClientReceiverInterface,
         *      SystemObject, DataDescription)
         *
         */
        public void abmelden(final DataDescription dbs) {
            if (angemeldet.contains(dbs)) {
                getDav().unsubscribeReceiver(this, getSystemObjekt().getSystemObject(), dbs);
                angemeldet.remove(dbs);
            }
        }

        /**
         * Meldet eine neue Sendeanmeldung an. Eine eventuell existierende Anmeldung
         * wird vorher abgemeldet.
         * 
         * Es erfolgt immer eine Anmeldung auch auf nachgelieferte Daten.
         *
         * @param dbs Die betroffene {@link DataDescription}.
         * @see ClientDavInterface#subscribeReceiver(ClientReceiverInterface,
         *      SystemObject, DataDescription, ReceiveOptions, ReceiverRole)
         *
         *      TODO weitere Optionen für {@link ReceiveOptions} (delta und normal);
         *      hier besteht evtl. Bedarf einen Receiver pro Listener anzulegen, da
         *      potentiell der eine alles und der andere nur Änderungen haben will.
         */
        public void anmelden(final DataDescription dbs) {
            abmelden(dbs);
            if (isSenke(getAspekt(dbs.getAspect()))) {
                getDav().subscribeReceiver(this, getSystemObjekt().getSystemObject(), dbs, ReceiveOptions.delayed(),
                        ReceiverRole.drain());
            } else {
                getDav().subscribeReceiver(this, getSystemObjekt().getSystemObject(), dbs, ReceiveOptions.delayed(),
                        ReceiverRole.receiver());
            }
            angemeldet.add(dbs);
        }

        /**
         * ermittelt, ob der Receiver f&uuml;r den Empfang der Daten des Datensatzes
         * angemeldet ist.
         *
         * @param dbs Die betroffene {@link DataDescription}.
         *
         * @return <code>true</code>, wenn die Anmeldung erfolgt ist.
         */
        public boolean isAngemeldet(final DataDescription dbs) {
            return angemeldet.contains(dbs);
        }

        @Override
        public void update(final ResultData[] results) {
            // asyncon zum Updater können im UpdateAbarbeiter
            // Konfigurationsanfragen gemacht werden.
            UPDATE_EXECUTOR.execute(new UpdateAbarbeiter(results));
        }
    }

    /**
     * Versteckt die Sendelogik des Datenverteilers vor dem Anwender.
     */
    private class SynchronerSender implements ClientSenderInterface {

        private final Map<Aspect, Status> sendesteuerung = Collections.synchronizedMap(new HashMap<Aspect, Status>());
        private final Map<Aspect, Integer> anzahlAnmeldungen = Collections
                .synchronizedMap(new HashMap<Aspect, Integer>());

        /**
         * Meldet eine vorhandene Sendeanmeldung wieder ab. Existiert keine Anmeldung,
         * passiert nichts. Noch nicht gesendet Datens&auml;tze werden aus dem
         * Sendepuffer entfernt.
         *
         * @param asp der betroffene Aspekt.
         */
        public void abmelden(final Aspekt asp) {
            if (anzahlAnmeldungen.containsKey(getAspect(asp))) {
                final int anzahl = anzahlAnmeldungen.get(getAspect(asp)) - 1;
                anzahlAnmeldungen.put(getAspect(asp), anzahl);
                if (anzahl == 0) {
                    final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp));
                    if (isQuelle(asp)) {
                        getDav().unsubscribeSender(this, getSystemObjekt().getSystemObject(), dbs);
                    } else {
                        SendRegistrationStore.unsubscribeSender(getDav(), getSystemObjekt().getSystemObject(), dbs);
                        SendRegistrationStore.removeSendRegistrationStoreDataRequestListener(
                                getSystemObjekt().getSystemObject(), dbs, AbstractOnlineDatensatz.this);
                    }
                    anzahlAnmeldungen.remove(getAspect(asp));
                    synchronized (sendesteuerung) {
                        sendesteuerung.remove(getAspect(asp));
                    }
                }
            }
        }

        /**
         * Meldet eine neue Sendeanmeldung an.
         *
         * @param asp der betroffene Aspekt.
         * @throws AnmeldeException wenn die Anmeldung schief ging.
         */
        public void anmelden(final Aspekt asp) throws AnmeldeException {
            if (anzahlAnmeldungen.containsKey(getAspect(asp))) {
                final int anzahl = anzahlAnmeldungen.get(getAspect(asp)) + 1;
                anzahlAnmeldungen.put(getAspect(asp), anzahl);
                return;
            }

            final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp));
            try {
                if (isQuelle(asp)) {
                    getDav().subscribeSender(this, getSystemObjekt().getSystemObject(), dbs, SenderRole.source());
                } else {
                    SendRegistrationStore.subscribeSender(getDav(), getSystemObjekt().getSystemObject(), dbs);
                }
                anzahlAnmeldungen.put(getAspect(asp), 1);
            } catch (final OneSubscriptionPerSendData ex) {
                anzahlAnmeldungen.remove(getAspect(asp));
                throw new AnmeldeException(ex);
            }
        }

        @Override
        public void dataRequest(final SystemObject object, final DataDescription dataDescription, final byte state) {
            Status status;
            status = Status.getStatus(state);
            synchronized (sendesteuerung) {
                sendesteuerung.put(dataDescription.getAspect(), status);
            }
            fireSendeSteuerungAktualisiert(dataDescription, status);
        }

        /**
         * Gibt den Status der Sendesteuerung zur&uuml;ck.
         *
         * @param asp der betroffene Aspekt.
         * @return der Wert.
         */
        public Status getStatus(final Aspekt asp) {
            if (!isQuelle(asp)) {
                final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp));
                return SendRegistrationStore.getInstance().isRegistered(getSystemObjekt().getSystemObject(), dbs)
                        ? Status.START
                        : Status.STOP;
            }
            return sendesteuerung.get(getAspect(asp));
        }

        /**
         * Gibt den Wert des Flags {@code angemeldet} zur&uuml;ck.
         *
         * @param asp der betroffene Aspekt.
         * @return der Wert.
         */
        public boolean isAngemeldet(final Aspekt asp) {
            return anzahlAnmeldungen.containsKey(getAspect(asp)) && (anzahlAnmeldungen.get(getAspect(asp)) > 0);
        }

        /**
         * Wenn Systemobjekt und Attributgruppe &uuml;bereinstimmen, dann wird
         * {@code true} zur&uuml;gegeben.
         *
         * {@inheritDoc}
         */
        @Override
        public boolean isRequestSupported(final SystemObject object, final DataDescription dataDescription) {
            if (object.equals(getSystemObjekt().getSystemObject())
                    && dataDescription.getAttributeGroup().equals(getSystemObject())) {
                return true;
            }
            return false;
        }

        /**
         * F&uuml;gt ein Datum der Warteschlange des Senders hinzu.
         *
         * @param d           ein zu sendentes Datum.
         * @param asp         der betroffene Aspekt.
         * @param zeitstempel der Zeitstempel, mit dem die Datengesendet werden.
         * @throws DatensendeException wenn die Daten nicht gesendet werden konnten.
         */
        public void sende(final Data d, final Aspekt asp, final Zeitstempel zeitstempel) throws DatensendeException {
            if (!isAngemeldet(asp)) {
                throw new DatensendeException("Der Datensatz wurde noch nicht zum Senden angemeldet.");
            }

            Status status;

            if (!isQuelle(asp)) {
                final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp));
                status = SendRegistrationStore.getInstance().isRegistered(getSystemObjekt().getSystemObject(), dbs)
                        ? Status.START
                        : Status.STOP;
            } else {
                status = sendesteuerung.get(getAspect(asp));
            }
            if (status == null) {
                if (isQuelle(asp)) {
                    status = Status.START;
                } else {
                    throw new DatensendeException("Es liegt noch keine Sendeerlaubnis vor.");
                }
            }

            switch (status) {
            case STOP:
                if (!isQuelle(asp)) {
                    throw new DatensendeException("Die Sendesteuerung hat das Senden verboten.");
                }
                break;
            case KEINE_RECHTE:
                throw new DatensendeException(
                        "Die Sendesteuerung hat das Senden verboten, weil keine ausreichenden Rechte vorhanden sind.");
            case ANMELDUNG_UNGUELTIG:
                throw new DatensendeException(
                        "Die Sendesteuerung hat das Senden verboten, weil die Anmeldung ungültig ist (doppelte Quelle?).");
            default:
                // Sendesteuerung ist positiv
            }

            final long z = zeitstempel != null ? zeitstempel.getTime() : getDav().getTime();
            final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp));
            final ResultData datensatz = new ResultData(getSystemObjekt().getSystemObject(), dbs, z, d);

            try {
                getDav().sendData(datensatz);
            } catch (final DataNotSubscribedException | SendSubscriptionNotConfirmed ex) {
                throw new DatensendeException(ex);
            }
        }

        public void anmeldenSenden(final Aspekt asp, final Data data, final long zeitstempel) throws AnmeldeException {
            if (anzahlAnmeldungen.containsKey(getAspect(asp))) {
                throw new AnmeldeException(
                        "Es ist bereits eine Quelle angemeldet: " + getSystemObject() + " / " + getAspect(asp));
            }

            final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp));
            setQuelle(asp, true);
            final ResultData initialData = new ResultData(getSystemObjekt().getSystemObject(), dbs, zeitstempel, data);
            try {
                getDav().subscribeSource(this, initialData);
                anzahlAnmeldungen.put(getAspect(asp), 1);
            } catch (final OneSubscriptionPerSendData ex) {
                anzahlAnmeldungen.remove(getAspect(asp));
                throw new AnmeldeException(ex);
            }
        }
    }

    /**
     * Standardtimeout von {@value} Millisekunden für das Senden und Empfangen von
     * Daten.
     */
    private static final long TIMEOUT = 60 * MILLIS_PER_SECOND;

    /** Der Empf&auml;nger dieses Datensatzes. */
    private final AsynchronerReceiver receiver;

    /** Der Sender dieses Datensatzes. */
    private final SynchronerSender sender;

    /** Flag ob dieser Datensatz als Quelle angemeldet werden soll. */
    private final Set<Aspekt> quellen;

    /** Flag ob dieser Datensatz als Senke angemeldet werden soll. */
    private final Set<Aspekt> senken;

    /** Liste der registrierten Listener. */
    private final Map<DataDescription, EventListenerList> updateListener;

    private final Map<DataDescription, EventListenerList> sendeSteuerungListeners;

    /**
     * Initialisiert den Onlinedatensatz.
     *
     * @param systemObjekt  das Systemobjekt zu dem der Datensatz gehört.
     * @param objektFactory die ObjektFabrik, die der Datensatz verwenden soll.
     */
    protected AbstractOnlineDatensatz(final SystemObjekt systemObjekt, final ObjektFactory objektFactory) {
        super(systemObjekt, objektFactory);

        quellen = Collections.synchronizedSet(new HashSet<Aspekt>());
        senken = Collections.synchronizedSet(new HashSet<Aspekt>());
        receiver = new AsynchronerReceiver();
        sender = new SynchronerSender();
        updateListener = Collections.synchronizedMap(new HashMap<DataDescription, EventListenerList>());
        sendeSteuerungListeners = Collections.synchronizedMap(new HashMap<DataDescription, EventListenerList>());
    }

    /**
     * Meldet den Datensatz als Sender oder Quelle am Datenverteiler an.
     *
     * @param asp der betroffene Aspekt.
     * @throws AnmeldeException wenn die Anmeldung nicht erfolgreich war.
     */
    @Override
    public void anmeldenSender(final Aspekt asp) throws AnmeldeException {
        final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp));
        SendRegistrationStore.addSendRegistrationStoreDataRequestListener(getSystemObjekt().getSystemObject(), dbs,
                this);
        sender.anmelden(asp);
    }

    /**
     * Meldet den Datensatz als Quelle am Datenverteiler an und versendet den
     * übergebenen Datensatz.
     *
     * @param asp   der betroffene Aspekt.
     * @param datum der initiale Datensatz
     * @throws AnmeldeException wenn die Anmeldung nicht erfolgreich war.
     */
    @Override
    public void anmeldenQuelle(final Aspekt asp, final T datum) throws AnmeldeException {
        setQuelle(asp, true);
        if (datum.dGetZeitstempel() == null) {
            datum.dSetZeitstempel(new Zeitstempel(getDav().getTime()));
        }
        sender.anmeldenSenden(asp, konvertiere(datum), datum.dGetZeitstempel().getTime());
    }

    /**
     * Meldet eine eventuell vorhandene Anmeldung als Sender oder Quelle wieder ab.
     * Noch nicht gesendet Datens&auml;tze werden aus dem Sendepuffer entfernt.
     *
     * @param asp der betroffene Aspekt.
     */
    @Override
    public void abmeldenSender(final Aspekt asp) {
        sender.abmelden(asp);
    }

    @Override
    public T getDatum(final Aspekt asp) {
        return getDatum(asp, Constants.MILLIS_PER_HOUR);
    }

    @Override
    public T getSingleDatum(final Aspekt asp) {
        return getDatum(asp, 0L);
    }

    @Override
    public T getDatum(final Aspekt asp, final long unsubscriptionIntervall) {
        final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp));
        final ResultData resultData = getDav().getData(getSystemObjekt().getSystemObject(), dbs,
                unsubscriptionIntervall);
        return konvertiere(resultData);
    }

    /**
     * Registriert einen Listener.
     *
     * @param asp      der betroffene Aspekt.
     * @param listener ein interessierte Listener.
     */
    @Override
    public void addUpdateListener(final Aspekt asp, final DatensatzUpdateListener listener) {
        addUpdateListener(asp, DataDescription.NO_SIMULATION_VARIANT_SET, listener);
    }

    /**
     * Registriert einen Listener für eine bestimmte Simulationsvariante.
     *
     * @param asp                 der betroffene Aspekt.
     * @param simulationsVariante die Simulationsvariante.
     * @param listener            ein interessierte Listener.
     */
    @Override
    public void addUpdateListener(final Aspekt asp, final short simulationsVariante,
            final DatensatzUpdateListener listener) {

        boolean anmelden;
        final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp), simulationsVariante);
        // Falls notwenig Initialisierung
        if (!updateListener.containsKey(dbs)) {
            updateListener.put(dbs, new EventListenerList());
        }

        // Listener für Aspekt registrieren
        anmelden = updateListener.get(dbs).getListenerCount(DatensatzUpdateListener.class) == 0;
        updateListener.get(dbs).add(DatensatzUpdateListener.class, listener);

        // Unter Aspekt als Empfänger anmelden
        if (anmelden) {
            receiver.anmelden(dbs);
        } else {
            /*
             * Beim Anmelden versendet der Datenverteiler die aktuellen Daten. Dieses
             * verhalten wird hiermit nachgebildet.
             */
            final T datum = getDatum(asp);
            listener.datensatzAktualisiert(new DatensatzUpdateEvent(this, asp, simulationsVariante, datum));
        }
    }

    /**
     * Deregistriert einen Listener. Beim Versuch einen nicht registrierten Listener
     * zu entfernen, wird keine Aktion ausgeführt.
     *
     * @param asp      der betroffene Aspekt.
     * @param listener ein nicht mehr interessierten Listener.
     */
    @Override
    public void removeUpdateListener(final Aspekt asp, final DatensatzUpdateListener listener) {
        removeUpdateListener(asp, DataDescription.NO_SIMULATION_VARIANT_SET, listener);
    }

    /**
     * Deregistriert einen Listener. Beim Versuch einen nicht registrierten Listener
     * zu entfernen, wird keine Aktion ausgeführt.
     *
     * @param asp                 der betroffene Aspekt.
     * @param simulationsVariante die betroffene Simulationsvariante.
     * @param listener            ein nicht mehr interessierten Listener.
     */
    @Override
    public void removeUpdateListener(final Aspekt asp, final short simulationsVariante,
            final DatensatzUpdateListener listener) {
        final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp), simulationsVariante);
        final EventListenerList listenerListe = updateListener.get(dbs);
        if (listenerListe != null) {
            listenerListe.remove(DatensatzUpdateListener.class, listener);
            if (listenerListe.getListenerCount(DatensatzUpdateListener.class) <= 0) {
                receiver.abmelden(dbs);
            }
        }
    }

    /**
     * Registriert einen Listener &uuml;r die Sendesteuerung.
     *
     * @param asp      der betroffene Aspekt.
     * @param listener ein interessierte Listener.
     */
    @Override
    public void addSendeSteuerungListener(final Aspekt asp, final SendeSteuerungListener listener) {
        addSendeSteuerungListener(asp, DataDescription.NO_SIMULATION_VARIANT_SET, listener);
    }

    /**
     * Registriert einen Listener für eine bestimmte Simulationsvariante.
     *
     * @param asp                 der betroffene Aspekt.
     * @param simulationsVariante die Simulationsvariante.
     * @param listener            ein interessierte Listener.
     */
    @Override
    public void addSendeSteuerungListener(final Aspekt asp, final short simulationsVariante,
            final SendeSteuerungListener listener) {
        final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp), simulationsVariante);
        // Falls notwendig Initialisierung
        if (!sendeSteuerungListeners.containsKey(dbs)) {
            sendeSteuerungListeners.put(dbs, new EventListenerList());
        }
        sendeSteuerungListeners.get(dbs).add(SendeSteuerungListener.class, listener);
        /*
         * Beim Anmelden versendet der Datenverteiler den aktuellen Status. Dieses
         * Verhalten wird hiermit nachgebildet.
         */
        final Status status = sender.getStatus(asp);
        listener.sendeSteuerungAktualisiert(new SendeSteuerungEvent(this, asp, simulationsVariante, status));
    }

    /**
     * Deregistriert einen Listener. Beim Versuch einen nicht registrierten Listener
     * zu entfernen, wird keine Aktion ausgeführt.
     *
     * @param asp      der betroffene Aspekt.
     * @param listener ein nicht mehr interessierten Listener.
     */
    @Override
    public void removeSendeSteuerungListener(final Aspekt asp, final SendeSteuerungListener listener) {
        removeSendeSteuerungListener(asp, DataDescription.NO_SIMULATION_VARIANT_SET, listener);
    }

    /**
     * Deregistriert einen Listener. Beim Versuch einen nicht registrierten Listener
     * zu entfernen, wird keine Aktion ausgeführt.
     *
     * @param asp                 der betroffene Aspekt.
     * @param simulationsVariante die betroffene Simulationsvariante.
     * @param listener            ein nicht mehr interessierten Listener.
     */
    @Override
    public void removeSendeSteuerungListener(final Aspekt asp, final short simulationsVariante,
            final SendeSteuerungListener listener) {
        final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp), simulationsVariante);
        final EventListenerList listenerListe = sendeSteuerungListeners.get(dbs);
        if (listenerListe != null) {
            listenerListe.remove(SendeSteuerungListener.class, listener);
        }
    }

    /**
     * Benachricht registrierte Listener &uuml;ber &Auml;nderungen am Datensatz.
     * Muss von abgeleiteten Klassen aufgerufen werden, wenn das Datum ge&auml;ndert
     * wurde.
     *
     * @param dbs   DataDescription
     * @param datum das Datum zum Zeitpunkt des Events.
     */
    protected void fireDatensatzAktualisiert(final DataDescription dbs, final OnlineDatum datum) {
        final DatensatzUpdateEvent event = new DatensatzUpdateEvent(this, getAspekt(dbs.getAspect()), datum);
        if (updateListener.get(dbs) != null) {
            for (final DatensatzUpdateListener listener : updateListener.get(dbs)
                    .getListeners(DatensatzUpdateListener.class)) {
                listener.datensatzAktualisiert(event);
            }
        }
    }

    /**
     * Benachricht registrierte Listener &uuml;ber &Auml;nderungen der
     * Sendesteuerung.
     *
     * @param dbs    DataDescription
     * @param status Status der Sendesteuerung
     *
     */
    protected void fireSendeSteuerungAktualisiert(final DataDescription dbs, final Status status) {
        final SendeSteuerungEvent event = new SendeSteuerungEvent(this, getAspekt(dbs.getAspect()), status);
        final EventListenerList eventListenerList = sendeSteuerungListeners.get(dbs);
        if (eventListenerList != null) {
            for (final SendeSteuerungListener listener : eventListenerList.getListeners(SendeSteuerungListener.class)) {
                listener.sendeSteuerungAktualisiert(event);
            }
        }
    }

    /**
     * Fragt, ob der Datensatz als Sender oder Quelle Daten senden darf.
     *
     * @param asp der betroffene Aspekt.
     * @return {@code true}, wenn der Datensatz als Sender oder Quelle Daten senden
     *         darf.
     */
    @Override
    public Status getStatusSendesteuerung(final Aspekt asp) {
        return sender.getStatus(asp);
    }

    /**
     * Fragt, ob der Datensatz als Sender oder Quelle angemeldet ist.
     *
     * @param asp der betroffene Aspekt.
     * @return {@code true}, wenn der Datensatz als Sender oder Quelle angemeldet
     *         ist.
     */
    @Override
    public boolean isAngemeldetSender(final Aspekt asp) {
        return sender.isAngemeldet(asp);
    }

    /**
     * Liest das Flag {@code autoUpdate}.
     *
     * @param asp der betroffene Aspekt.
     * @return {@code true}, wenn der Datensatz neue Daten automatisch vom
     *         Datenverteiler empf&auml;ngt.
     */
    @Override
    public boolean isAutoUpdate(final Aspekt asp) {
        return receiver.isAngemeldet(new DataDescription(getSystemObject(), getAspect(asp)));
    }

    /**
     * Veranlasst den Datensatz ein Datum an den Datenverteiler zusenden. Ist der
     * Zeitstempel des Datums nicht gesetzt oder gleich 0, wird automatisch der
     * aktuelle Zeitstempel beim Versand verwendet. Es wird das Standardtimeout
     * verwendet.
     *
     * @param asp   der betroffene Aspekt.
     * @param datum das zu sendende Datum.
     * @throws DatensendeException wenn die Daten nicht gesendet werden konnten. Der
     *                             Sendecache wird in dem Fall nicht geleert.
     * @see #createDatum()
     */
    @Override
    public void sendeDatum(final Aspekt asp, final T datum) throws DatensendeException {
        sendeDatum(asp, datum, TIMEOUT);
    }

    /**
     * Veranlasst den Datensatz ein Datum an den Datenverteiler zusenden. Ist der
     * Zeitstempel des Datums nicht gesetzt oder gleich 0, wird automatisch der
     * aktuelle Zeitstempel beim Versand verwendet.
     *
     * @param asp     der betroffene Aspekt.
     * @param datum   das zu sendende Datum.
     * @param timeout die Zeitspanne in der die Daten gesendet werden müssen.
     * @throws DatensendeException wenn die Daten nicht gesendet werden konnten. Der
     *                             Sendecache wird in dem Fall nicht geleert.
     * @see #createDatum()
     */
    @Override
    public void sendeDatum(final Aspekt asp, final T datum, final long timeout) throws DatensendeException {
        if (!sender.isAngemeldet(asp)) {
            throw new DatensendeException("Das Datum wurde nicht zum Senden angemeldet, Datensatz=" + this);
        }

        Status statusSendesteuerung = getStatusSendesteuerung(asp);
        if ((statusSendesteuerung != Status.START) && !isQuelle(asp)) {
            final DataDescription dbs = new DataDescription(getSystemObject(), asp.getSystemObject());
            try {
                SendRegistrationStore.waitForRegistration(getSystemObjekt().getSystemObject(), dbs);
                statusSendesteuerung = getStatusSendesteuerung(asp);
            } catch (final OperationTimedOutException ex) {
                throw new DatensendeException(ex);
            }
        }
        if ((statusSendesteuerung == Status.START)
                || (isQuelle(asp) && statusSendesteuerung != Status.ANMELDUNG_UNGUELTIG
                        && statusSendesteuerung != Status.KEINE_RECHTE)) {
            if (datum.dGetZeitstempel() == null) {
                datum.dSetZeitstempel(new Zeitstempel(getDav().getTime()));
            }
            sender.sende(konvertiere(datum), asp, datum.dGetZeitstempel());
        } else {
            throw new DatensendeException("Timeout, Quelle=" + isQuelle(asp) + ", Sendesteuerung="
                    + getStatusSendesteuerung(asp) + ", Datensatz=" + this + ", Daten=" + datum);
        }

        ((AbstractDatum) datum).dSetAspekt(asp);
    }

    @Override
    public boolean isQuelle(final Aspekt asp) {
        return quellen.contains(asp);
    }

    @Override
    public void setQuelle(final Aspekt asp, final boolean quelle) {
        if (quelle) {
            quellen.add(asp);
        } else {
            quellen.remove(asp);
        }
    }

    @Override
    public boolean isSenke(final Aspekt asp) {
        return senken.contains(asp);
    }

    @Override
    public void setSenke(final Aspekt asp, final boolean senke) {
        if (senke) {
            senken.add(asp);
        } else {
            senken.remove(asp);
        }
    }

    @Override
    public Iterator<T> getArchivdatenIterator(final Aspekt asp, final Interval intervall, final boolean nurAenderungen,
            final ArchiveDataKind... dataKinds) {
        final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp));
        final List<ArchiveDataSpecification> anfrage = ArchivUtilities.getAnfrage(
                Collections.singletonList(getSystemObjekt().getSystemObject()), dbs, intervall, nurAenderungen,
                dataKinds);
        final Iterator<ResultData> iterator = new ArchivIterator(getDav(), anfrage);

        return new Iterator<T>() {

            @Override
            public boolean hasNext() {
                return iterator.hasNext();
            }

            @Override
            public T next() {
                return konvertiere(iterator.next());
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }

        };
    }

    @Override
    public Iterator<T> getArchivdatenIterator(final Aspekt asp, final long zeitstempel, final int anzahlDatensaetze,
            final boolean nurAenderungen, final ArchiveDataKind... dataKinds) {
        final DataDescription dbs = new DataDescription(getSystemObject(), getAspect(asp));
        final List<ArchiveDataSpecification> anfrage = ArchivUtilities.getAnfrage(
                Collections.singletonList(getSystemObjekt().getSystemObject()), dbs, zeitstempel, anzahlDatensaetze,
                nurAenderungen, dataKinds);
        final Iterator<ResultData> iterator = new ArchivIterator(getDav(), anfrage);

        return new Iterator<T>() {

            @Override
            public boolean hasNext() {
                return iterator.hasNext();
            }

            @Override
            public T next() {
                return konvertiere(iterator.next());
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }

        };
    }

    @Override
    public List<T> getArchivdaten(final Aspekt asp, final Interval intervall, final boolean nurAenderungen,
            final ArchiveDataKind... dataKinds) {
        final List<T> result = new ArrayList<>();
        final Iterator<T> iterator = getArchivdatenIterator(asp, intervall, nurAenderungen, dataKinds);
        while (iterator.hasNext()) {
            result.add(iterator.next());
        }
        return result;
    }

    @Override
    public List<T> getArchivdaten(final Aspekt asp, final long zeitstempel, final int anzahlDatensaetze,
            final boolean nurAenderungen, final ArchiveDataKind... dataKinds) {
        final List<T> result = new ArrayList<>();
        final Iterator<T> iterator = getArchivdatenIterator(asp, zeitstempel, anzahlDatensaetze, nurAenderungen,
                dataKinds);
        while (iterator.hasNext()) {
            result.add(iterator.next());
        }
        return result;
    }

    @Override
    public Data konvertiere(final T datum) {
        final Data daten = createSendeCache();
        datum.bean2Atl(daten, getObjektFactory());
        return daten;
    }

    @Override
    public T konvertiere(final ResultData result) {
        check(result);

        final AbstractOnlineDatum datum = (AbstractOnlineDatum) createDatum();
        if (result.hasData()) {
            final Data daten = result.getData();
            datum.atl2Bean(daten, getObjektFactory());
        }

        datum.dSetDatenStatus(OnlineDatum.Status.getStatus(result.getDataState()));
        datum.dSetZeitstempel(new Zeitstempel(result.getDataTime()));
        final Aspect asp = result.getDataDescription().getAspect();
        datum.dSetAspekt(getAspekt(asp));

        return (T) datum;
    }

    @Override
    public void registrationStoreDataRequest(final SystemObject obj, final DataDescription desc, final byte state) {
        sender.dataRequest(obj, desc, state);
    }
}
