/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.impl;

import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.AnmeldeException;
import de.bsvrz.sys.funclib.bitctrl.modell.Aspekt;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensatzUpdateListener;
import de.bsvrz.sys.funclib.bitctrl.modell.DatensendeException;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.OnlineDatum;
import de.bsvrz.sys.funclib.bitctrl.modell.ParameterDatensatz;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;

/**
 * Implementiert gemeinsame Funktionen von Parametern.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @param <T> Der Typ des Datums den der Datensatz sichert.
 */
public abstract class AbstractParameterDatensatz<T extends OnlineDatum> extends AbstractOnlineDatensatz<T>
		implements ParameterDatensatz<T> {

	/**
	 * Die Defaultaspekte eines Parameters.
	 */
	protected static class DefaultAspekte extends AbstractSystemObjektInternal implements Aspekt {

		static Aspekt[] values() {
			return new Aspekt[] { DefaultAspekte.ParameterIst, DefaultAspekte.ParameterSoll,
					DefaultAspekte.ParameterVorgabe, DefaultAspekte.ParameterDefault, };
		}

		static final Aspekt ParameterIst = new DefaultAspekte("Ist-Parameter-Aspekt", "asp.parameterIst");
		static final Aspekt ParameterSoll = new DefaultAspekte("Soll-Parameter-Aspekt", "asp.parameterSoll");
		static final Aspekt ParameterVorgabe = new DefaultAspekte("Vorgabe-Parameter-Aspekt", "asp.parameterVorgabe");
		static final Aspekt ParameterDefault = new DefaultAspekte("Default-Parameter-Aspekt", "asp.parameterDefault");

		private final String name;
		private final String pid;

		private DefaultAspekte(final String name, final String pid) {
			this.name = name;
			this.pid = pid;
		}

		@Override
		public String getName() {
			if (getSystemObject() == null) {
				return name;
			}
			return super.getName();
		}

		@Override
		public String getPid() {
			if (getSystemObject() == null) {
				return pid;
			}
			return super.getPid();
		}

		@Override
		public Aspect getSystemObject() {
			return (Aspect) super.getSystemObject();
		}

		@Override
		protected String doGetTypPid() {
			return "typ.aspekt";
		}

	}

	/**
	 * Initialisiert den Parameterdatensatz.
	 *
	 * @param systemObjekt  das Systemobjekt zu dem der Datensatz gehört.
	 * @param objektFactory die ObjektFabrik, die der Datensatz verwenden soll.
	 */
	protected AbstractParameterDatensatz(final SystemObjekt systemObjekt, final ObjektFactory objektFactory) {
		super(systemObjekt, objektFactory);
	}

	@Override
	public void init(final SystemObject systemObject, final ObjektFactory objektFactory) {
		super.init(systemObject, objektFactory);
		for (final Aspekt asp : DefaultAspekte.values()) {
			((AbstractSystemObjektInternal) asp).init(objektFactory.getModellobjekt(asp.getPid()).getSystemObject(),
					objektFactory);
		}
	}

	@Override
	public void abmeldenSender() {
		abmeldenSender(DefaultAspekte.ParameterVorgabe);
	}

	@Override
	public T getDatum() {
		return super.getDatum(DefaultAspekte.ParameterSoll);
	}

	@Override
	public void addUpdateListener(final DatensatzUpdateListener l) {
		addUpdateListener(DefaultAspekte.ParameterSoll, l);
	}

	@Override
	public void anmeldenSender() throws AnmeldeException {
		anmeldenSender(DefaultAspekte.ParameterVorgabe);
	}

	@Override
	public Status getStatusSendesteuerung() {
		return super.getStatusSendesteuerung(DefaultAspekte.ParameterVorgabe);
	}

	@Override
	public boolean isAngemeldetSender() {
		return isAngemeldetSender(DefaultAspekte.ParameterVorgabe);
	}

	@Override
	public boolean isAutoUpdate() {
		return isAutoUpdate(DefaultAspekte.ParameterVorgabe);
	}

	@Override
	public void removeUpdateListener(final DatensatzUpdateListener l) {
		removeUpdateListener(DefaultAspekte.ParameterSoll, l);
	}

	@Override
	public void sendeDatum(final T datum) throws DatensendeException {
		sendeDatum(DefaultAspekte.ParameterVorgabe, datum);
	}

	@Override
	public void sendeDatum(final T datum, final long timeout) throws DatensendeException {
		sendeDatum(DefaultAspekte.ParameterVorgabe, datum, timeout);
	}

}
