/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.impl;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.Datensatz;
import de.bsvrz.sys.funclib.bitctrl.modell.Datum;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;

/**
 * Implementierung der gemeinsamen Methoden der Systemobjektschnittstelle.
 *
 * @author BitCtrl Systems GmbH, Schumann
 */
public abstract class AbstractSystemObjekt extends AbstractSystemObjektInternal {

	private final Map<Class<? extends Datensatz<? extends Datum>>, Datensatz<? extends Datum>> datensaetze = Collections
			.synchronizedMap(new HashMap<Class<? extends Datensatz<? extends Datum>>, Datensatz<? extends Datum>>());

	/**
	 * Der Defaultkonstruktor tut nichts.
	 *
	 * <p>
	 * <em>Hinweis:</em> Wenn der Defaultkonstruktor verwendet wird, muss das
	 * Systemobjekt mit {@link #init(SystemObject, ObjektFactory)} initialisietr
	 * werden, bevor es benutzt wird.
	 *
	 * @see #init(SystemObject, ObjektFactory)
	 */
	protected AbstractSystemObjekt() {
		// tut nix
	}

	/**
	 * Ruft {@link #init(SystemObject, ObjektFactory)} auf.
	 *
	 * @param objekt  das zu kapselnde Systemobjekt.
	 * @param factory die Factory, die das Objekt erzeugt hat.
	 */
	protected AbstractSystemObjekt(final SystemObject objekt, final ObjektFactory factory) {
		super(objekt, factory);
	}

	/**
	 * Generiert aus der Datensatzklasse ein Objekt. Dazu muss ein öffentlicher
	 * Konstruktor existieren, der als einzigen Parameter ein SystemObjekt
	 * entgegennimmt.
	 *
	 * @param datensatzTyp die Klasse eines Datensatzes.
	 * @param <D>          Der Typ des Datensatzes.
	 * @return ein Objekt der Klasse oder {@code null}, wenn der Datensatz am
	 *         Systemobjekt nicht unterstützt wird.
	 */
	public final <D extends Datensatz<? extends Datum>> D getDatensatz(final Class<D> datensatzTyp) {

		D result = null;

		/*
		 * das Abfragen und gegegebenfalls Anlegen eines Datensatzes muss synchronisiert
		 * erfolgen, da ansonsten mehrere Instanzen eines Datensatzes für das
		 * eigentliche gleiche Datum enstehen können, von denen letztendlich nur einer
		 * sinnvoll in Verwendung bleibt.
		 */
		synchronized (datensaetze) {

			if (datensaetze.containsKey(datensatzTyp)) {
				return (D) datensaetze.get(datensatzTyp);
			}

			if (Modifier.isAbstract(datensatzTyp.getModifiers()) || Modifier.isInterface(datensatzTyp.getModifiers())) {
				throw new IllegalArgumentException(
						"Datensatz " + datensatzTyp.getName() + " kann nicht instantiiert werden, da es sich um eine "
								+ "Schnittstelle oder abstrakte Klasse handelt.");
			}

			for (final Constructor<?> c : datensatzTyp.getConstructors()) {
				final Class<?>[] parameterTypes = c.getParameterTypes();

				if (Modifier.isPublic(c.getModifiers()) && (parameterTypes.length == 2)
						&& parameterTypes[0].isAssignableFrom(getClass())
						&& parameterTypes[1].isAssignableFrom(ObjektFactory.class)) {
					try {
						result = (D) c.newInstance(this, getObjektFactory());
					} catch (final IllegalArgumentException | InstantiationException | IllegalAccessException | InvocationTargetException ex) {
						// Tritt ein, wenn der aufgerufene Konstruktor eine
						// Exception geworfen hat
						throw new IllegalArgumentException(
								"Datensatz " + datensatzTyp.getName() + " kann nicht instantiiert werden.", ex);
					}
				}
			}

			if (result == null) {
				throw new IllegalArgumentException(
						"Datensatz " + datensatzTyp + " kann nicht instantiiert werden, da der öffentlicher "
								+ "Konstruktor mit einem Parameter vom Typ SystemObjekt fehlt.");
			}

			final List<AttributeGroup> atgListe = getSystemObject().getType().getAttributeGroups();
			if (atgListe.contains(result.getSystemObject())) {
				datensaetze.put(datensatzTyp, result);
			} else {
				throw new IllegalArgumentException("Datensatz " + result.getSystemObject() + " kann nicht mit Objekt "
						+ getSystemObject().getType()
						+ " verwendet werden. Das Modell ist möglicherweise nicht mehr aktuell.");
			}
		}

		return result;
	}

}
