/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.bitctrl.internal.modell.ModellUtil;
import de.bsvrz.sys.funclib.bitctrl.modell.impl.AbstractSystemObjekt;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Globale Registry für alle in der Applikation zur Verfügung stehenden
 * {@link ModellObjektFactory}s.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @version $Id$
 */
public class ModellObjektFactoryRegistry {

	private static class DefaultModellObjektFactory implements ModellObjektFactory {

		private final Debug log = Debug.getLogger();

		/** Tabelle für die Zuordnung nicht modellierter Systemobjekt-Typen, */
		private final Map<SystemObjectType, SystemObjectType> replacementMap = new HashMap<>();

		@Override
		public SystemObjekt createModellobjekt(final SystemObject obj, final ObjektFactory factory) {
			try {

				final Class<?> clazz = getSystemObjectTypeClass(obj.getType());
				if (clazz != null) {
					final AbstractSystemObjekt modell = (AbstractSystemObjekt) clazz.newInstance();
					modell.init(obj, factory);
					return modell;
				}

				/*
				 * Der Typ des Systemobjekts ist im Modell nicht vorhanden und es gibt keinen
				 * bekannten Basistyp, also wird ein Dummy-Objekt zurückgeliefert.
				 */
				return new MockSystemObjekt(obj, factory);
			} catch (final InstantiationException | IllegalAccessException ex) {
				throw new IllegalStateException(ex);
			}
		}

		private Class<?> getSystemObjectTypeClass(final SystemObjectType type) {
			try {
				final String className = ModellUtil.determineJavaPackage(type) + ".objekte.impl."
						+ ModellUtil.determineJavaName(type) + "Impl";
				return Class.forName(className);
			} catch (final ClassNotFoundException ex) {
				SystemObjectType newType = replacementMap.get(type);
				if (newType == null) {

					log.warning("Der Systemobjekttyp " + type + " ist unbekannt, es wird allgemeinerer Typ gesucht.");

					final List<SystemObjectType> superTypes = type.getSuperTypes();
					newType = superTypes.get(0);
					if (superTypes.size() > 1) {
						log.warning("Der Systemobjekttyp " + type + " ist von mehreren Typen abgeleitet (" + superTypes
								+ "). Es wird der erste gefundene Typ (" + newType + ") als verwendet.");
					}

					replacementMap.put(type, newType);
				}

				if (!newType.isBaseType()) {
					return getSystemObjectTypeClass(newType);
				}

				return null;
			}
		}

		@Override
		public Collection<? extends SystemObjektTyp> getTypen() {
			return Collections.emptySet();
		}

	}

	/**
	 * Die einzige Instanz der Registry (Singleton).
	 */
	public static final ModellObjektFactoryRegistry INSTANZ = new ModellObjektFactoryRegistry();

	/**
	 * Diese Factory kann als Fallback verwendet werdet werden, wenn keine passende
	 * Factory registriert wurde. Diese Factory kann aber ebenfalls nicht
	 * sicherstellen, dass ein Modellobjekt generiert werden kann.
	 */
	public static final ModellObjektFactory DEFAULT_FACTORY = new DefaultModellObjektFactory();

	private final Set<Class<? extends ModellObjektFactory>> factories = new HashSet<>();
	private final Map<SystemObjektTyp, ModellObjektFactory> typ2factories = new HashMap<>();
	private final Map<String, ModellObjektFactory> typPid2factories = new HashMap<>();

	/**
	 * Registriert eine neue Modellobjektfabrik.
	 *
	 * <p>
	 * Die Factory wird nur registriert, wenn noch keine Instanz der übergebenen
	 * Factory registriert wurde. Damit sollen doppelte Registrierungen
	 * ausgeschlossen werden.
	 *
	 * @param factoryClass die Klasse der zu registrierenden Modellobjektfabriken.
	 */
	public synchronized void addFactory(final Class<? extends ModellObjektFactory> factoryClass) {
		if (!factories.contains(factoryClass)) {
			try {
				final ModellObjektFactory factory = factoryClass.newInstance();

				for (final SystemObjektTyp typ : factory.getTypen()) {
					typ2factories.put(typ, factory);
					typPid2factories.put(typ.getPid(), factory);
				}

				factories.add(factoryClass);
			} catch (final InstantiationException ex) {
				throw new IllegalArgumentException("Kann keine Instanz der Fabrik anlegen: " + factoryClass, ex);
			} catch (final IllegalAccessException ex) {
				throw new IllegalArgumentException("Der Konstruktor der Fabrik ist nicht sichtbar: " + factoryClass,
						ex);
			}
		}
	}

	/**
	 * Entfernt eine Modellobjektfabrik wieder aus der Registrierung. Wurde die
	 * Fabrik nicht registriert, passiert nichts.
	 *
	 * @param factoryClass die aus der Registrierung zu entfernende Fabrik.
	 */
	public synchronized void removeFactory(final Class<? extends ModellObjektFactory> factoryClass) {
		final Iterator<Map.Entry<String, ModellObjektFactory>> pidIterator = typPid2factories.entrySet().iterator();
		while (pidIterator.hasNext()) {
			final Map.Entry<String, ModellObjektFactory> entry = pidIterator.next();
			if (entry.getValue().getClass() == factoryClass) {
				pidIterator.remove();
			}
		}

		final Iterator<Map.Entry<SystemObjektTyp, ModellObjektFactory>> typIterator = typ2factories.entrySet()
				.iterator();
		while (typIterator.hasNext()) {
			final Map.Entry<SystemObjektTyp, ModellObjektFactory> entry = typIterator.next();
			if (entry.getValue().getClass() == factoryClass) {
				typIterator.remove();
			}
		}

		factories.remove(factoryClass);
	}

	/**
	 * Gibt die Fabrik zurück, die benötigt wird, um das Modellobjekt zu einem
	 * Datenverteilerobjekte zu bestimmten.
	 *
	 * @param typ der Objekttyp zu dem die Fabrik gesucht wird.
	 * @return die Fabrik, die Objekte des angegebenen Typs anlegt oder
	 *         <code>null</code>, wenn keine passende Fabrik registriert wurde.
	 */
	public ModellObjektFactory getFactory(final SystemObjectType typ) {
		return typPid2factories.get(typ.getPid());
	}

	/**
	 * Gibt die Fabrik zurück, die benötigt wird, um Objekte eines bestimmten Typs
	 * anzulegen.
	 *
	 * @param typ der Objekttyp zu dem die Fabrik gesucht wird.
	 * @return die Fabrik, die Objekte des angegebenen Typs anlegt oder
	 *         <code>null</code>, wenn keine passende Fabrik registriert wurde.
	 */
	public ModellObjektFactory getFactory(final SystemObjektTyp typ) {
		return typ2factories.get(typ);
	}

	/**
	 * Gibt die Fabrik zurück, die benötigt wird, um Objekte eines bestimmten Typs
	 * anzulegen.
	 *
	 * @param typPid die PID des Objekttyps zu dem die Fabrik gesucht wird.
	 * @return die Fabrik, die Objekte des angegebenen Typs anlegt oder
	 *         <code>null</code>, wenn keine passende Fabrik registriert wurde.
	 */
	public ModellObjektFactory getFactory(final String typPid) {
		return typPid2factories.get(typPid);
	}

	/**
	 * Gibt die Menge aller unterstützen Systemobjekttypen zurück.
	 *
	 * @return die Systemobjekttypen für die Objekte mit den registrieren Fabriken
	 *         erzeugt werden können.
	 */
	public Set<SystemObjektTyp> getSystemObjektTypen() {
		return Collections.unmodifiableSet(typ2factories.keySet());
	}

	private ModellObjektFactoryRegistry() {
		// Keine Instanz von Utility-Klasse erlaubt
	}

}
