/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell;

import java.util.Collection;
import java.util.List;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.daf.DavProvider;
import de.bsvrz.sys.funclib.dynobj.DynObjektException;

/**
 * Dient dem Erzeugen von Objekten des logischen Modells aus dem Datenmodell des
 * Datenverteilers. Eine "Super-Factory" für Modellobjekte.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public interface ObjektFactory extends DavProvider {

	/**
	 * Legt die zu verwendende Datenverteilerverbindung fest.
	 *
	 * @param dav die zu verwendende Verbindung.
	 */
	void setDav(final ClientDavInterface dav);

	/**
	 * Bestimmt eine Liste der Modellobjekte zu vorgegebenen PIDs von
	 * Systemobjekten. Ist eine PID ein Konfigurationsbereich, werden alle Objekte
	 * des Konfigurationsbereichs bestimmt. Ist eine PID ein Typ, werden alle
	 * Objekte des Typs bestimmt.
	 *
	 * <p>
	 * Für einen Konfigurationsbereich werden die enthaltenen Typen nicht aufgelöst.
	 * Da laut Konvention Typen und Objekte nicht im selben Konfigurationsbereich
	 * angelegt werden sollen, ist dieses Verhalten angebracht. Ansonsten könnte man
	 * die definierten Typen eines Konfigurationsbereichs bestimmen.
	 *
	 * @param pids PIDs der zu &uuml;bersetzenden Systemobjekte
	 * @return Tabelle von IDs und Modellobjekten
	 */
	List<? extends SystemObjekt> bestimmeModellobjekte(final String... pids);

	/**
	 * Bestimmt das Modellobjekt zu einer PID.
	 *
	 * @param pid die PID des gesuchten Systemobjekts.
	 * @return das Systemobjekt oder {@code null}, wenn keines existiert oder es
	 *         nicht als Modellobjekt darstellbar ist.
	 */
	SystemObjekt getModellobjekt(final String pid);

	/**
	 * Sucht mit Hilfe der registrierten Fabriken nach einem Systemobjekt, dessen Id
	 * bekannt ist.
	 *
	 * @param id die Id eines Systemobjekts.
	 * @return das korrespondierende Modellobjekt oder {@code null}, wenn es keines
	 *         mit der Id gibt.
	 */
	SystemObjekt getModellobjekt(final long id);

	/**
	 * Versucht mit Hilfe der registrierten Fabriken ein Systemobjekt in ein
	 * Modellobjekt zu überführen. Gibt es mehrere Fabriken, die dazu in der Lage
	 * sind, wird die Fabrik benutzt, die zuerst registriert wurde. Existiert keine
	 * passende Fabrik, wird generisches {@link SystemObjekt} zurückgegeben.
	 *
	 *
	 * @param obj Ein Systemobjekt
	 * @return das korrespondierende Modellobjekt, niemals {@code null}.
	 */
	SystemObjekt getModellobjekt(final SystemObject obj);

	/**
	 * Sammelt alle erzeugbaren Objekttypen der registrierten Factories.
	 *
	 * @return die Menge der erzeugbaren Objekttypen.
	 */
	Collection<SystemObjektTyp> getTypen();

	/**
	 * Erzeugt ein dynamisches Objekt.
	 *
	 * @param <T>   der Typ des zu erzeugenden Objekts.
	 * @param clazz die Klasse des zu erzeugenden Objekts.
	 * @param name  der Name des Objekts.
	 * @param pid   die PID des Objekts.
	 * @param daten optionale Menge von Konfigurierenden Daten die initial am Objekt
	 *              gesetzt werden sollen.
	 * @return das angelegte Objekt.
	 * @throws DynObjektException wenn das Erzeugen schiefgegangen ist.
	 */
	<T extends DynamischesObjekt> T createDynamischesObjekt(final Class<T> clazz, final String name, final String pid,
			final KonfigurationsDatum... daten) throws DynObjektException;

	/**
	 * Invalidiert ein dynamisches Objekt.
	 *
	 * <p>
	 * Das Invalidieren ist nicht gleichzusetzen mit dem Löschen eines Objekts. An
	 * dem betroffenen Objekt wird lediglich ein <em>gelöscht</em>-Flag gesetzt. Das
	 * Objekt wird nun von den normalen Operationen des Datenverteilers nicht mehr
	 * berührt. Das Objekt kann jedoch noch mit speziellen Methoden wiedergeholt
	 * werden, wobei nicht das Objekt wiederhergestellt wird, sondern hierbei wird
	 * lediglich das <em>gelöscht</em> -Flag ignoriert.
	 *
	 * @param objekt das zu invalidierende Objekt.
	 * @throws DynObjektException wenn das invalidieren schiefgegangen ist.
	 */
	void invalidateDynamischesObjekt(final DynamischesObjekt objekt) throws DynObjektException;

	/**
	 * Ruft die konfigurierenden Daten mehrerer Objekte in einem Vorgang ab. Werden
	 * die Daten sehr vieler Objekte benötigt, sollte diese Methode verwendet
	 * werden, anstatt jedes Datum einzeln abzurufen.
	 *
	 * @param <T>          der Typ der Daten.
	 * @param objekte      die Objekte für die die Daten abgerufen werden sollen.
	 * @param datensatzTyp der konfigurierende Datensatz der abgerufen werden soll.
	 * @return die konfigurienden Daten in der selben Reihenfolge, wie die
	 *         Objektliste.
	 */
	<T extends KonfigurationsDatum> List<T> getDaten(final List<? extends SystemObjekt> objekte,
			final Class<? extends KonfigurationsDatensatz<T>> datensatzTyp);

}
