/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;

import com.bitctrl.util.Interval;

import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.bitctrl.archiv.ArchivIterator;

/**
 * Schnittstelle f&uuml;r Onlinedatens&auml;tze.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @param <T> Der Typ des Datums den der Datensatz sichert.
 */
public interface OnlineDatensatz<T extends OnlineDatum> extends Datensatz<T> {

	/**
	 * Die Statuscodes der Sendesteuerung des Datenverteilers als {@code Enum}.
	 */
	public enum Status {

		/** Der Versand von Daten kann gestartet werden. */
		START(ClientSenderInterface.START_SENDING, "Der Versand von Daten kann gestartet werden."),

		/**
		 * Der Versand von Daten soll angehalten werden, weil momentan kein Abnehmer
		 * sich für die Daten interessiert.
		 */
		STOP(ClientSenderInterface.STOP_SENDING,
				"Der Versand von Daten soll angehalten werden, weil momentan kein Abnehmer sich für die Daten interessiert."),

		/**
		 * Der Versand von Daten soll angehalten werden, weil momentan keine Rechte für
		 * den Versand vorliegen.
		 */
		KEINE_RECHTE(ClientSenderInterface.STOP_SENDING_NO_RIGHTS,
				"Der Versand von Daten soll angehalten werden, weil momentan keine Rechte für den Versand vorliegen."),

		/**
		 * Der Versand von Daten soll angehalten werden, weil die entsprechende
		 * Anmeldung momentan nicht gültig ist (z.B. wegen doppelter Quelle).
		 */
		ANMELDUNG_UNGUELTIG(ClientSenderInterface.STOP_SENDING_NOT_A_VALID_SUBSCRIPTION,
				"Der Versand von Daten soll angehalten werden, weil die entsprechende Anmeldung momentan nicht gültig ist (z.B. wegen doppelter Quelle).");

		/**
		 * Bestimmt den Status zu einem Code.
		 *
		 * @param code ein Code.
		 * @return der gesuchte Status.
		 */
		public static Status getStatus(final int code) {
			for (final Status s : values()) {
				if (s.getCode() == code) {
					return s;
				}
			}

			throw new NoSuchElementException("Ungültiger Code");
		}

		/** Die Eigenschaft {@code beschreibung}. */
		private final String beschreibung;

		/** Die Eigenschaft {@code code}. */
		private final int code;

		/**
		 * Initialisiert das Objekt.
		 *
		 * @param code         der Statuscode in den
		 *                     Datenverteilerapplikationsfunktionen.
		 * @param beschreibung die Beschreibung des Status.
		 */
		Status(final int code, final String beschreibung) {
			this.code = code;
			this.beschreibung = beschreibung;
		}

		/**
		 * Gibt die Beschreibung des Status wieder.
		 *
		 * @return {@code beschreibung}.
		 */
		public String getBeschreibung() {
			return beschreibung;
		}

		/**
		 * Gibt den passenden Statuscode in den Datenverteilerapplikationsfunktionen
		 * wieder.
		 *
		 * @return {@code code}.
		 */
		public int getCode() {
			return code;
		}

	}

	/**
	 * Meldet den Datensatz als Sender oder Quelle am Datenverteiler an.
	 *
	 * @param asp der betroffene Aspekt.
	 * @throws AnmeldeException wenn die Anmeldung nicht erfolgreich war.
	 */
	void anmeldenSender(Aspekt asp) throws AnmeldeException;

	/**
	 * Meldet den Datensatz als Quelle am Datenverteiler an und versendet den
	 * übergebenen Datensatz.
	 *
	 * @param asp   der betroffene Aspekt.
	 * @param datum der initiale Datensatz
	 * @throws AnmeldeException wenn die Anmeldung nicht erfolgreich war.
	 */
	void anmeldenQuelle(final Aspekt asp, final T datum) throws AnmeldeException;

	/**
	 * Meldet eine eventuell vorhandene Anmeldung als Sender oder Quelle wieder ab.
	 *
	 * @param asp der betroffene Aspekt.
	 */
	void abmeldenSender(Aspekt asp);

	/**
	 * Registriert einen Listener.
	 *
	 * @param asp der betroffene Aspekt.
	 * @param l   ein interessierte Listener.
	 */
	void addUpdateListener(Aspekt asp, DatensatzUpdateListener l);

	/**
	 * Registriert einen Listener für eine bestimmte Simulationsvariante.
	 *
	 * @param asp                 der betroffene Aspekt.
	 * @param simulationsVariante die Simulationsvariante.
	 * @param listener            ein interessierte Listener.
	 */
	void addUpdateListener(final Aspekt asp, final short simulationsVariante, final DatensatzUpdateListener listener);

	/**
	 * Deregistriert einen Listener.
	 *
	 * @param asp der betroffene Aspekt.
	 * @param l   ein nicht mehr interessierten Listener.
	 */
	void removeUpdateListener(Aspekt asp, DatensatzUpdateListener l);

	/**
	 * Deregistriert einen Listener.
	 *
	 * @param asp                 der betroffene Aspekt.
	 * @param simulationsVariante die betroffene Simulationsvariante.
	 * @param listener            ein nicht mehr interessierten Listener.
	 */
	void removeUpdateListener(final Aspekt asp, final short simulationsVariante,
			final DatensatzUpdateListener listener);

	/**
	 * Registriert einen Listener &uuml;r die Sendesteuerung.
	 *
	 * @param asp      der betroffene Aspekt.
	 * @param listener ein interessierte Listener.
	 */
	void addSendeSteuerungListener(final Aspekt asp, final SendeSteuerungListener listener);

	/**
	 * Registriert einen Listener für eine bestimmte Simulationsvariante.
	 *
	 * @param asp                 der betroffene Aspekt.
	 * @param simulationsVariante die Simulationsvariante.
	 * @param listener            ein interessierte Listener.
	 */
	void addSendeSteuerungListener(final Aspekt asp, final short simulationsVariante,
			final SendeSteuerungListener listener);

	/**
	 * Deregistriert einen Listener. Beim Versuch einen nicht registrierten Listener
	 * zu entfernen, wird keine Aktion ausgeführt.
	 *
	 * @param asp      der betroffene Aspekt.
	 * @param listener ein nicht mehr interessierten Listener.
	 */
	void removeSendeSteuerungListener(final Aspekt asp, final SendeSteuerungListener listener);

	/**
	 * Deregistriert einen Listener. Beim Versuch einen nicht registrierten Listener
	 * zu entfernen, wird keine Aktion ausgeführt.
	 *
	 * @param asp                 der betroffene Aspekt.
	 * @param simulationsVariante die betroffene Simulationsvariante.
	 * @param listener            ein nicht mehr interessierten Listener.
	 */
	void removeSendeSteuerungListener(final Aspekt asp, final short simulationsVariante,
			final SendeSteuerungListener listener);

	/**
	 * Gibt die verf&uuml;gbaren Aspekte zur&uuml;ck.
	 *
	 * @return die Menge der verf&uuml;gbaren Aspekte.
	 */
	@Override
	Collection<? extends Aspekt> getAspekte();

	/**
	 * Fragt, ob der Datensatz als Sender oder Quelle Daten senden darf.
	 *
	 * @param asp der betroffene Aspekt.
	 * @return {@code true}, wenn der Datensatz als Sender oder Quelle Daten senden
	 *         darf.
	 */
	Status getStatusSendesteuerung(Aspekt asp);

	/**
	 * Fragt, ob der Datensatz als Sender oder Quelle angemeldet ist.
	 *
	 * @param asp der betroffene Aspekt.
	 * @return {@code true}, wenn der Datensatz als Sender oder Quelle angemeldet
	 *         ist.
	 */
	boolean isAngemeldetSender(Aspekt asp);

	/**
	 * Liest das Flag {@code autoUpdate}.
	 *
	 * @param asp der betroffene Aspekt.
	 * @return {@code true}, wenn der Datensatz neue Daten automatisch vom
	 *         Datenverteiler empf&auml;ngt.
	 */
	boolean isAutoUpdate(Aspekt asp);

	/**
	 * Gibt das Flag {@code quelle} zur&uuml;ck.
	 *
	 * @param asp der betroffene Aspekt.
	 * @return {@code true}, wenn der Datensatz als Quelle und {@code false}, wenn
	 *         er als Sender angemeldet werden soll.
	 */
	boolean isQuelle(Aspekt asp);

	/**
	 * Legt fest, ob Anmeldungen als Quelle durchgef&uuml;hrt werden sollen. Eine
	 * bereits bestehende Anmeldung wird dadurch nicht beeinflusst.
	 *
	 * @param asp    der betroffene Aspekt.
	 * @param quelle {@code true}, wenn die Anmeldung als Quelle erfolgen soll,
	 *               ansonsten erfolgt sie als Sender.
	 */
	void setQuelle(Aspekt asp, boolean quelle);

	/**
	 * Gibt das Flag {@code senke} zur&uuml;ck.
	 *
	 * @param asp der betroffene Aspekt.
	 * @return {@code true}, wenn der Datensatz als Senke und {@code false}, wenn er
	 *         als Empf&auml;nger angemeldet werden soll.
	 */
	boolean isSenke(Aspekt asp);

	/**
	 * Legt fest, ob Anmeldungen als Senke durchgef&uuml;hrt werden sollen. Eine
	 * bereits bestehende Anmeldung wird dadurch nicht beeinflusst.
	 *
	 * @param asp   der betroffene Aspekt.
	 * @param senke {@code true}, wenn die Anmeldung als Senke erfolgen soll,
	 *              ansonsten erfolgt sie als Empf&auml;nger.
	 */
	void setSenke(Aspekt asp, boolean senke);

	/**
	 * Veranlasst den Datensatz ein Datum an den Datenverteiler zusenden. Ist der
	 * Zeitstempel des Datums nicht gesetzt oder gleich 0, wird automatisch der
	 * aktuelle Zeitstempel beim Versand verwendet.
	 *
	 * @param asp   der betroffene Aspekt.
	 * @param datum das zu sendende Datum.
	 * @throws DatensendeException wenn die Daten nicht gesendet werden konnten. Der
	 *                             Sendecache wird in dem Fall nicht geleert.
	 * @see #createDatum()
	 */
	void sendeDatum(Aspekt asp, T datum) throws DatensendeException;

	/**
	 * Veranlasst den Datensatz ein Datum an den Datenverteiler zusenden. Ist der
	 * Zeitstempel des Datums nicht gesetzt oder gleich 0, wird automatisch der
	 * aktuelle Zeitstempel beim Versand verwendet.
	 *
	 * @param asp     der betroffene Aspekt.
	 * @param datum   das zu sendende Datum.
	 * @param timeout die Zeit in der der Datensatz gesendet werden muss.
	 * @throws DatensendeException wenn die Daten nicht gesendet werden konnten. Der
	 *                             Sendecache wird in dem Fall nicht geleert.
	 * @see #createDatum()
	 */
	void sendeDatum(Aspekt asp, T datum, long timeout) throws DatensendeException;

	/**
	 * Erzeugt aus den Parametern eine äquivalente Archivanfrage für einen Zeitraum.
	 *
	 * @param asp            der Aspekt für den Daten gesucht werden.
	 * @param intervall      das Zeitintervall der Archivanfrage.
	 * @param nurAenderungen <code>true</code>, wenn nur geänderten Datensätze
	 *                       zurückgeben werden sollen. Aufeinanderfolgende
	 *                       identische Datensätze werden hierbei zu einem Datensatz
	 *                       zusammengefasst.
	 * @param dataKinds      die gewünschten Datensatzarten. Wenn nicht angegeben,
	 *                       werden nur Onlinedaten abgefragt.
	 * @return die Liste der Archivanfragen.
	 */
	Iterator<T> getArchivdatenIterator(Aspekt asp, Interval intervall, boolean nurAenderungen,
			ArchiveDataKind... dataKinds);

	/**
	 * Erzeugt aus den Parametern eine äquivalente Archivanfrage für eine bestimmte
	 * Anzahl Datensätze vor einem Endzeitpunkt.
	 *
	 * @param asp               der Aspekt für den Daten gesucht werden.
	 * @param zeitstempel       der Zeitpunkt vor dem die Datensätze liegen sollen.
	 * @param anzahlDatensaetze die Anzahl der gewünschten Datensätze.
	 * @param nurAenderungen    <code>true</code>, wenn nur geänderten Datensätze
	 *                          zurückgeben werden sollen. Aufeinanderfolgende
	 *                          identische Datensätze werden hierbei zu einem
	 *                          Datensatz zusammengefasst.
	 * @param dataKinds         die gewünschten Datensatzarten. Wenn nicht
	 *                          angegeben, werden nur Onlinedaten abgefragt.
	 * @return die Liste der Archivanfragen.
	 */
	Iterator<T> getArchivdatenIterator(Aspekt asp, long zeitstempel, int anzahlDatensaetze, boolean nurAenderungen,
			ArchiveDataKind... dataKinds);

	/**
	 * Ruft Archivdaten in einen Rutsch ab. Diese Methode sollte nur verwendet
	 * werden, wenn die zu erwartenden Liste der Archivdaten nicht zu groß ist.
	 *
	 * <p>
	 * <em>Hinweis:</em> Diese Methode sollte nur für Anfragen benutzt werden, die
	 * relativ kleine Datenmengen abfragen, da die Abfrage sonst sehr lange dauern
	 * oder gar fehlschlagen kann. Besser ist es die Methode
	 * {@link #getArchivdatenIterator(Aspekt, Interval, boolean, ArchiveDataKind...)}
	 * zu verwenden.
	 *
	 * @param asp            der Aspekt für den Daten gesucht werden.
	 * @param intervall      das Zeitintervall der Archivanfrage.
	 * @param nurAenderungen <code>true</code>, wenn nur geänderten Datensätze
	 *                       zurückgeben werden sollen. Aufeinanderfolgende
	 *                       identische Datensätze werden hierbei zu einem Datensatz
	 *                       zusammengefasst.
	 * @param dataKinds      die gewünschten Datensatzarten. Wenn nicht angegeben,
	 *                       werden nur Onlinedaten abgefragt.
	 * @return die Liste der Archivdaten.
	 * @see #getArchivdatenIterator(Aspekt, Interval, boolean, ArchiveDataKind...)
	 */
	List<T> getArchivdaten(Aspekt asp, Interval intervall, boolean nurAenderungen, ArchiveDataKind... dataKinds);

	/**
	 * Liefert eine beliebige Anzahl an Archivdatensätzen vor einem definierten
	 * Zeitpunkt.
	 *
	 * <p>
	 * <em>Hinweis:</em> Diese Methode sollte nur für Anfragen benutzt werden, die
	 * relativ kleine Datenmengen abfragen, da die Abfrage sonst sehr lange dauern
	 * oder gar fehlschlagen kann. Besser ist es die Methode
	 * {@link #getArchivdatenIterator(Aspekt, long, int, boolean, ArchiveDataKind...)}
	 * zu verwenden.
	 *
	 * @param asp               der Aspekt für den Daten gesucht werden.
	 * @param zeitstempel       der Zeitpunkt vor dem die Datensätze liegen sollen.
	 * @param anzahlDatensaetze die Anzahl der gewünschten Datensätze.
	 * @param nurAenderungen    <code>true</code>, wenn nur geänderten Datensätze
	 *                          zurückgeben werden sollen. Aufeinanderfolgende
	 *                          identische Datensätze werden hierbei zu einem
	 *                          Datensatz zusammengefasst.
	 * @param dataKinds         die gewünschten Datensatzarten. Wenn nicht
	 *                          angegeben, werden nur Onlinedaten abgefragt.
	 * @return die Liste der Archivdaten.
	 * @see ArchivIterator
	 * @see #getArchivdatenIterator(Aspekt, long, int, boolean, ArchiveDataKind...)
	 */
	List<T> getArchivdaten(Aspekt asp, long zeitstempel, int anzahlDatensaetze, boolean nurAenderungen,
			ArchiveDataKind... dataKinds);

	/**
	 * Gibt die aktuellen Daten des Datensatzes zurück. Die implizite
	 * Empfängeranmeldung wird sofort abgemeldet.
	 *
	 * @param asp der betroffene Aspekt.
	 * @return ein Datum, welches die Daten des Datensatzes kapselt.
	 */
	T getSingleDatum(Aspekt asp);

	/**
	 * Gibt die aktuellen Daten des Datensatzes zurück. Es erfolgt eine implizite
	 * Empfängeranmeldung für das angegebene Intervall in Millisekunden.
	 *
	 * @param asp                     der betroffene Aspekt.
	 * @param unsubscriptionIntervall das Intervall für die eine implizite
	 *                                Empfängeranmeldung gültig sein soll.
	 * @return ein Datum, welches die Daten des Datensatzes kapselt.
	 *
	 */
	T getDatum(Aspekt asp, long unsubscriptionIntervall);

}
