/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell;

/**
 * Schnittstelle für Parameterdatensätze.
 *
 * <p>
 * Für Parameterdaten sind für Applikationen in der Regel nur die Aspekte
 * ParameterSoll zum Empfangen (einfacher Empfänger, keine Senke) und
 * ParameterVorgabe zum Senden (einfacher Sender, keine Quelle) relevant. Der
 * entsprechende Zugriff auf die Daten dieser beiden Aspekte wird durch
 * zusätzliche Methoden vereinfacht. Für die Verwendung des Aspekts ParameterIst
 * bzw. für die Anmeldung als Quelle oder Senke können die Methoden von
 * {@link OnlineDatensatz} verwendet werden.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @param <T> Der Typ des Datums den der Datensatz sichert.
 */
public interface ParameterDatensatz<T extends OnlineDatum> extends OnlineDatensatz<T> {

	/**
	 * Meldet den Parameter zum Senden an.
	 *
	 * @throws AnmeldeException wenn die Anmeldung nicht erfolgreich war.
	 */
	void anmeldenSender() throws AnmeldeException;

	/**
	 * Meldet eine eventuell vorhandene Anmeldung als Sender oder Quelle wieder ab.
	 */
	void abmeldenSender();

	/**
	 * Registriert einen Listener.
	 *
	 * @param l ein interessierte Listener.
	 */
	void addUpdateListener(DatensatzUpdateListener l);

	/**
	 * Deregistriert einen Listener.
	 *
	 * @param l ein nicht mehr interessierten Listener.
	 */
	void removeUpdateListener(DatensatzUpdateListener l);

	/**
	 * Fragt, ob der Datensatz als Sender oder Quelle Daten senden darf.
	 *
	 * @return {@code true}, wenn der Datensatz als Sender oder Quelle Daten senden
	 *         darf.
	 */
	Status getStatusSendesteuerung();

	/**
	 * Fragt, ob der Datensatz als Sender oder Quelle angemeldet ist.
	 *
	 * @return {@code true}, wenn der Datensatz als Sender oder Quelle angemeldet
	 *         ist.
	 */
	boolean isAngemeldetSender();

	/**
	 * Liest das Flag {@code autoUpdate}.
	 *
	 * @return {@code true}, wenn der Datensatz neue Daten automatisch vom
	 *         Datenverteiler empf&auml;ngt.
	 */
	boolean isAutoUpdate();

	/**
	 * Ruft das Datum unter dem Standardaspekt <code>asp.parameterSoll</code> ab.
	 *
	 * @return das aktuelle Datum unter dem Standardaspekt.
	 */
	T getDatum();

	/**
	 * Veranlasst den Datensatz ein Datum an den Datenverteiler zusenden. Ist der
	 * Zeitstempel des Datums nicht gesetzt oder gleich 0, wird automatisch der
	 * aktuelle Zeitstempel beim Versand verwendet.
	 *
	 * @param datum das zu sendende Datum.
	 * @throws DatensendeException wenn die Daten nicht gesendet werden konnten. Der
	 *                             Sendecache wird in dem Fall nicht geleert.
	 * @see #createDatum()
	 */
	void sendeDatum(T datum) throws DatensendeException;

	/**
	 * Veranlasst den Datensatz ein Datum an den Datenverteiler zusenden. Ist der
	 * Zeitstempel des Datums nicht gesetzt oder gleich 0, wird automatisch der
	 * aktuelle Zeitstempel beim Versand verwendet.
	 *
	 * @param datum   das zu sendende Datum.
	 * @param timeout die Zeit in der der Datensatz gesendet werden muss.
	 * @throws DatensendeException wenn die Daten nicht gesendet werden konnten. Der
	 *                             Sendecache wird in dem Fall nicht geleert.
	 * @see #createDatum()
	 */
	void sendeDatum(T datum, long timeout) throws DatensendeException;

}
