/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.att;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.List;
import java.util.RandomAccess;

/**
 * Kapselt ein Feld eines beliebiges Attributs eines Datensatzes.
 *
 * <p>
 * <em>Hinweis:</em> Abgeleitete Klassen müssen einen Standardkonstruktor
 * besitzen, damit {@link #clone()} funktioniert.
 *
 * @author BitCtrl Systems GmbH, Falko Schuman
 * @param <E> Der Typ der Feldelemente.
 */
public class Feld<E> extends AbstractList<E> implements RandomAccess, Cloneable, Serializable {

	private static final long serialVersionUID = 1L;

	private final List<E> liste = new ArrayList<>();
	private final int maxSize;
	private final boolean dynamicSize;

	/**
	 * Initialisiert das Feld.
	 *
	 * @param maxSize     die maximale Größe des Feldes oder 0 für unbegrenzt.
	 * @param dynamicSize Flag, ob die Länge des Feldes änderbar ist.
	 */
	public Feld(final int maxSize, final boolean dynamicSize) {
		this.maxSize = maxSize;
		this.dynamicSize = dynamicSize;
		if (!dynamicSize) {
			// default - Initialisierung des statischen Feldes
			for (int i = 0; i < maxSize; i++) {
				liste.add(null);
			}
		}
	}

	/**
	 * Initialisisert ein nicht dynamisches Feld aus dem Inhalt des übergebenen
	 * Feldes.
	 *
	 * @param feld das Feld aus dem die neue Instanz initialisiert werden soll
	 */
	public Feld(final List<E> feld) {
		this.maxSize = feld.size();
		this.dynamicSize = false;
		liste.addAll(feld);
	}

	@Override
	public E get(final int index) {
		return liste.get(index);
	}

	@Override
	public int size() {
		return liste.size();
	}

	@Override
	public E set(final int index, final E element) {
		return liste.set(index, element);
	}

	@Override
	public void add(final int index, final E element) {
		if (!isDynamicSize()) {
			throw new UnsupportedOperationException(
					"Die Länge der Liste " + getClass() + " kann nicht verändert werden.");
		}
		if ((getMaxSize() != 0) && (size() == getMaxSize())) {
			throw new IllegalStateException("Die Länge der Liste " + getClass() + " hat ihre maximale Länge von "
					+ getMaxSize() + " Elementen erreicht.");
		}
		liste.add(index, element);
	}

	@Override
	public E remove(final int index) {
		if (!isDynamicSize()) {
			throw new UnsupportedOperationException(
					"Die Länge der Liste " + getClass() + " kann nicht verändert werden.");
		}
		return liste.remove(index);
	}

	/**
	 * Gibt die maximale Größe des Feldes zurück.
	 *
	 * @return die Maximalgröße oder 0 für unendlich.
	 */
	public int getMaxSize() {
		return maxSize;
	}

	/**
	 * Flag, ob die Größe des Feldes geändert werden kann. Ist das Flag
	 * {@code false}, so hat das Feld genau {@link #getMaxSize()} Elemente.
	 *
	 * @return {@code true}, wenn die Elementanzahl variabel ist.
	 */
	public boolean isDynamicSize() {
		return dynamicSize;
	}

	@Override
	public Feld<E> clone() {
		try {
			final Feld<E> clone;
			if (isDynamicSize()) {
				clone = getClass().getConstructor(int.class, boolean.class).newInstance(getMaxSize(), true);
				for (final E e : this) {
					if (e instanceof Cloneable) {
						clone.add((E) e.getClass().getMethod("clone").invoke(e));
					} else {
						clone.add(e);
					}
				}
			} else {
				clone = getClass().getConstructor(List.class).newInstance(liste);
			}

			return clone;
		} catch (final IllegalArgumentException | SecurityException | IllegalAccessException | InvocationTargetException | NoSuchMethodException | InstantiationException ex) {
			throw new IllegalStateException(ex);
		}
	}

}
