/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.modell.att;

import java.text.NumberFormat;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import de.bsvrz.sys.funclib.bitctrl.modell.annotation.Wertebereich;

/**
 * Definiert den Wertebereich eines Ganzzahlattributs im Datenkatalog.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 */
public class WerteBereich {

	private static final WerteBereich NULL_BEREICH = new WerteBereich(0, 0, 0, null);
	private static final Map<Class<? extends Zahl>, WerteBereich> CACHE = new ConcurrentHashMap<>();

	/**
	 * Bestimmt den Wertebereich der Zahl.
	 *
	 * @param zahl eine Zahlendefinition.
	 * @return der Wertebereich der Zahl oder <code>null</code>, wenn kein
	 *         Wertebereich definiert ist.
	 */
	public static WerteBereich getWerteBereich(final Class<? extends Zahl> zahl) {

		final WerteBereich result = getWerteBereichInternal(zahl);
		if (result == NULL_BEREICH) {
			return null;
		}
		return result;
	}

	private static WerteBereich getWerteBereichInternal(final Class<? extends Zahl> zahl) {

		if (CACHE.containsKey(zahl)) {
			return CACHE.get(zahl);
		}

		WerteBereich result = NULL_BEREICH;
		final Wertebereich annotation = zahl.getAnnotation(Wertebereich.class);
		if (annotation != null) {
			result = new WerteBereich(annotation.minimum(), annotation.maximum(), annotation.skalierung(),
					annotation.einheit());
		}

		CACHE.put(zahl, result);
		return result;
	}

	private final long minimum;
	private final long maximum;
	private final double skalierung;
	private final String einheit;

	private WerteBereich(final long minimum, final long maximum, final double skalierung, final String einheit) {
		this.minimum = minimum;
		this.maximum = maximum;
		this.skalierung = skalierung;
		if ((einheit != null) && (einheit.length() > 0)) {
			this.einheit = einheit;
		} else {
			this.einheit = null;
		}
	}

	/**
	 * Gibt die kleinste Zahl zurück, die in diesem Wertebreich liegt. Der Wert ist
	 * bereits skaliert.
	 *
	 * @return das Minimum des Wertebereichs.
	 */
	public double getMinimum() {
		return minimum * skalierung;
	}

	/**
	 * Gibt die größte Zahl zurück, die in diesem Wertebreich liegt. Der Wert ist
	 * bereits skaliert.
	 *
	 * @return das Maximum des Wertebereichs.
	 */
	public double getMaximum() {
		return maximum * skalierung;
	}

	/**
	 * Gibt das unskalierte Minimum des Wertebereich zurück. Dieser Wert entspricht
	 * der Definition im Datenkatalog.
	 *
	 * @return das unskalierte Minimum.
	 */
	public long getUnskaliertesMinimum() {
		return minimum;
	}

	/**
	 * Gibt das unskalierte Maximum des Wertebereich zurück. Dieser Wert entspricht
	 * der Definition im Datenkatalog.
	 *
	 * @return das unskalierte Maximum.
	 */
	public long getUnskaliertesMaximum() {
		return maximum;
	}

	/**
	 * Gibt die Skalierung von Minimum und Maximum zurück, wie sie im Datenkatalog
	 * definiert ist.
	 *
	 * @return die Skalierung des Wertebereichs.
	 */
	public double getSkalierung() {
		return skalierung;
	}

	/**
	 * Gibt die Einheit der Zahl zurück.
	 *
	 * @return die Einheit oder <code>null</code>, wenn keine existiert.
	 */
	public String getEinheit() {
		return einheit;
	}

	@Override
	public String toString() {
		StringBuilder s = new StringBuilder();

		s.append(getMinimum());
		s.append(" bis ");
		s.append(NumberFormat.getInstance().format(getMaximum()));
		if (einheit != null) {
			s.append(" ").append(einheit);
		}

		return s.toString();
	}

}
