/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.impl;

import java.util.NoSuchElementException;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.sys.funclib.bitctrl.modell.Aspekt;
import de.bsvrz.sys.funclib.bitctrl.modell.Datensatz;
import de.bsvrz.sys.funclib.bitctrl.modell.Datum;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;

/**
 * Implementiert gemeinsame Funktionen der Datens&auml;tze.
 *
 * @author BitCtrl Systems GmbH, Falko Schumann
 * @param <T> Der Typ des Datums den der Datensatz sichert.
 */
public abstract class AbstractDatensatz<T extends Datum> extends AbstractSystemObjektInternal implements Datensatz<T> {

	/** Das Systemobjekt. */
	private final SystemObjekt systemObjekt;

	/**
	 * Initialisiert den Datensatz.
	 *
	 * @param systemObjekt  das Systemobjekt zu dem der Datensatz gehört.
	 * @param objektFactory die ObjektFabrik, die der Datensatz verwenden soll.
	 */
	public AbstractDatensatz(final SystemObjekt systemObjekt, final ObjektFactory objektFactory) {
		this.systemObjekt = systemObjekt;

		init(objektFactory.getDav().getDataModel().getAttributeGroup(doGetPid()), objektFactory);
	}

	@Override
	protected String doGetTypPid() {
		return "typ.attributgruppe";
	}

	/**
	 * Gibt die PID der Attributgruppe des Datensatzes zurück.
	 *
	 * @return die PID der gekapselten Attributgruppe.
	 */
	protected abstract String doGetPid();

	@Override
	public SystemObjekt getSystemObjekt() {
		return systemObjekt;
	}

	/**
	 * Pr&uuml;ft, ob das {@code ResultData} zum Datensatz geh&ouml;rt. Es wird die
	 * Attributgruppe aus der Datenbeschreibung des {@code ResultData} mit der
	 * Attributgruppe des Datensatzes. Au&szlig;erdem wird gepr&uuml;ft, ob der
	 * Aspekt des {@code ResultData} bekannt ist.
	 * <p>
	 * Geh&ouml;hrt das {@code ResultData} nicht zum Datensatz wird eine
	 * {@link IllegalArgumentException} geworfen.
	 *
	 * @param result ein {@code ResultSet}.
	 */
	protected void check(final ResultData result) {
		if (!result.getDataDescription().getAttributeGroup().equals(getSystemObject())) {
			throw new IllegalArgumentException("Das Datum muss zur Attributgruppe " + getSystemObject()
					+ " gehören (gefunden: " + result.getDataDescription().getAttributeGroup() + ").");
		}

		boolean gefunden = false;
		for (final Aspekt asp : getAspekte()) {
			if (getAspect(asp).equals(result.getDataDescription().getAspect())) {
				gefunden = true;
				break;
			}
		}
		if (!gefunden) {
			throw new IllegalArgumentException("Unbekanner Aspekt im Datum gefunden (gefunden: "
					+ result.getDataDescription().getAspect() + ", bekannt: " + getAspekte() + ")");
		}
	}

	/**
	 * Gibt einen leeren Sendecache zurück.
	 *
	 * @return ein leeres {@code Data} für den Datensatz.
	 * @see #konvertiere(Datum)
	 */
	protected Data createSendeCache() {
		return getDav().createData(getSystemObject());
	}

	/**
	 * Identisch mit <code>getObjektFactory().getDav()</code>.
	 *
	 * @return die Datenverteilerverbindung.
	 */
	protected ClientDavInterface getDav() {
		return getObjektFactory().getDav();
	}

	@Override
	public AttributeGroup getSystemObject() {
		return (AttributeGroup) super.getSystemObject();
	}

	/**
	 * Gibt zum Modell-Aspekt den Datenverteiler-Aspect zurück.
	 *
	 * @param asp der Modell-Aspekt.
	 * @return der dazugehörige Datenverteiler-Aspect.
	 */
	protected Aspect getAspect(final Aspekt asp) {
		return asp.getSystemObject();
	}

	/**
	 * Gibt zum Datenverteiler-Aspect den Modell-Aspekt zurück.
	 *
	 * @param asp der Datenverteiler-Aspect.
	 * @return der dazugehörige Modell-Aspekt.
	 */
	protected Aspekt getAspekt(final Aspect asp) {
		for (final Aspekt e : getAspekte()) {
			if (e.getPid().equals(asp.getPid())) {
				return e;
			}
		}
		throw new NoSuchElementException("Der Aspekt " + asp + " ist am Datensatz " + this + " nicht erlaubt.");
	}

}
