/*
 * Allgemeine Funktionen BitCtrl Modell
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.modell.impl;

import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.bitctrl.modell.ObjektFactory;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjekt;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjektBereich;
import de.bsvrz.sys.funclib.bitctrl.modell.SystemObjektTyp;

/**
 * Implementierung der gemeinsamen Methoden der Systemobjektschnittstelle.
 *
 * @author BitCtrl Systems GmbH, Schumann
 */
abstract class AbstractSystemObjektInternal implements SystemObjekt, Comparable<SystemObjekt> {

	private SystemObject systemObject;
	private ObjektFactory objektFactory;

	/**
	 * Der Defaultkonstruktor tut nichts.
	 *
	 * <p>
	 * <em>Hinweis:</em> Wenn der Defaultkonstruktor verwendet wird, ist das Objekt
	 * nicht initialisiert.
	 */
	protected AbstractSystemObjektInternal() {
		// tut nix
	}

	/**
	 * Initialisiert das Systemobjekt.
	 *
	 * @param systemObject  das zu kapselnde Systemobjekt.
	 * @param objektFactory die Factory, die das Objekt erzeugt hat.
	 */
	protected AbstractSystemObjektInternal(final SystemObject systemObject, final ObjektFactory objektFactory) {
		init(systemObject, objektFactory);
	}

	/**
	 * Initialisiert das Systemobjekt.
	 *
	 * <p>
	 * <em>Hinweis:</em> Diese Methode darf nur genau einmal aufgerufen werden. Muss
	 * aber aufgerufen werden, bevor das Systemobjekt verwendet wird.
	 *
	 * @param systemObject  das zu kapselnde Systemobjekt.
	 * @param objektFactory die Factory, die das Objekt erzeugt hat.
	 */
	@SuppressWarnings("hiding")
	public void init(final SystemObject systemObject, final ObjektFactory objektFactory) {
		this.systemObject = systemObject;
		this.objektFactory = objektFactory;

		if ((systemObject != null) && !systemObject.isOfType(doGetTypPid())) {
			throw new IllegalArgumentException("Systemobjekt ist kein " + getClass() + ": " + systemObject);
		}
	}

	public ObjektFactory getObjektFactory() {
		return objektFactory;
	}

	/**
	 * übernimmt die Methode von {@link SystemObject}, wenn dieses und das andere
	 * Systemobjekt initialisiert sind. Andernfalls werden die PIDs der beiden
	 * Objekt verglichen.
	 */
	@Override
	public boolean equals(final Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj instanceof SystemObjekt) {
			final SystemObjekt other = (SystemObjekt) obj;
			if ((getSystemObject() != null) && (other.getSystemObject() != null)) {
				return systemObject.equals(other.getSystemObject());
			}
			return getPid().equals(other.getPid());
		}
		return false;
	}

	/**
	 * Verwendet den Hash der PID.
	 */
	@Override
	public int hashCode() {
		return getPid().hashCode();
	}

	@Override
	public long getId() {
		return systemObject.getId();
	}

	@Override
	public String getName() {
		return systemObject.getNameOrPidOrId();
	}

	@Override
	public String getPid() {
		return systemObject.getPid();
	}

	/**
	 * Gibt die PID des Systemobjekttyps dieses Systemobjekts zurück.
	 *
	 * @return die PID des Typs des Systemsobjekts.
	 */
	protected abstract String doGetTypPid();

	@Override
	public SystemObjektTyp getTyp() {
		if (systemObject != null) {
			return (SystemObjektTyp) getObjektFactory().getModellobjekt(systemObject.getType());
		}

		return (SystemObjektTyp) getObjektFactory().getModellobjekt(doGetTypPid());
	}

	@Override
	public SystemObjektBereich getKonfigurationsBereich() {
		if (systemObject != null) {
			return (SystemObjektBereich) getObjektFactory().getModellobjekt(systemObject.getConfigurationArea());
		}

		return null;
	}

	@Override
	public SystemObject getSystemObject() {
		return systemObject;
	}

	/**
	 * übernimmt die Methode von {@link SystemObject}.
	 */
	@Override
	public String toString() {
		return systemObject.toString();
	}

	@Override
	@SuppressWarnings("unchecked")
	public int compareTo(final SystemObjekt o) {
		return getSystemObject().compareTo(o.getSystemObject());
	}

}
