/*
 * Allgemeine Funktionen mit und ohne Datenverteilerbezug
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.interpreter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

/**
 * Ein Nicht-Terminal-Symbol des Interpreters, sprich ein Operatorsymbol bzw
 * eine Operation.
 *
 * @author BitCtrl Systems GmbH, Schumann
 */
public class Operation<T, O> implements Ausdruck<T> {

	/**
	 * Der Operator dieser Operation.
	 */
	private final Operator operator;

	/**
	 * Die Liste der Operanden.
	 */
	private final List<Ausdruck<O>> operanden;

	/**
	 * Konstruiert ein Symbol mit den gegebenen Operator und Operanden.
	 *
	 * @param operator  Operator
	 * @param operanden Liste mit Operanden
	 * @throws NullPointerException Wenn ein Funktionsparameter {@code null} ist
	 */
	public Operation(final Operator operator, final Ausdruck<O>... operanden) {
		this(operator, Arrays.asList(operanden));
	}

	/**
	 * Konstruiert ein Symbol mit den gegebenen Operator und Operanden.
	 *
	 * @param operator  Operator
	 * @param operanden Liste mit Operanden
	 * @throws NullPointerException Wenn ein Funktionsparameter {@code null} ist
	 */
	public Operation(final Operator operator, final List<? extends Ausdruck<O>> operanden) {
		this.operator = Objects.requireNonNull(operator, "operator");
		this.operanden = new ArrayList<>(Objects.requireNonNull(operanden, "operanden"));
	}

	/**
	 * Konstruiert ein Symbol mit den gegebenen Operatorsymbol und Operanden.
	 *
	 * @param operatorSymbol Operator
	 * @param operanden      Liste mit Operanden
	 * @throws NullPointerException Wenn ein Funktionsparameter {@code null} ist
	 */
	public Operation(final String operatorSymbol, final Ausdruck<O>... operanden) {
		this(Operator.getOperator(operatorSymbol), operanden);
	}

	/**
	 * Konstruiert ein Symbol mit den gegebenen Operatorsymbol und Operanden.
	 *
	 * @param operatorSymbol Operator
	 * @param operanden      Liste mit Operanden
	 * @throws NullPointerException Wenn ein Funktionsparameter {@code null} ist
	 */
	public Operation(final String operatorSymbol, final List<? extends Ausdruck<O>> operanden) {
		this(Operator.getOperator(operatorSymbol), operanden);
	}

	@Override
	public List<? extends Ausdruck<O>> getNachfolger() {
		return Collections.unmodifiableList(operanden);
	}

	/**
	 * Gibt die Operanden der Operation zur&uuml;ck.
	 *
	 * @return Die Liste der Operanden
	 */
	public List<? extends Ausdruck<O>> getOperanden() {
		return operanden;
	}

	/**
	 * Gibt den Operator dieser Operation zur&uuml;ck.
	 *
	 * @return Operator
	 */
	public Operator getOperator() {
		assert operator != null;

		return operator;
	}

	@Override
	public T interpret(final Kontext kontext) {
		assert operator != null;

		final List<O> values = new ArrayList<>();
		for (final Ausdruck<O> a : operanden) {
			if (a != null) {
				values.add(a.interpret(kontext));
			} else {
				values.add(null);
			}
		}

		return (T) operator.execute(values);
	}

	@Override
	public String toString() {
		assert operator != null;

		StringBuilder result = new StringBuilder("(");

		if (operanden.size() == 1) {
			// Es gibt nur einen Operanden
			result.append(operator).append(" ").append(operanden.get(0));
		} else {
			// Es gibt mehr als einen Operanden
			final Iterator<Ausdruck<O>> i = operanden.iterator();
			Ausdruck a = i.next();

			result.append(a.toString()); // erster Operand
			while (i.hasNext()) {
				a = i.next();
				result.append(" ").append(operator).append(" ").append(a); // restliche Operanden
			}
		}
		result.append(")");

		return result.toString();
	}

}
