/*
 * Allgemeine Funktionen mit und ohne Datenverteilerbezug
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.interpreter.logik;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.bitctrl.i18n.Messages;

import de.bsvrz.sys.funclib.bitctrl.interpreter.Handler;
import de.bsvrz.sys.funclib.bitctrl.interpreter.HandlerValidation;
import de.bsvrz.sys.funclib.bitctrl.interpreter.InterpreterException;
import de.bsvrz.sys.funclib.bitctrl.interpreter.InterpreterMessages;
import de.bsvrz.sys.funclib.bitctrl.interpreter.Operator;

/**
 * Handler f&uuml;r (fuzzy-)logische Ausdr&uuml;cke. Abgebildet sind die
 * Basisoperatoren, alle anderen lassen auf diese zur&uuml;ckf&uuml;hren.
 *
 * @author BitCtrl Systems GmbH, Schumann
 */
public class LogikHandler extends Handler {

	/** Logisches "und" */
	public static final Operator UND = Operator.getOperator("und");

	/** Logisches "oder" */
	public static final Operator ODER = Operator.getOperator("oder");

	/** Logisches "nicht" */
	public static final Operator NICHT = Operator.getOperator("nicht");

	/** Logische Implikation. */
	public static final Operator IMPLIKATION = Operator.getOperator("->");

	/**
	 * Liste der vom Handler unterst&uuml;tzten Operatoren.
	 */
	private static Operator[] operatoren = { UND, ODER, NICHT, IMPLIKATION };

	@Override
	public Operator[] getHandledOperators() {
		return operatoren;
	}

	@Override
	public Object perform(Operator operator, List<?> operanden) {
		if ((operator == null) || !validiereHandler(operator, operanden).isValid()) {
			throw new InterpreterException(Messages.get(InterpreterMessages.HandlerNotFound));
		}

		if (operator == UND) {
			return und(operanden.toArray(new LogischerWert[0]));
		} else if (operator == ODER) {
			return oder(operanden.toArray(new LogischerWert[0]));
		} else if (operator == NICHT) {
			return nicht((LogischerWert) operanden.get(0));
		} else if (operator == IMPLIKATION) {
			return implikation((LogischerWert) operanden.get(0), (LogischerWert) operanden.get(1));
		}

		throw new IllegalStateException("[unreachable code] unbekannter Operator: " + operator);
	}

	@Override
	public HandlerValidation validiereHandler(Operator operator, List<?> operanden) {
		final boolean anzahlOk = validiereAnzahl(operator, operanden);
		final boolean typOk = validiereTyp(operanden);
		return new HandlerValidation(anzahlOk, typOk);
	}

	private static boolean validiereAnzahl(Operator operator, List<?> operanden) {
		if (UND.equals(operator) || ODER.equals(operator)) {
			return operanden.size() >= 2;
		} else if (NICHT.equals(operator)) {
			return operanden.size() == 1;
		} else if (IMPLIKATION.equals(operator)) {
			return operanden.size() == 2;
		} else {
			return false;
		}
	}

	private static boolean validiereTyp(List<?> operanden) {
		return operanden.stream().map(o -> o instanceof LogischerWert).reduce(Boolean::logicalAnd).orElse(false);
	}

	protected LogischerWert und(LogischerWert... operanden) {
		return und(Arrays.asList(operanden));
	}

	protected LogischerWert und(Collection<LogischerWert> operanden) {
		return operanden.stream().map(LogischerWert::get).reduce(Boolean::logicalAnd).map(LogischerWert::of).get();
	}

	protected LogischerWert oder(LogischerWert... operanden) {
		return oder(Arrays.asList(operanden));
	}

	protected LogischerWert oder(Collection<LogischerWert> operanden) {
		return operanden.stream().map(LogischerWert::get).reduce(Boolean::logicalOr).map(LogischerWert::of).get();
	}

	protected LogischerWert nicht(LogischerWert a) {
		return LogischerWert.of(!a.get());
	}

	protected LogischerWert implikation(LogischerWert a, LogischerWert b) {
		return oder(nicht(a), b);
	}

}
