/*
 * Allgemeine Funktionen mit und ohne Datenverteilerbezug
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
/* Copyright by BitCtrl Systems Leipzig */
/* BitCtrl Systems Leipzig */
/* Weisenfelser Str. 67 */
/* 04229 Leipzig */
/* Tel.: +49 341 49067 - 0 */
/* Fax.: +49 341 49067 - 15 */
/* mailto:info@bitctrl.de */
/* http://www.bitctrl.de */
/*---------------------------------------------------------------*/

package de.bsvrz.sys.funclib.bitctrl.testdaten;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Random;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;

/**
 * Test-Applikation zur Erzeugung einigermaßen realistischer Langzeitdaten.
 */
public class LzdTestSender implements StandardApplication, ClientSenderInterface {

	private final Debug LOGGER = Debug.getLogger();

	/**
	 * Liste aller Fahrstreifen mit ihrem aktuellen Verbindungszustand
	 */
	private final Collection<SystemObject> fahrStreifenListe = new ArrayList<>();

	private DataDescription descLangzeitDaten;

	private final Random dataSource = new Random();

	private LocalDateTime startZeit;

	private ClientDavInterface dav;

	private long intervall;

	private LzdTestSender() {
		super();

	}

	public void generateLZD() {

		while (startZeit.isBefore(LocalDateTime.now())) {
			LOGGER.info("Erzeuge Daten fuer " + startZeit);
			for (final SystemObject fahrStreifen : fahrStreifenListe) {
				Data data;
				try {

					data = getVerkehrsDaten();
					dav.sendData(new ResultData(fahrStreifen, descLangzeitDaten,
							startZeit.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli(), data));
				} catch (final Exception e) {
					LOGGER.error("Versenden der Daten für FS " + fahrStreifen + " fehlgeschlagen.", e);
				}
			}
			startZeit = startZeit.plusHours(1);
			try {
				Thread.sleep(intervall);
			} catch (final InterruptedException e) {
				LOGGER.info("Erzeugung der LZD Testdaten unterbrochen.", e);
				return;
			}
		}
		LOGGER.info("Erzeugung von Langzeitdaten erfolgreich abgeschlossen :)");
	}

	@Override
	public void initialize(ClientDavInterface dav) {
		this.dav = dav;

		final AttributeGroup atg = dav.getDataModel().getAttributeGroup("atg.verkehrsDatenLangZeitIntervall");
		final Aspect asp = dav.getDataModel().getAspect("asp.externeErfassung");
		descLangzeitDaten = new DataDescription(atg, asp);

		final SystemObjectType type = dav.getDataModel().getType("typ.fahrStreifen");
		fahrStreifenListe.addAll(type.getElements());

		try {
			dav.subscribeSender(this, fahrStreifenListe.toArray(new SystemObject[0]), descLangzeitDaten,
					SenderRole.source());
		} catch (final OneSubscriptionPerSendData e) {
			LOGGER.error("Anmeldung als Sender für LZD für Fahrstreifen fehlgeschlagen.", e);
		}

		Executors.newSingleThreadScheduledExecutor().scheduleWithFixedDelay(this::generateLZD, 0, 1, TimeUnit.HOURS);
	}

	@Override
	public void parseArguments(ArgumentList argumentList) throws Exception {
		final String startStr = argumentList.fetchArgument("-startDatum=").asString();
		if ((startStr != null) && (startStr.length() > 0)) {
			startZeit = LocalDate.parse(startStr).atStartOfDay();
		} else {
			startZeit = LocalDate.now().minusDays(1).atStartOfDay();
		}

		intervall = argumentList.fetchArgument("-intervall=30Sekunden").asRelativeTime();
	}

	/**
	 * Die Hauptfunktion des Observers.
	 * 
	 * @param args Programm-Argumente
	 */
	public static void main(final String[] args) {
		StandardApplicationRunner.run(new LzdTestSender(), args);
	}

	@Override
	public boolean isRequestSupported(SystemObject object, DataDescription dataDescription) {
		return false;
	}

	/**
	 * Erzeugt einen zufälligen LZD Datensatz.
	 * 
	 * @return
	 * @throws Exception
	 */
	private Data getVerkehrsDaten() throws Exception {

		final int qKfzValue = dataSource.nextInt(100);
		final int qLkwValue = dataSource.nextInt(100);
		final int vPkwValue = dataSource.nextInt(100);
		final int vLkwValue = dataSource.nextInt(100);

		final Data data = dav.createData(descLangzeitDaten.getAttributeGroup());

		// Intervalllänge aus IntervalldatenHeader
		data.getTimeValue("T").setSeconds(3600);

		final String[] valStrings = { "qKfz", "qPkwÄ", "qKfzNk", "qPkwG", "qKrad", "qLfw", "qLkwÄ", "qPkwA", "qBus",
				"qLkwK", "qLkwA", "qSattelKfz", "vPkwÄ", "vKfzNk", "vPkwG", "vKrad", "vLfw", "vLkwÄ", "vPkwA", "vBus",
				"vLkwK", "vLkwA", "vSattelKfz", "vKfz", "qLkw", "vLkw", "qPkw", "vPkw", "sKfz", "sPkwÄ", "sKfzNk",
				"sPkwG", "sPkw", "sKrad", "sLkw", "sLfw", "sLkwÄ", "sPkwA", "sBus", "sLkwK", "sLkwA", "sSattelKfz",
				"v85Kfz", "v85PkwÄ", "v85KfzNk", "v85PkwG", "v85Pkw", "v85Krad", "v85Lfw", "v85LkwÄ", "v85PkwA",
				"v85Lkw", "v85Bus", "v85LkwK", "v85LkwA", "v85SattelKfz" };

		for (int idx = 0; idx < valStrings.length; idx++) {
			data.getItem(valStrings[idx])// .getUnscaledValue("Wert").set(5);
					.getUnscaledValue("Wert").setText("nicht ermittelbar");
			data.getItem(valStrings[idx]).getItem("Status").getItem("Erfassung").getUnscaledValue("NichtErfasst")
					.setText("Nein");
			data.getItem(valStrings[idx]).getItem("Status").getItem("PlFormal").getUnscaledValue("WertMax")
					.setText("Nein");
			data.getItem(valStrings[idx]).getItem("Status").getItem("PlFormal").getUnscaledValue("WertMin")
					.setText("Nein");
			data.getItem(valStrings[idx]).getItem("Status").getItem("PlLogisch").getUnscaledValue("WertMaxLogisch")
					.setText("Nein");
			data.getItem(valStrings[idx]).getItem("Status").getItem("PlLogisch").getUnscaledValue("WertMinLogisch")
					.setText("Nein");
			data.getItem(valStrings[idx]).getItem("Status").getItem("MessWertErsetzung").getUnscaledValue("Implausibel")
					.setText("Nein");
			data.getItem(valStrings[idx]).getItem("Status").getItem("MessWertErsetzung")
					.getUnscaledValue("Interpoliert").setText("Nein");
			data.getItem(valStrings[idx]).getItem("Güte").getUnscaledValue("Index").set(-1);
			data.getItem(valStrings[idx]).getItem("Güte").getUnscaledValue("Verfahren").set(0);
		}

		data.getItem("qKfz").getUnscaledValue("Wert").set(qKfzValue);
		data.getItem("qKfz").getItem("Güte").getUnscaledValue("Index").set(10);

		data.getItem("qLkw").getUnscaledValue("Wert").set(qLkwValue);
		data.getItem("qLkw").getItem("Güte").getUnscaledValue("Index").set(10);

		data.getItem("vPkw").getUnscaledValue("Wert").set(vPkwValue);
		data.getItem("vPkw").getItem("Güte").getUnscaledValue("Index").set(10);

		data.getItem("vLkw").getUnscaledValue("Wert").set(vLkwValue);
		data.getItem("vLkw").getItem("Güte").getUnscaledValue("Index").set(10);

		// Nicht erfasste Werte qPkw und vKfz berechnen

		final int qPkwValue = (qKfzValue - qLkwValue) >= 0 ? (qKfzValue - qLkwValue) : -1;
		data.getItem("qPkw").getUnscaledValue("Wert").set(qPkwValue);
		data.getItem("qPkw").getItem("Güte").getUnscaledValue("Index").set(10);
		data.getItem("qPkw").getItem("Status").getItem("Erfassung").getUnscaledValue("NichtErfasst").setText("Ja");

		final int vKfzValue = (qLkwValue + qPkwValue) > 0
				? ((qLkwValue * vLkwValue) + (qPkwValue * vPkwValue)) / (qLkwValue + qPkwValue)
				: -1;

		data.getItem("vKfz").getUnscaledValue("Wert").set(Math.abs(vKfzValue));
		data.getItem("vKfz").getItem("Güte").getUnscaledValue("Index").set(10);
		data.getItem("vKfz").getItem("Status").getItem("Erfassung").getUnscaledValue("NichtErfasst").setText("Ja");

		return data;

	}

	@Override
	public void dataRequest(SystemObject object, DataDescription dataDescription, byte state) {
		// nüscht
	}
}
