/*
 * Allgemeine Funktionen mit und ohne Datenverteilerbezug
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
package de.bsvrz.sys.funclib.bitctrl.datenpunkt;

import java.util.Arrays;
import java.util.NoSuchElementException;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage.Usage;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.dav.daf.main.impl.InvalidArgumentException;

/**
 * Eine Repräsentation einer Attributliste vom Format der
 * <code>atl.datenpunkt</code> mitsamt den Live-Daten vom Datenverteiler, d.h.
 * das Objekt ist angemeldet!
 *
 * @author BitCtrl Systems GmbH, Albrecht Uhlmann
 */
public class Datenpunkt implements ClientReceiverInterface {

	private Data atlDatenpunkt;

	private Data lastValue;

	private ResultData lastResult;

	private ClientDavInterface connection;

	private SystemObject object;

	private DataDescription dataDescription;

	private String[] pfadKomponenten;

	/**
	 * Abmeldung vom Datenverteiler. Falls schon abgemeldet, passiert nichts.
	 */
	protected void abmelden() {
		if (null == atlDatenpunkt) {
			return;
		}
		connection.unsubscribeReceiver(this, object, dataDescription);
		atlDatenpunkt = null;
		dataDescription = null;
		object = null;
		pfadKomponenten = null;
	}

	private void anmelden() throws InvalidArgumentException {
		final SystemObjectType type = (SystemObjectType) atlDatenpunkt.getReferenceValue("Typ").getSystemObject();
		final SystemObject o = atlDatenpunkt.getReferenceValue("Objekt").getSystemObject();
		if (!o.isOfType(type.getPid())) {
			throw new InvalidArgumentException("Objekt '" + o + "' ist nicht vom Typ '" + type + "'");
		}
		final AttributeGroup atg = (AttributeGroup) atlDatenpunkt.getReferenceValue("Attributgruppe").getSystemObject();
		if (!type.getAttributeGroups().contains(atg)) {
			throw new InvalidArgumentException(
					"Attributgruppe '" + atg + "' ist am Typ '" + type + "' nicht definiert");
		}
		final Aspect asp = (Aspect) atlDatenpunkt.getReferenceValue("Aspekt").getSystemObject();
		if (!atg.getAspects().contains(asp)) {
			throw new InvalidArgumentException(
					"Aspekt '" + asp + "' ist an Attributgruppe '" + atg + "' nicht definiert");
		}
		final String pfad = atlDatenpunkt.getTextValue("Pfad").getText();
		if ((null == pfad) || pfad.isEmpty()) {
			throw new InvalidArgumentException("Leerer Pfad ist nicht erlaubt");
		}
		pfadKomponenten = pfad.split("\\.");
		if (null == atg.getAttribute(pfadKomponenten[0])) {
			throw new InvalidArgumentException("Das Attribut '" + pfadKomponenten[0] + "' ist an der Attributgruppe '"
					+ atg + "' nicht definiert");
		}
		dataDescription = new DataDescription(atg, asp);
		object = o;
		final ReceiverRole role;
		final Usage usage = atg.getAttributeGroupUsage(asp).getUsage();
		if (AttributeGroupUsage.Usage.OnlineDataAsSenderDrain.equals(usage)) {
			role = ReceiverRole.drain();
		} else if (AttributeGroupUsage.Usage.OnlineDataAsSourceReceiver.equals(usage)
				|| AttributeGroupUsage.Usage.OnlineDataAsSourceReceiverOrSenderDrain.equals(usage)) {
			role = ReceiverRole.receiver();
		} else {
			throw new InvalidArgumentException("Unzulässige Attributgruppenverwendung: " + usage);
		}
		connection.subscribeReceiver(this, o, dataDescription, ReceiveOptions.delayed(), role);
	}

	/**
	 * Liefert die Beschreibung des aktuellen Datenpunktes, also ein Data-Objekt vom
	 * Format der <code>atl.datenpunkt</code>.
	 *
	 * @return der Datenpunkt
	 */
	public Data getAtlDatenpunkt() {
		return atlDatenpunkt;
	}

	/**
	 * Setzt den Datenpunkt.
	 *
	 * @param atlDatenpunkt der Datenpunkt
	 * @throws InvalidArgumentException falls das übergebene Objekt inkonsistent
	 *                                  ist, z.B. das referenzierte Objekt nicht vom
	 *                                  angegebenen Typ ist oder der referenzierte
	 *                                  Typ die referenzierte Attributgruppe gar
	 *                                  nicht hat, usw.
	 */
	public void setAtlDatenpunkt(final Data atlDatenpunkt) throws InvalidArgumentException {
		abmelden();
		this.atlDatenpunkt = atlDatenpunkt;
		anmelden();
	}

	/**
	 * @return the lastValue
	 */
	public Data getLastValue() {
		return lastValue;
	}

	@Override
	public void update(final ResultData[] results) {
		if (null == atlDatenpunkt) {
			return;
		}
		for (final ResultData result : results) {
			update(result);
		}
	}

	/**
	 * Aktualisiert einen Ergebnisdatensatz aus dem Feld von Ergebnisdatensätzen,
	 * die über die DAF-API kamen.
	 *
	 * @param result der Ergebnisdatensatz
	 */
	protected void update(final ResultData result) {
		if (result.getDataDescription().equals(dataDescription)) {
			lastResult = result;
			if (!result.hasData()) {
				lastValue = null;
			} else {
				try {
					Data currentData = result.getData();
					for (final String pfad : pfadKomponenten) {
						currentData = currentData.getItem(pfad);
					}
					lastValue = currentData;
				} catch (@SuppressWarnings("unused") final NoSuchElementException e) {
					lastValue = null;
				}
			}
		}
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((dataDescription == null) ? 0 : dataDescription.hashCode());
		result = prime * result + Arrays.hashCode(pfadKomponenten);
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Datenpunkt other = (Datenpunkt) obj;
		if (dataDescription == null) {
			if (other.dataDescription != null)
				return false;
		} else if (!dataDescription.equals(other.dataDescription))
			return false;
		if (!Arrays.equals(pfadKomponenten, other.pfadKomponenten))
			return false;
		return true;
	}

	

	/**
	 * Liefert die dataDescription.
	 *
	 * @return die dataDescription, kann <code>null</code> sein.
	 */
	public DataDescription getDataDescription() {
		return dataDescription;
	}

	/**
	 * Liefert die Pfadkomponenten.
	 *
	 * @return die Pfadkomponenten, kann null sein.
	 */
	public String[] getPfadKomponenten() {
		return pfadKomponenten;
	}

	/**
	 * Setzt die Datenverteilerverbindung.
	 *
	 * @param connection die Datenverteilerverbindung
	 */
	public void setConnection(final ClientDavInterface connection) {
		this.connection = connection;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		if ((dataDescription == null) || (object == null) || (pfadKomponenten == null)) {
			return "Keine Daten";
		}

		StringBuilder result = new StringBuilder(object.getPid()).append("-").append(dataDescription.toString())
				.append("-").append(pfadKomponenten[0]);
		int loop;
		for (loop = 1; loop < pfadKomponenten.length; ++loop) {
			result.append(".").append(pfadKomponenten[loop]);
		}
		result.append(":");
		result.append(getLastValue());
		return result.toString();
	}

	/**
	 * Liefert das Systemobjekt, dem dieser Datenpunkt zugeordnet ist.
	 *
	 * @return das Objekt oder null, wenn noch keine Anmeldung erfolgt ist.
	 */
	public SystemObject getObject() {
		return object;
	}

	/**
	 * Liefert den letzten empfangenen Ergebnisdatensatz.
	 *
	 * @return der Ergebnisdatensatz. Kann <code>null</code> sein, wenn noch nie
	 *         einer empfangen wurde. Für den Fall, dass der Ergebnisdatensatz keine
	 *         Daten enthält, wird er trotzdem gespeichert, nur #lastValue wird
	 *         null.
	 */
	public ResultData getLastResult() {
		return lastResult;
	}
}
