/*
 * Allgemeine Funktionen mit und ohne Datenverteilerbezug
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */

package de.bsvrz.sys.funclib.bitctrl.interpreter;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Basisklasse aller Symbole des Interpeters.
 *
 * @author BitCtrl Systems GmbH, Uwe Peuker, Schumann
 */
public interface Ausdruck<T> {

	/**
	 * Bestimmt Rekursiv die Menge der Termsymbole im Ausdruck, die Variablen
	 * darstellen.
	 *
	 * @param ausdruck Ein beliebiger Ausdruck
	 * @return Menge der Variablennamen
	 */
	static Set<String> getVariablen(final Ausdruck<?> ausdruck) {
		final Set<String> variablen = new HashSet<>();
		if (ausdruck instanceof Variable) {
			variablen.add(((Variable) ausdruck).getName());
		} else {
			for (final Ausdruck<?> a : ausdruck.getNachfolger()) {
				variablen.addAll(getVariablen(a));
			}
		}
		return variablen;
	}

	/**
	 * Gibt eine Liste der verschachtelten Ausdr&uuml;cke zur&uuml;ck.
	 *
	 * @return Liste der Ausdr&uuml;cke <em>direkt</em> unter diesen Ausdruck. Die
	 *         Methode arbeiten im Gegensatz zu {@link #interpret(Kontext)} nicht
	 *         rekursiv.
	 */
	default List<? extends Ausdruck<?>> getNachfolger() {
		return Collections.emptyList();
	}

	/**
	 * Interpretiert den Ausdruck im gegebenen Kontext.
	 *
	 * @param kontext Kontext, indem der Ausdruck ausgewertet wird
	 * @return Wert des interpretierten Ausdrucks
	 */
	T interpret(Kontext kontext);

}
