/*
 * Allgemeine Funktionen mit und ohne Datenverteilerbezug
 * Copyright (C) 2007-2021 BitCtrl Systems GmbH 
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Contact Information:
 * BitCtrl Systems GmbH
 * Weissenfelser Strasse 67
 * 04229 Leipzig
 * Phone: +49 341-490670
 * mailto: info@bitctrl.de
 */
/* Copyright by BitCtrl Systems Leipzig */
/* BitCtrl Systems Leipzig */
/* Weisenfelser Str. 67 */
/* 04229 Leipzig */
/* Tel.: +49 341 49067 - 0 */
/* Fax.: +49 341 49067 - 15 */
/* mailto:info@bitctrl.de */
/* http://www.bitctrl.de */
/*---------------------------------------------------------------*/
package de.bsvrz.sys.funclib.bitctrl.testdaten;

import java.io.FileInputStream;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Logger;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataNotSubscribedException;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;

/**
 * Test-Applikation zur Erzeugung einigermaßen realistischer Kurzzeitdaten.
 */
public final class KzdTestSender extends TimerTask implements StandardApplication, ClientSenderInterface {

	private double vLkwMax = 100.0;

	private double qKfzMax = 80.0;

	private double vPkwMax = 140.0;

	private final Logger logger = Logger.getLogger(KzdTestSender.class.getName());

	/**
	 * Liste aller Fahrstreifen mit ihrem aktuellen Verbindungszustand.
	 */
	private final Collection<SystemObject> fahrStreifenListe = new ArrayList<>();

	private DataDescription descKurzzeitDaten;

	private Timer checkTimer = new Timer("KzdCheckTimer", true);

	private Random dataSource = new Random();

	private long startZeit;

	/**
	 * index.
	 */
	private int idxData;

	/**
	 *
	 */
	private final ArrayList<Integer[]> fileData = new ArrayList<>();

	/**
	 * true, wenn Testdaten aus einer Datei erzeugt werden sollen.
	 */
	private boolean useFileData;

	/**
	 * intervall, in dem neue Daten erzeugt werden.
	 */
	private long delay = 60000L;

	/**
	 * Die Liste aller DE im System.
	 */
	private SystemObject[] alleDeLve;

	private DataDescription deFehlerBeschreibung;

	private ClientDavInterface con;

	private boolean aktiv;

	private final Map<SystemObject, ResultData> latestResultsDeFehler = new LinkedHashMap<>();

	private final Map<SystemObject, ResultData> latestResultsKzd = new LinkedHashMap<>();

	/**
	 * Standardkonstruktor.
	 */
	private KzdTestSender() {
		super();
	}

	/**
	 * Prüft die Verfügbarkeit von Kurzzeitdaten aller Fahrstreifen<br>
	 * und sendet für die Stationen, die länger als X Minuten keine KZD empfangen
	 * haben eine Warnung an die BMV.
	 */
	@Override
	public void run() {
		final long now = System.currentTimeMillis();
		if (startZeit == 0) {
			startZeit = (now / delay) * delay;
		}

		logger.fine(() -> "Sende Daten für " + fahrStreifenListe.size() + " Fahrstreifen");

		int loop = 0;
		while ((loop < 20) && (startZeit < now)) {
			final List<ResultData> resultList = new ArrayList<>();

			try {
				for (final SystemObject fahrStreifen : fahrStreifenListe) {
					Data data;
					if (useFileData) {
						data = getTestDatenFromFile();

					} else {
						data = getVerkehrsDaten();
					}
					final ResultData result = new ResultData(fahrStreifen, descKurzzeitDaten, startZeit - delay, data);
					resultList.add(result);
					latestResultsKzd.put(fahrStreifen, result);
				}
				if (aktiv) {
					con.sendData(resultList.toArray(new ResultData[resultList.size()]));
					logger.info(() -> "Kurzzeitdaten mit Zeitstempel " + new Date(startZeit - delay) + " versendet für "
							+ resultList.size() + " Objekte");
				}
			} catch (final DataNotSubscribedException | SendSubscriptionNotConfirmed e) {
				logger.warning(e.getLocalizedMessage());
			} catch (final Exception e) {
				logger.severe(e.getLocalizedMessage());
			}
			startZeit += (1L * delay);
			loop++;
		}
	}

	@Override
	public void initialize(final ClientDavInterface dav) throws Exception {
		con = dav;

		// als Quelle fuer alle DeLVE anmelden, damit der globaleDeFehler
		// auf OK gesetzt wird
		alleDeLve = dav.getDataModel().getType("typ.deLve").getElements().toArray(new SystemObject[0]);
		deFehlerBeschreibung = new DataDescription(dav.getDataModel().getAttributeGroup("atg.tlsGloDeFehler"),
				dav.getDataModel().getAspect("asp.tlsAntwort"));

		final AttributeGroup atg = dav.getDataModel().getAttributeGroup("atg.verkehrsDatenKurzZeitIntervall");
		final Aspect asp = dav.getDataModel().getAspect("asp.externeErfassung");
		descKurzzeitDaten = new DataDescription(atg, asp);

		fahrStreifenListe.addAll(dav.getDataModel().getType("typ.fahrStreifen").getElements());

		synchronized (fahrStreifenListe) {
			try {
				for (final SystemObject obj : alleDeLve) {
					final ResultData initialData = latestResultsDeFehler.get(obj);
					if (initialData != null) {
						con.subscribeSource(this, initialData);
					} else {
						con.subscribeSender(this, obj, deFehlerBeschreibung, SenderRole.source());
					}
				}
			} catch (final OneSubscriptionPerSendData e) {
				logger.warning(e.getLocalizedMessage());
			}
			try {
				for (final SystemObject obj : fahrStreifenListe) {
					final ResultData initialData = latestResultsKzd.get(obj);
					if (initialData != null) {
						con.subscribeSource(this, initialData);
					} else {
						con.subscribeSender(this, obj, descKurzzeitDaten, SenderRole.source());
					}
				}
			} catch (final Exception e) {
				logger.severe(e.getLocalizedMessage());
			}
			aktiv = true;

			// TODO Zeitpunkt mit dem Datenintervall synchronisieren.
			checkTimer.scheduleAtFixedRate(this, 1000L, delay);
		}
	}

	@Override
	public void parseArguments(final ArgumentList argumentList) throws Exception {
		final String startStr = argumentList.fetchArgument("-startDatum=").asString();
		if ((startStr != null) && (startStr.length() > 0)) {
			startZeit = DateFormat.getDateInstance().parse(startStr).getTime();
		}

		if (argumentList.hasArgument("-delay")) {
			delay = argumentList.fetchArgument("-delay=60000").longValue();
		}

		if (argumentList.hasArgument("-file")) {
			final String dataFile = argumentList.fetchArgument("-file=").asString();
			if ((dataFile != null) && (dataFile.length() > 0)) {
				initWithFile(dataFile);
			}
		}

		qKfzMax = argumentList.fetchArgument("-qKfzMax=80.0").doubleValue();
		vPkwMax = argumentList.fetchArgument("-vPkwMax=140.0").doubleValue();
		vLkwMax = argumentList.fetchArgument("-vLkwMax=100.0").doubleValue();

	}

	private void initWithFile(final String file) throws Exception {

		final Properties properties = new Properties();
		try (FileInputStream inputStream = new FileInputStream(file)) {
			properties.load(inputStream);
		}

		if (properties.containsKey("delay")) {
			delay = Long.parseLong(properties.get("delay").toString());
		}

		final ArrayList<Integer> values = new ArrayList<>();
		for (final String currval : properties.get("qKfz").toString().split(",")) {
			values.add(Integer.parseInt(currval));
		}

		final Integer[] qKfz = values.toArray(new Integer[values.size()]);
		values.clear();

		for (final String currval : properties.get("qLkw").toString().split(",")) {
			values.add(Integer.parseInt(currval));
		}
		final Integer[] qLkw = values.toArray(new Integer[values.size()]);
		values.clear();

		for (final String currval : properties.get("vPkw").toString().split(",")) {
			values.add(Integer.parseInt(currval));
		}
		final Integer[] vPkw = values.toArray(new Integer[values.size()]);
		values.clear();

		for (final String currval : properties.get("vLkw").toString().split(",")) {
			values.add(Integer.parseInt(currval));
		}
		final Integer[] vLkw = values.toArray(new Integer[values.size()]);
		values.clear();

		for (final Integer element : qKfz) {
			for (final Integer element2 : qLkw) {
				for (final Integer element3 : vPkw) {
					for (final Integer element4 : vLkw) {
						final Integer[] array = new Integer[4];
						array[0] = element;
						array[1] = element2;
						array[2] = element3;
						array[3] = element4;
						fileData.add(array);

					}
				}
			}
		}

		idxData = fileData.size() - 1;
		useFileData = true;

	}

	/**
	 * Die Hauptfunktion des Observers.
	 *
	 * @param args Programm-Argumente
	 */
	public static void main(final String[] args) {
		StandardApplicationRunner.run(new KzdTestSender(), args);
	}

	@Override
	public void dataRequest(final SystemObject object, final DataDescription dataDescription, final byte state) {
		if ((state == ClientSenderInterface.START_SENDING) && dataDescription.equals(deFehlerBeschreibung)) {

			versendeDeFehlerZustandOk();
		}

	}

	@Override
	public boolean isRequestSupported(final SystemObject object, final DataDescription dataDescription) {
		boolean result = false;
		if (dataDescription.equals(deFehlerBeschreibung) || dataDescription.equals(descKurzzeitDaten)) {
			result = true;
		} else {
			logger.warning("Unerwarteter Request: Attributgruppe " + dataDescription.getAttributeGroup().getName()
					+ " Aspekt " + dataDescription.getAspect().getName() + " Objekt " + object.getPid());
		}
		return result;
	}

	/**
	 * @return die Daten
	 * @throws Exception
	 */
	private Data getVerkehrsDaten() throws Exception {

		int qKfz; // qKfz-Wert aus Telegramm
		int qLkw; // qLkw-Wert aus Telegramm
		int vPkwAe; // vPkwÄ-Wert aus Telegramm
		int vLkwAe; // vLkwÄ-Wert aus Telegramm
		int belegung; // b-Wert (Belegung)

		qKfz = (int) (dataSource.nextDouble() * qKfzMax); // 0
		// <=
		// qKfz
		// <
		// 80
		// Fz/Intervall
		final double anteilLkw = dataSource.nextDouble() * .4; // Zwischen 0%
		// und 40%
		// Lkw
		qLkw = (int) (qKfz * anteilLkw);
		if (qKfz <= 0) {
			vPkwAe = -1; // Geschw. ist nicht ermittelbar, wenn keine gefahren
			// sind!
		} else {
			vPkwAe = (int) (dataSource.nextDouble() * vPkwMax);
		}
		if (qLkw <= 0) {
			vLkwAe = -1; // Geschw. ist nicht ermittelbar, wenn keine gefahren
			// sind!
		} else {
			vLkwAe = (int) (dataSource.nextDouble() * vLkwMax);
		}
		belegung = (int) (dataSource.nextDouble() * 40.);

		return buildData(qKfz, qLkw, vPkwAe, vLkwAe, belegung);
	}

	/**
	 * @return die Daten
	 * @throws Exception
	 */
	private Data getTestDatenFromFile() throws Exception {

		if (idxData >= fileData.size()) {
			idxData = 0;
		}
		int qKfz = fileData.get(idxData)[0];
		int qLkw = fileData.get(idxData)[1];
		int vPkw = fileData.get(idxData)[2];
		int vLkw = fileData.get(idxData)[3];

		idxData++;

		return buildData(qKfz, qLkw, vPkw, vLkw, 0);
	}

	/**
	 * Methode um einen Datensatz zu erzeugen.
	 *
	 * @param _qKfz_ die Anzahl der Kfz
	 * @param _qLkw_ die Anzahl der Lkw
	 * @param _vPkw_ die Geschwindigkeit der Pkw
	 * @param _vLkw_ die Geschwindigkeit der Lkw
	 * @param _b_    die Belegung
	 * @return einen neuen Verkehrsdatensatz
	 */
	private Data buildData(final int _qKfz_, final int _qLkw_, final int _vPkw_, final int _vLkw_, final int _b_) {

		final Data data = con.createData(descKurzzeitDaten.getAttributeGroup());

		// Intervalllänge aus IntervalldatenHeader
		// ---------------------------------------------------------------------
		data.getTimeValue("T").setMillis(delay);

		// Art der Mittelwertbildung am DE aus Betriebsparametern
		// ------------------------------------------------------
		data.getUnscaledValue("ArtMittelwertbildung").setText("gleitende Mittelwertbildung");

		final String[] valStrings = { "qKfz", "vKfz", "qLkw", "vLkw", "qPkw", //$NON-NLS-4$ //$NON-NLS-5$
				"vPkw", "b", "tNetto", "sKfz", "vgKfz" }; //$NON-NLS-4$ //$NON-NLS-5$

		for (final String valString : valStrings) {
			data.getItem(valString).getUnscaledValue("Wert").setText("nicht ermittelbar");
			data.getItem(valString).getItem("Status").getItem("Erfassung").getUnscaledValue("NichtErfasst")
					.setText("Nein");
			data.getItem(valString).getItem("Status").getItem("PlFormal").getUnscaledValue("WertMax").setText("Nein");
			data.getItem(valString).getItem("Status").getItem("PlFormal").getUnscaledValue("WertMin").setText("Nein");
			data.getItem(valString).getItem("Status").getItem("PlLogisch").getUnscaledValue("WertMaxLogisch")
					.setText("Nein");
			data.getItem(valString).getItem("Status").getItem("PlLogisch").getUnscaledValue("WertMinLogisch")
					.setText("Nein");
			data.getItem(valString).getItem("Status").getItem("MessWertErsetzung").getUnscaledValue("Implausibel")
					.setText("Nein");
			data.getItem(valString).getItem("Status").getItem("MessWertErsetzung").getUnscaledValue("Interpoliert")
					.setText("Nein");
			data.getItem(valString).getItem("Güte").getUnscaledValue("Index").set(-1);
			data.getItem(valString).getItem("Güte").getUnscaledValue("Verfahren").set(0);
		}

		data.getItem("qKfz").getUnscaledValue("Wert").set(_qKfz_);
		data.getItem("qKfz").getItem("Güte").getUnscaledValue("Index").set(10);

		data.getItem("qLkw").getUnscaledValue("Wert").set(_qLkw_);
		data.getItem("qLkw").getItem("Güte").getUnscaledValue("Index").set(10);

		data.getItem("vPkw").getUnscaledValue("Wert").set(_vPkw_);
		data.getItem("vPkw").getItem("Güte").getUnscaledValue("Index").set(10);

		data.getItem("vLkw").getUnscaledValue("Wert").set(_vLkw_);
		data.getItem("vLkw").getItem("Güte").getUnscaledValue("Index").set(10);

		data.getItem("b").getUnscaledValue("Wert").set(_b_);
		data.getItem("b").getItem("Güte").getUnscaledValue("Index").set(10);

		int _qPkw_;
		int _vKfz_;
		// Nicht erfasste Werte qPkw und vKfz berechnen
		_qPkw_ = (_qKfz_ - _qLkw_) >= 0 ? (_qKfz_ - _qLkw_) : -1;
		data.getItem("qPkw").getUnscaledValue("Wert").set(_qPkw_);
		data.getItem("qPkw").getItem("Güte").getUnscaledValue("Index").set(10);
		data.getItem("qPkw").getItem("Status").getItem("Erfassung").getUnscaledValue("NichtErfasst").setText("Ja");

		_vKfz_ = (_qLkw_ + _qPkw_) > 0 ? ((_qLkw_ * _vLkw_) + (_qPkw_ * _vPkw_)) / (_qLkw_ + _qPkw_) : -1;
		data.getItem("vKfz").getUnscaledValue("Wert").set(_vKfz_);
		data.getItem("vKfz").getItem("Güte").getUnscaledValue("Index").set(10);
		data.getItem("vKfz").getItem("Status").getItem("Erfassung").getUnscaledValue("NichtErfasst").setText("Ja");

		return data;

	}

	/**
	 * Versendet einen DaV-Datensatz an ein oder mehrere Systemobjekte
	 * (normalerweise eine DE-Liste, ansonsten vermutlich DaV-Exception).
	 *
	 * @param deList die Liste der Systemobjekte
	 * @param data   der Datensatz
	 */
	private void versendeDeFehlerZuMehrerenDE(final SystemObject[] deList, final Data data) {
		final ResultData[] results = new ResultData[deList.length];
		int loop = 0;
		for (final SystemObject obj : deList) {
			final long timeStamp = System.currentTimeMillis();
			final ResultData result = new ResultData(obj, deFehlerBeschreibung, timeStamp, data);
			latestResultsDeFehler.put(obj, result);
			results[loop] = result;
			++loop;
		}
		if (aktiv) {
			try {
				con.sendData(results);
			} catch (final DataNotSubscribedException | SendSubscriptionNotConfirmed e) {
				logger.warning("Fehler beim versenden des Globalen-DE-Fehlers: " + e.getLocalizedMessage());
			}
		}
	}

	/**
	 * Versendet an alle WZG-DE den globalen DE-Fehlerzustand OK.
	 */
	private void versendeDeFehlerZustandOk() {
		final Data data = con.createData(con.getDataModel().getAttributeGroup("atg.tlsGloDeFehler"));

		data.getUnscaledValue("DEFehlerStatus").set(0);
		data.getUnscaledValue("DEProjektierungsStatus").set(0);
		data.getUnscaledValue("DEKanalStatus").set(0);
		data.getUnscaledValue("HerstellerDefinierterCode").set(0);
		data.getUnscaledValue("Hersteller").set(0);

		versendeDeFehlerZuMehrerenDE(alleDeLve, data);
	}

}
